/*
 * $Id: User.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.ImageIcon;

import csbase.exception.administration.AdministrationDeleteException;
import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;
import tecgraf.javautils.core.lng.LNG;

import javax.swing.*;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Representa um usurio do sistema. Possui informaes como o nome, login,
 * senha, email, perfil e UserGroup. Atua como uma classe que pode ser observada
 * localmente. Mantm uma cache dos usurios instanciados localmente.
 */
public class User implements Serializable {
  /**
   * Identificador do Login na Hashtable de atributos do UserInfo
   */
  public static final String LOGIN = "login";

  /**
   * Identificador do Login do Super-Usurio na Hashtable de atributos do
   * UserInfo
   */
  public static final String SUPER_USER_LOGIN = "superuserlogin";

  /**
   * Identificador do nome na Hashtable de atributos do UserInfo
   */
  public static final String NAME = "name";

  /**
   * Identificador dos emails na Hashtable de atributos do UserInfo
   */
  public static final String EMAILS = "emails";

  /** Identificador dos perfis na Hashtable de atributos do UserInfo */
  public static final String ROLE_IDS = "roleIds";

  /**
   * Identificador das permisses na Hashtable de atributos do UserInfo
   */
  public static final String PERMISSION_IDS = "permissionIds";

  /**
   * Identificador da senha na Hashtable de atributos do UserInfo
   */
  public static final String PASSWORD = "password";

  /**
   * Identificador da senha na Hashtable de atributos do UserInfo
   */
  public static final String PASSWORD_DIGEST = "passwordDigest";

  /**
   * Identificador da url para foto na Hashtable de atributos do UserInfo
   */
  public static final String PHOTO_LINK = "photoLink";


  /**
   * Identificador da ltima atualizao na Hashtable de atributos do UserInfo
   */
  public static final String LAST_UPDATE = "lastUpdate";

  /**
   * Identificador da configurao de login local forado nos atributos da
   * UserInfo.
   */
  public static final String FORCE_LOCAL_LOGIN = "forceLocalLogin";

  /**
   * Identificador do grupo de usurio
   */
  public static String USER_GROUP = "userGroup";
  
  /** Data que o usurio foi criado na Hashtable de atributos do UserInfo. */
  public static final String CREATION_DATE = "creationDate";

  /** Cache local de usurios. A chave  o identificador do usurio. */
  private static final Hashtable<String, User> _USERS =
    new Hashtable<String, User>();
  /** Utilizado no mtodo de carregar as informaes do cache de usurios. */
  private static final Lock _LOAD_USERS_LOCK = new ReentrantLock();

  /** Login do administrador. */
  private static final String ADMIN_LOGIN = "admin";

  /**
   * O usurio que est logado no cliente. Essa informao somente  usada no
   * cliente, j que no servidor podem existir vrios usurios logados.
   */
  private static User loggedUser = null;

  /**
   * Indica se a cache de usurios est completo
   */
  private static boolean hasAllUsers = false;

  /**
   * Identificao do usurio
   */
  private final Object id;

  /**
   * Informaes do usurio
   */
  private final UserInfo info;

  /**
   * A chave que identifica a sesso de login desse usurio. S faz sentido no
   * cliente, pois no servidor podemos ter um mesmo usurio logado vrias vezes.
   */
  private transient Object key;

  /** Objeto que permite a observao da classe User */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Cria um usurio. S ser executado no servidor. Objetos so serializados
   * para o cliente.
   * 
   * @param id o identificador do usurio
   * @param info os dados do usurio
   */
  public User(Object id, UserInfo info) {
    this.id = id;
    this.info = info;
  }

  /**
   * Registra o login de um usurio do sistema. Esse mtodo  chamado pelo
   * {@link ClientRemoteLocator} aps um login ser feito com sucesso, ou quando
   * o usurio corrente mudar.
   * 
   * @param key chave da sesso
   * @param user usurio
   */
  public synchronized static void registerLogin(Object key, User user) {
    loggedUser = user;
    loggedUser.setKey(key);
  }

  /**
   * Registra o logout do usurio logado. Esse mtodo  chamado pelo
   * {@link ClientRemoteLocator} aps o logout ser feito com sucesso.
   */
  public synchronized static void registerLogout() {
    loggedUser = null;
  }

  /**
   * Recupera o usurio logado no sistema. Esse mtodo somente deve ser usado no
   * cliente.
   * 
   * @return o usurio.
   */
  public synchronized static User getLoggedUser() {
    return loggedUser;
  }

  /**
   * Muda a chave da sesso de login.
   * 
   * @param key A chave que identifica a sesso de login desse usurio.
   */
  public void setKey(Object key) {
    this.key = key;
  }

  /**
   * Recupera a chave que identifica a sesso de login desse usurio.
   * 
   * @return A chave de login.
   */
  public Object getKey() {
    return this.key;
  }

  /**
   * Verifica se um usurio  igual a outro. Dois usurios so considerados o
   * mesmo se eles possuirem o mesmo identificador.
   * 
   * @param obj o outro objeto com o qual esse usurio est sendo comparado
   * 
   * @return true ou false, se o usurio for ou no igual ao outro
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof User)) {
      return false;
    }
    User user = (User) obj;
    return user.getId().equals(id);
  }

  /**
   * Calcula o cdigo hash do objeto.
   * 
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return id.hashCode();
  }

  /**
   * Solicita ao servio de administrao um usurio que possui uma determinada
   * identificao.
   * 
   * @param id a identificao do usurio
   * 
   * @return o usurio que possui a identificao requisitada ou null caso esse
   *         usurio no exista
   * 
   * @throws RemoteException falha de rmi
   */
  public static User getUser(Object id) throws RemoteException {
    if (id == null) {
      return null;
    }

    loadLocalUsersCache();

    String key = createKeyForLocalUsersCache(id);
    User user = _USERS.get(key);
    if (user == null) {
      user = ClientRemoteLocator.administrationService.getUser(id);
      if (user != null) {
        _USERS.put(key, user);
      }
    }
    return user;
  }

  /**
   * Solicita o usurio que possui um determinado login. O login  o prprio
   * identificador do usurio.
   * 
   * @param login o login do usurio procurado
   * 
   * @return o usurio que possui o login procurado ou null caso esse usurio
   *         no exista.
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static User getUserByLogin(String login) throws Exception {
    /* Dado que o login  o prpio identificador, procura o usurio */
    return getUser(login);
  }

  /**
   * Solicita a lista de informaes sumarizadas de todos os usurios do
   * sistema.
   * 
   * @return uma lista com informaes resumidas de todos os usurios
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   * 
   * @see #getAllUsers()
   * @see #getNumRegisteredUsers()
   */
  public static List<UserOutline> getAllOutlines() throws Exception {
    if (hasAllUsers) {
      return usersToOutlines();
    }
    return ClientRemoteLocator.administrationService.getAllUserOutlines();
  }

  /**
   * Obtm um Vector com os outlines dos identificadores informados.
   * 
   * @param userIds ids dos usurios.
   * 
   * @return Vector com os outlines dos identificadores informados.
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static Vector<UserOutline> getOutlines(Object[] userIds)
    throws Exception {
    Vector<UserOutline> outlines = new Vector<UserOutline>();
    if (userIds == null) {
      return outlines;
    }
    for (int i = 0; i < userIds.length; i++) {
      User user = getUser(userIds[i]);
      if (user != null) {
        outlines.add(user.getOutline());
      }
    }
    return outlines;
  }

  /**
   * Solicita a lista de todos os usurios cadastrados no sistema. Os usurios
   * so colocadas na cache. Esse mtodo deve ser usado com cautela,
   * considerando que o nmero de usurios pode ser muito grande. Atualmente,
   * somente  usado no servidor.
   * 
   * @return um vetor com todas os usurios
   * 
   * @throws RemoteException em caso de falha (eventualmente por RMI) no acesso
   *         aos dados.
   * 
   * @see #getNumRegisteredUsers()
   * @see #getAllOutlines()
   */
  public static List<User> getAllUsers() throws RemoteException {
    loadLocalUsersCache();
    return usersToVector();
  }

  /**
   * Obtm o nmero de usurios cadastrados no sistema (inclui o admin).  mais
   * eficiente usar este mtodo do que obter esta informao indiretamente via
   * {@link #getAllUsers()}.
   * 
   * @return nmero de usurios cadastrados no sistema
   * @throws RemoteException em caso de falha (eventualmente por RMI) no acesso
   *         aos dados.
   *
   * @see #getAllUsers()
   * @see #getAllOutlines()
   */
  public static int getNumRegisteredUsers() throws RemoteException {
    loadLocalUsersCache();
    return _USERS.size();
  }

  /**
   * Carrega lista de usurios locais da cache.
   * 
   * @throws RemoteException em caso de falha (eventualmente por RMI) no acesso
   *         aos dados.
   */
  private static void loadLocalUsersCache() throws RemoteException {
    if (hasAllUsers) {
      return;
    }

    _LOAD_USERS_LOCK.lock();
    try {
      // Pega todos os usurios do sistema
      List<User> allUsers = ClientRemoteLocator.administrationService
        .getAllUsers();
      if (allUsers == null) {
        return;
      }
      // Limpa o cache atual antes de recri-lo
      _USERS.clear();
      // Refaz o cache
      for (User user : allUsers) {
        String key = createKeyForLocalUsersCache(user.getId());
        _USERS.put(key, user);
      }

      hasAllUsers = true;
    }
    finally {
      _LOAD_USERS_LOCK.unlock();
    }
  }

  /**
   * Solicita o nome de um usurio a partir do id
   * 
   * @param id do usurio
   * 
   * @return nome do usurio
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static String getName(Object id) throws Exception {
    User user = getUser(id);
    if (null != user) {
      return user.getName();
    }
    return null;
  }

  /**
   * Adio de perfil ao usurio.
   * 
   * @param role o perfil
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public void addRole(Role role) throws Exception {
    if (this.isAdmin()) {
      return;
    }
    Object[] roles = this.getRoleIds();
    ArrayList<Object> newRoles = new ArrayList<Object>();
    for (int i = 0; i < roles.length; i++) {
      if (roles[i].equals(role.getId())) {
        return;
      }
      newRoles.add(roles[i]);
    }
    newRoles.add(role.getId());
    this.setRoleIds(newRoles.toArray());
  }

  /**
   * Remoo de perfil do usurio
   * 
   * @param role perfil.
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public void removeRole(Role role) throws Exception {
    Object[] roles = this.getRoleIds();
    ArrayList<Object> newRoles = new ArrayList<Object>();
    for (int i = 0; i < roles.length; i++) {
      if (!roles[i].equals(role.getId())) {
        newRoles.add(roles[i]);
      }
    }
    this.setRoleIds(newRoles.toArray());
  }

  /**
   * Adio de permisso ao usurio.
   * 
   * @param permission permisso.
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public void addPermission(Permission permission) throws Exception {
    if (this.isAdmin()) {
      return;
    }
    Object[] permissions = this.getPermissionIds();
    ArrayList<Object> newPermissions = new ArrayList<Object>();
    for (int i = 0; i < permissions.length; i++) {
      if (permissions[i].equals(permission.getId())) {
        return;
      }
      newPermissions.add(permissions[i]);
    }
    newPermissions.add(permission.getId());
    this.setPermissionIds(newPermissions.toArray());
  }

  /**
   * Remoo de permisso do usurio.
   * 
   * @param permission permisso.
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public void removePermission(Permission permission) throws Exception {
    Object[] permissions = this.getPermissionIds();
    ArrayList<Object> newPermissions = new ArrayList<Object>();
    for (int i = 0; i < permissions.length; i++) {
      if (!permissions[i].equals(permission.getId())) {
        newPermissions.add(permissions[i]);
      }
    }
    this.setPermissionIds(newPermissions.toArray());
  }

  /**
   * Solicita os usurios administradores do sistema. Atualmente, somente 
   * usado no servidor.
   * 
   * @return lista de usurios administradores
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static User[] getAdmins() throws Exception {
    Vector<User> users = new Vector<User>();
    users.add(getUserByLogin((String) getAdminId()));
    for (User user : getAllUsers()) {
      if (user.getPermission(GlobalAdminPermission.class) != null) {
        users.add(user);
      }
    }

    return users.toArray(new User[0]);
  }

  /**
   * Solicita os identificadores dos usurios administradores do sistema.
   * Atualmente, somente  usado no servidor.
   * 
   * @return lista de identificadores dos usurios administradores
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static List<Object> getAdminIds() throws Exception {
    List<Object> users = new LinkedList<Object>();
    users.add(getAdminId());
    for (User user : getAllUsers()) {
      if (user.getPermission(GlobalAdminPermission.class) != null) {
        users.add(user.getId());
      }
    }

    return users;
  }

  /**
   * Obtm o identificador nico do administrador.<br>
   * O id de um usurio  o mesmo que o login daquele usurio.
   * 
   * @return o id do usurio.
   */
  public static Object getAdminId() {
    return ADMIN_LOGIN;
  }

  /**
   * Solicita a criao de um novo usurio no sistema. No se pode criar um
   * usurio com o mesmo login de outro j existente. O usurio criado 
   * colocado na cache.
   * 
   * @param info os dados do novo usurio
   * @return O usurio criado.
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static User createUser(UserInfo info) throws Exception {
    if ((info.getLogin() == null) || info.getLogin().equals("")) {
      throw new Exception(LNG.get("csbase.logic.EmptyLogin"));
    }
    if (getUserByLogin(info.getLogin()) != null) {
      return null;
    }
    
    User user = ClientRemoteLocator.administrationService.createUser(info);
    if (user != null) {
      String key = createKeyForLocalUsersCache(user.getId());
      _USERS.put(key, user);
    }
    return user;
  }

  /**
   * Mtodo no usado...
   * 
   * @param id id
   * @return imagem
   */
  public static ImageIcon getPhoto(final Object id) {
    try {
      return ClientRemoteLocator.administrationService.getPhoto(id);
    }
    catch (Exception e1) {
      try {
        return ClientRemoteLocator.administrationService.getPhoto(null);
      }
      catch (Exception e2) {
        return null;
      }
    }
  }

  /**
   * Modifica um usurio no sistema. No  permitido alterar o login.
   * 
   * @param id a identificao do usurio a ser modificado
   * @param info os dados do usurio a ser modificado
   * 
   * @return o novo usurio aps a modificao ou null caso a modificao no
   *         possa ter sido feita.
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static User modifyUser(Object id, UserInfo info) throws Exception {
    User user = getUser(id);
    if (user == null) {
      return null;
    }
    if (!user.getLogin().equals(info.getLogin())) {
      return null;
    }
    user = ClientRemoteLocator.administrationService.modifyUser(id, info);
    String key = createKeyForLocalUsersCache(id);
    _USERS.put(key, user);
    return user;
  }

  /**
   * Remove um usurio do sistema.
   * 
   * @param id a identificao do usurio a ser removido do sistema.
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   * @throws AdministrationDeleteException em caso de erro ao apagar.
   */
  public static void deleteUser(Object id) throws Exception,
    AdministrationDeleteException {
    ClientRemoteLocator.administrationService.deleteUser(id);
    String key = createKeyForLocalUsersCache(id);
    _USERS.remove(key);
  }

  /**
   * Adiciona um observador local da classe. Sempre que a classe  avisada pelo
   * servio de administrao de que alguma mudana ocorreu, os observadores
   * locais tambm so notificados.
   * 
   * @param o um observador local
   */
  public static void addObserver(Observer o) {
    observable.addObserver(o);
  }

  /**
   * Remove um observador local da classe.
   * 
   * @param o o observador a ser removido
   */
  public static void deleteObserver(Observer o) {
    observable.deleteObserver(o);
  }

  /**
   * Esse mtodo  chamado quando o servio de administrao sofre alguma
   * alterao relativa a usurios. Ele atualiza o cache de usurios e notifica
   * seus observadores locais.
   * 
   * @param event o evento que ocorreu no servio de administrao
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public static void update(AdministrationEvent event) throws Exception {
    User user = (User) event.item;
    Object id = user.getId();
    String key = createKeyForLocalUsersCache(id);
    switch (event.type) {
      case AdministrationEvent.CREATE:
      case AdministrationEvent.MODIFY:
        _USERS.put(key, user);
        synchronized (User.class) {
          if (loggedUser != null) {
            if (id.equals(loggedUser.getId())) {
              /*
               * FIXME isto  exatamente o que registerLogin(...) faz. Podemos
               * cham-lo aqui ao invs das 2 linhas abaixo?
               */
              user.setKey(loggedUser.getKey());
              loggedUser = user;
            }
          }
        }
        break;
      case AdministrationEvent.DELETE:
        _USERS.remove(key);
        break;
    }
    observable.notifyObservers(new AdministrationEvent(event.type, user
      .getOutline()));
  }

  /**
   * Obtm um vetor com os resumos dos usurios que esto na cache.
   * 
   * @return vetor contendo os resumos dos usurios presentes na cache
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  private static List<UserOutline> usersToOutlines() throws Exception {
    List<UserOutline> list = new ArrayList<UserOutline>();
    for (User user : _USERS.values()) {
      list.add(user.getOutline());
    }
    return list;
  }

  /**
   * Devolve um vetor com todas os usurios da cache.
   * 
   * @return um vetor com todas os usurios da cache
   */
  private static List<User> usersToVector() {
    return new ArrayList<User>(_USERS.values());
  }

  /**
   * Verifica se o usurio  o administrador.
   * 
   * @return true se for o administrador, false caso contrrio
   */
  public boolean isAdmin() {
    if (info.getLogin().equals(ADMIN_LOGIN)) {
      return true;
    }
    else if (getPermission(GlobalAdminPermission.class) != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Verifica se um usurio  o administrador.
   * 
   * @param uid - identificador do usurio
   * 
   * @return true se o usurio  o administrador
   */
  public static boolean isAdmin(Object uid) {
    User user = null;
    try {
      user = User.getUser(uid);
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }

    if (uid == null || user == null) {
      return false;
    }
    else if (ADMIN_LOGIN.equals(uid)) {
      return true;
    }
    else if (user.getPermission(GlobalAdminPermission.class) != null) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Obtm um atributo do usurio dado sua chave.
   * 
   * @param attrKey Chave do Atributo.
   * 
   * @return o atributo do usurio.
   */
  public Object getAttribute(String attrKey) {
    return info.getAttribute(attrKey);
  }

  /**
   * Obtm o identificador do usurio.
   * 
   * @return o identificador do usurio
   */
  public Object getId() {
    return id;
  }

  /**
   * Obtm um sumrio das informaes sobre esse usurio.
   * 
   * @return um sumrio do usurio
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public UserOutline getOutline() throws Exception {
    return new UserOutline(this);
  }

  /**
   * Obtm o login do usurio.
   * 
   * @return o login do usurio
   */
  public String getLogin() {
    return info.getLogin();
  }
  
  /**
   * Obtm o login do super-usurio. Retornar nulo se no tiver sido utilizada
   * a autenticao por delegao.
   * 
   * @return o login do super-usurio
   */
  public String getSuperUserLogin() {
    return (String) info.getAttribute(SUPER_USER_LOGIN);
  }

  /**
   * Obtm o digest da senha do usurio.
   * 
   * @return o digest da senha
   */
  public String getPasswordDigest() {
    return (String) info.getAttribute(PASSWORD_DIGEST);
  }

  /**
   * Obtm o nome do usurio.
   * 
   * @return o nome do usurio
   */
  public String getName() {
    return (String) info.getAttribute(NAME);
  }
  
  /**
   * @return A data de criao do usurio.
   */
  public Date getCreationDate() {
    return (Date) info.getAttribute(CREATION_DATE);
  }
  
  /**
   * Obtm os emails do usurio.
   * 
   * @return os emails do usurio.
   */
  public String[] getEmails() {
    return (String[]) info.getAttribute(EMAILS);
  }

  /**
   * Obtm os perfis do usurio.
   * 
   * @return um array com os perfis atribudos ao usurio.
   */
  public Object[] getRoleIds() {
    return (Object[]) info.getAttribute(ROLE_IDS);
  }

  /**
   * Obtm as permisses do usurio. Essa lista inclui apenas as permisses
   * individuais atribudas ao usurio.
   * 
   * @return um array com as permisses ao usurio.
   */
  public Object[] getPermissionIds() {
    return (Object[]) info.getAttribute(PERMISSION_IDS);
  }

  /**
   * Obtm todos os identificadores das permisses dos usurios. Essa lista
   * inclui no apenas as permisses individuais do usurio mas tambm as
   * relativas aos seus perfis. Os perfis so consultados no momento da criao
   * desta lista.
   * 
   * @return um Vector com todas as permisses ao usurio.
   * 
   * @throws Exception em caso de falha (eventualmente por RMI) no acesso aos
   *         dados.
   */
  public Vector<Object> getAllPermissionIds() throws Exception {
    Vector<Object> allPermissionIds = new Vector<Object>();
    Object[] permIds = getPermissionIds();
    if (permIds != null) {
      for (int i = 0; i < permIds.length; i++) {
        allPermissionIds.add(permIds[i]);
      }
    }
    Object[] roleIds = getRoleIds();
    if (roleIds != null) {
      for (int i = 0; i < roleIds.length; i++) {
        Role role = Role.getRole(roleIds[i]);
        Object[] permissionIds = role.getPermissionIds();
        for (int j = 0; j < permissionIds.length; j++) {
          allPermissionIds.add(permissionIds[j]);
        }
      }
    }
    return allPermissionIds;
  }

  /**
   * Verifica se o usurio possui determinado perfil. A classe do perfil 
   * recebida como parmetro e, caso o usurio possua algum perfil dessa classe,
   * uma (qualquer) delas  retornada. Caso o usurio no possua nenhum perfil
   * da classe especificada,  retornado null. Esse mtodo verifica todas os
   * perfis individuais que o usurio possui.
   * 
   * @param roleId .
   * 
   * @return Um perfil do tipo fornecido, caso o usurio possua algum, ou null
   *         caso contrrio.
   */
  public Role getRole(Object roleId) {
    // Pendncia: essa implementao  muito ineficiente para o servidor.
    // Devemos manter uma estrutura de consulta rpida e usar nesse mtodo.
    try {
      Object[] ids = getRoleIds();
      for (int i = 0; i < ids.length; i++) {
        Role r = Role.getRole(ids[i]);
        if (ids[i].equals(roleId)) {
          return r;
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Verifica se o usurio possui determinada permisso. A classe da permisso 
   * recebida como parmetro e, caso o usurio possua alguma permisso dessa
   * classe, uma (qualquer) delas  retornada. Caso o usurio no possua nenhuma
   * permisso da classe especificada,  retornado null. Esse mtodo verifica
   * todas as permisses individuais que o usurio possui, bem como todas as
   * permisses de todos os seus perfis.
   * 
   * @param <T> .
   * @param permissionClass classe que representa a permisso.
   * 
   * @return Uma permisso do tipo fornecido, caso o usurio possua alguma, ou
   *         null caso contrrio.
   */
  public <T extends Permission> T getPermission(Class<T> permissionClass) {
    // Pendncia: essa implementao  muito ineficiente para o servidor.
    // Devemos manter uma estrutura de consulta rpida e usar nesse mtodo.
    try {
      Vector<Object> ids = getAllPermissionIds();
      for (int i = 0; i < ids.size(); i++) {
        Permission p = Permission.getPermission(ids.get(i));
        if (p.getClass().equals(permissionClass)) {
          return permissionClass.cast(p);
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Verifica se o usurio possui ao menos uma permisso que estenda uma
   * determinada permisso. A classe da permisso  recebida como parmetro e,
   * caso o usurio possua alguma permisso que estenda esta classe, uma
   * (qualquer) delas  retornada. Caso o usurio no possua nenhuma permisso
   * da classe especificada,  retornado null. Esse mtodo verifica todas as
   * permisses individuais que o usurio possui, bem como todas as permisses
   * de todos os seus perfis.
   * 
   * @param <T> .
   * @param permissionClass classe que representa a permisso.
   * 
   * @return Uma permisso que estenda a permisso do tipo fornecido, caso o
   *         usurio possua alguma, ou null caso contrrio.
   */
  public <T extends Permission> T getPermissionAssignableTo(
    Class<T> permissionClass) {
    // Pendncia: essa implementao  muito ineficiente para o servidor.
    // Devemos manter uma estrutura de consulta rpida e usar nesse mtodo.
    try {
      Vector<Object> ids = getAllPermissionIds();
      for (int i = 0; i < ids.size(); i++) {
        Permission p = Permission.getPermission(ids.get(i));
        if (permissionClass.isAssignableFrom(p.getClass())) {
          return permissionClass.cast(p);
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Verifica se o usurio possui uma determinada permissao. O Id da permisso 
   * recebido como parmetro e, caso o usurio possua essa permisso, ela 
   * retornada. Caso contrrio,  retornado null. Esse mtodo verifica todas as
   * permisses individuais que o usurio possui.
   * 
   * @param permissionId .
   * 
   * @return A permisso, caso o usurio possua, ou null caso contrrio.
   */
  public Permission getPermission(Object permissionId) {
    try {
      Object[] ids = getPermissionIds();
      for (int i = 0; i < ids.length; i++) {
        if (ids[i].equals(permissionId)) {
          return Permission.getPermission(ids[i]);
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Verifica se o usurio possui determinada permisso. A classe da permisso e
   * um atributo so recebidos como parmetros e, caso o usurio possua alguma
   * permisso dessa classe que contenha o atributo fornecido, uma (qualquer)
   * delas  retornada. Caso o usurio no possua permisso da classe
   * especificada contendo o atributo especificado,  retornado null. Esse
   * mtodo verifica todas as permisses individuais que o usurio possui, bem
   * como todas as permisses de todos os seus perfis.
   * 
   * @param permissionClass .
   * @param attribute .
   * 
   * @return Uma permisso do tipo fornecido que contm o atributo fornecido,
   *         caso o usurio possua alguma, ou null caso contrrio.
   */
  public AttributesPermission getAttributesPermission(Class<?> permissionClass,
    String attribute) {
    // Pendncia: essa implementao  muito ineficiente para o servidor.
    // Devemos manter uma estrutura de consulta rpida e usar nesse mtodo.
    try {
      Vector<Object> ids = getAllPermissionIds();
      for (int i = 0; i < ids.size(); i++) {
        Permission p = Permission.getPermission(ids.get(i));
        if (p instanceof AttributesPermission) {
          if (p.getClass().equals(permissionClass)) {
            AttributesPermission ap = (AttributesPermission) p;
            if (ap.hasAttribute(attribute)) {
              return ap;
            }
          }
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Verifica se o usurio possui determinada permisso. A classe da permisso e
   * um atributo so recebidos como parmetros e, caso o usurio possua alguma
   * permisso dessa classe que contenha algum atributo que case com o texto
   * fornecido, uma (qualquer) delas  retornada. Caso o usurio no possua
   * permisso da classe especificada que case com o texto especificado, 
   * retornado null. Esse mtodo verifica todas as permisses individuais que o
   * usurio possui, bem como todas as permisses de todos os seus perfis.
   * 
   * @param permissionClass .
   * @param attribute .
   * 
   * @return Uma permisso do tipo fornecido que contm o atributo fornecido,
   *         caso o usurio possua alguma, ou null caso contrrio.
   * @throws Exception erro ocorrido durante a busca pela permisso (ex: de rmi)
   */
  public AttributesPermission getMatchAttributesPermission(
    Class<?> permissionClass, String attribute) throws Exception {
    // Pendncia: essa implementao  muito ineficiente para o servidor.
    // Devemos manter uma estrutura de consulta rpida e usar nesse mtodo.
    Vector<Object> ids = getAllPermissionIds();
    for (int i = 0; i < ids.size(); i++) {
      Permission p = Permission.getPermission(ids.get(i));
      if (p instanceof AttributesPermission) {
        if (p.getClass().equals(permissionClass)) {
          AttributesPermission ap = (AttributesPermission) p;
          if (ap.getMatchAttribute(attribute) != null) {
            return ap;
          }
        }
      }
    }
    return null;
  }

  /**
   * Verifica se o usurio possui uma permisso de uma determinada classe e que
   * contenha todos os atributos especificados como parmetros.
   * 
   * Caso um dos atributos no case com o texto fornecido, ento a busca
   * continua para a prxima permisso do mesmo tipo de classe. Quando for
   * encontrado uma permisso para o usurio que seja do tipo especificado e que
   * possua todos os atributos especificados, ento essa permisso  retornada.
   * Caso contrrio, o mtodo retorna null.
   * 
   * Esse mtodo verifica todas as permisses individuais que o usurio possui,
   * assim como todas as permisses de todos os seus perfis.
   * 
   * @param permissionClass classe da permisso
   * @param attributes atributos procurados para validao (TODOS precisam fazer
   *        parte da permisso procurada)
   * 
   * @return retorna a permisso caso seja encontrada uma permisso do tipo
   *         especificado e que possua todos os atributos especificados, caso
   *         contrrio, o mtodo retorna null
   * @throws Exception erro ocorrido durante a busca pela permisso (ex: de rmi)
   */
  public AttributesPermission getMatchAttributesPermission(
    Class<?> permissionClass, String... attributes) throws Exception {
    if (attributes == null) {
      return null;
    }
    Vector<Object> ids = getAllPermissionIds();
    for (Object permissionId : ids) {
      Permission p = Permission.getPermission(permissionId);
      AttributesPermission ap = checkClassPermission(p, permissionClass);
      if (ap != null) {
        if (checkPermissionAttributes(ap, attributes)) {
          return ap;
        }
      }
    }
    return null;
  }

  /**
   * Verifica se a permisso especificada  de uma determinada classe. Se for,
   * retorna a permisso convertida para o tipo de permisso de atributo.
   * 
   * @param p permisso
   * @param permissionClass classe da permisso
   * 
   * @return se for vlida, retorna a permisso convertida para o tipo de
   *         permisso de atributo, caso contrrio, retorna null
   */
  private AttributesPermission checkClassPermission(Permission p,
    Class<?> permissionClass) {
    if (p instanceof AttributesPermission) {
      if (p.getClass().equals(permissionClass)) {
        return (AttributesPermission) p;
      }
    }
    return null;
  }

  /**
   * Verifica se uma determinada permisso possui os atributos especificados.
   * 
   * @param ap permisso de atributo
   * @param attributes atributos procurados na permisso
   * 
   * @return retorna false se algum dos atributos no pertencer a permisso
   *         especificada, caso contrrio, retorna true
   */
  private boolean checkPermissionAttributes(AttributesPermission ap,
    String... attributes) {
    if (attributes == null) {
      return false;
    }
    for (String attribute : attributes) {
      if (ap.getMatchAttribute(attribute) == null) {
        return false;
      }
    }
    return true;
  }

  /**
   * Verifica se o usurio possui uma determinada permisso com um conjunto de
   * valores definidos em <code>attributes</code>.
   * 
   * @param permissionClass A classe da permisso que ser verificada.
   * @param attributes Os valores que a permisso deve possuir.
   * 
   * @return O boolean indicando se o usurio possui a permisso.
   */
  public boolean hasPermission(Class<?> permissionClass, Map<?, ?> attributes) {
    try {
      Vector<Object> ids = getAllPermissionIds();
      for (int i = 0; i < ids.size(); i++) {
        Permission p = Permission.getPermission(ids.get(i));
        if (p instanceof ChoicePermission) {
          if (p.getClass().equals(permissionClass)) {
            ChoicePermission cp = (ChoicePermission) p;
            if (cp.hasPermission(attributes)) {
              return true;
            }
          }
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return false;
  }

  /**
   * Verifica se algum usurio possui uma determinada permisso.
   * 
   * @param permissionId Identificador da permisso.
   * 
   * @return true caso o usurio possua esta permisso, ou false, caso
   *         contrrio.
   * 
   * @throws RemoteException falha na chamada remota rmi
   */
  public static boolean hasAnyUserWithPermission(Object permissionId)
    throws RemoteException {
    List<User> us = getAllUsers();
    for (int i = 0; i < us.size(); i++) {
      Object[] perms = us.get(i).getPermissionIds();
      if (perms != null) {
        for (int j = 0; j < perms.length; j++) {
          if (permissionId.equals(perms[j])) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * Verifica se algum usurio possui um determinado perfil.
   * 
   * @param roleId Identificador do perfil.
   * 
   * @return true caso o usurio possua este perfil, ou false, caso contrrio.
   * 
   * @throws RemoteException falha de rmi
   */
  public static boolean hasAnyUserWithRole(Object roleId)
    throws RemoteException {
    List<User> us = getAllUsers();
    for (int i = 0; i < us.size(); i++) {
      Object[] roles = us.get(i).getRoleIds();
      if (roles != null) {
        for (int j = 0; j < roles.length; j++) {
          if (roleId.equals(roles[j])) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   * Obtm a data da ltima alterao deste usurio.
   * 
   * @return a data em milisegundos desde 1/1/1970.
   */
  public long getLastUpdate() {
    return ((Long) info.getAttribute(LAST_UPDATE)).longValue();
  }

  /**
   * Obtm uma cpia do UserInfo deste usuio.
   * 
   * @return a cpia do UserInfo
   */
  public UserInfo getUserInfo() {
    return (UserInfo) info.clone();
  }

  /**
   * Muda o login do usurio. S pode ser usado para criar um novo usurio.
   * 
   * @param login o login do usurio
   */
  public void setLogin(String login) {
    this.info.setAttribute(LOGIN, login);
  }

  /**
   * Muda o login do super-usurio. Ser utilizado aps um login por delegao,
   * para armazenar o login do super-usurio na sua sesso, de forma que poder
   * ser consultado no cliente.
   * 
   * @param login o login do super-usurio
   */
  public void setSuperUserLogin(String login) {
    this.info.setAttribute(SUPER_USER_LOGIN, login);
  }

  /**
   * Muda o nome do usurio.
   * 
   * @param name o novo nome
   */
  public void setName(String name) {
    this.info.setAttribute(NAME, name);
  }
  
  /**
   * Muda a data de criao do usurio.
   * 
   * @param creationDate a nova data de criao
   */
  public void setCreationDate(Date creationDate) {
    this.info.setAttribute(CREATION_DATE, creationDate);
  }

  /**
   * Muda os emails do usurio.
   * 
   * @param emails os novos emails
   */
  public void setEmails(String[] emails) {
    this.info.setAttribute(EMAILS, emails);
  }

  /**
   * Muda os perfis do usurio.
   * 
   * @param roleIds o novo array de ids de perfis
   */
  public void setRoleIds(Object[] roleIds) {
    this.info.setAttribute(ROLE_IDS, roleIds);
  }

  /**
   * Muda as permisses do usurio.
   * 
   * @param permissionIds a novo array de permisses
   */
  public void setPermissionIds(Object[] permissionIds) {
    this.info.setAttribute(PERMISSION_IDS, permissionIds);
  }

  /**
   * Muda a senha do usurio.
   * 
   * @param password a nova senha
   */
  public void setPassword(String password) {
    this.info.setAttribute(PASSWORD, password);
  }

  /**
   * Muda o digest da senha do usurio.
   * 
   * @param passwordDigest o novo digest
   */
  public void setPasswordDigest(String passwordDigest) {
    this.info.setAttribute(PASSWORD_DIGEST, passwordDigest);
  }

  /**
   * Cria uma chave a partir do identificador nico de cada usurio, para ser
   * utilizada no cache local.
   * 
   * @param id identificador nico de um usurio.
   * @return chave utilizada no cache local para referenciar um usurio.
   */
  private static String createKeyForLocalUsersCache(Object id) {
    return id.toString().toUpperCase();
  }

  /**
   * Obtem uma descrio do usurio.
   * 
   * @return um texto que descreve as informaes desse usurio
   */
  @Override
  public String toString() {
    String userString = "";
    userString += info.getLogin();
    userString += "\t";
    userString += getName();
    userString += "\t";
    userString += Arrays.toString(getEmails());
    userString += "\t";
    return userString;
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        User._USERS.clear();
        User.hasAllUsers = false;
        User.observable.deleteObservers();
      }
    });
  }
}
