package csbase.logic;

/**
 * Indica as possveis causas de falha na execuo de um comando. Esses valores
 * so usados em conjunto com o {@link CommandFinalizationType#FAILED}.
 *
 * @author Tecgraf / PUC-Rio
 */
public enum FailureFinalizationType {
  /**
   * A causa na falha no foi prevista.
   */
  UNKNOWN("Falha imprevista") {
    @Override
    public void changeDescription(String description) {
      this.description = description;
    }
  },

  /**
   * No foi encontrado no servidor um comando identificado como finalizado pelo
   * SGA. O identificador  enviado pelo SGA ao trmino do comando.
   */
  COMMAND_IDENTIFIER_NOT_FOUND("No foi encontrado no servidor um comando identificado como finalizado pelo SGA"),
  /**
   * Erro inesperado na execuo do comando no SGA. Em decorrncia do erro, o
   * SGA no retorna para o servidor um identificador do comando.
   */
  SGA_EXECUTION_ERROR("O SGA no executou o comando em decorrncia de algum erro inesperado"),
  /**
   * O servio CSFS no est disponvel para sincronizar a rea de projetos com
   * a sandbox.
   */
  CSFS_SERVICE_UNAVAILABLE("O servio CSFS no est disponvel para sincronizar a rea de projetos com a sandbox"),
  /**
   * Houve uma falha na na preparao do ambiente de execuo do comando. Essa
   * falha pode ser em decorrncia da verificao do acesso aos arquivos
   * necessrios no ambiente de execuo (binrios, projeto, etc) ou na
   * preparao do ambiente CSFS para execuo do comando (ex: criao da
   * sand-box, transferncia dos arquivos, etc)
   */
  FAILED_SETUP_EXECUTION_ENVIRONMENT("Houve falha na preparao do ambiente de execuo do comando"),
  /**
   * O comando possui prioridade ROOT e no h um SGA disponvel para execuo
   * quando o escalonador seleciona o comando na fila.
   */
  NO_SGA_AVAILABLE_TO_ROOT_COMMAND("No foi encontrado um SGA disponvel para executar o comando com prioridade ROOT"),
  /**
   * O SGA selecionado para execuo no est disponvel.
   */
  SGA_IS_NOT_AVAILABLE("O SGA selecionado no est disponvel para execuo do comando"),
  /**
   * No foi possvel ter acesso a rea do projeto usado para a execuo do
   * comando.
   */
  PROJECT_NOT_FOUND("O projeto usado para execuo do comando no foi encontrado."),
  /**
   * O usurio no possui permisso para execuo no SGA selecionado.
   */
  USER_WITHOUT_PERMISSION_FOR_EXECUTION("O usurio no possui permisso para execuo no SGA selecionado."),
  /**
   * A quantidade mxima de retentativas de submisso de um comando foi
   * atingida.
   */
  MAX_SUBMISSION_RETRIES_REACHED("Quantidade mxima de retentativas de submisso do comando atingida.");

  /**
   * Um texto que descreve a causa da falha.
   */
  protected String description;

  /**
   * Construtor.
   *
   * @param description um texto que descreve a causa da falha do comando.
   */
  private FailureFinalizationType(String description) {
    this.description = description;
  }

  /**
   * Obtm um texto que descreve a causa da falha do comando.
   *
   * @return a descrio da causa da falha
   */
  public String getDescription() {
    return description;
  }

  /**
   * Altera a descrio padro. Somente deve ser usada para a falha
   * {@link #UNKNOWN}.
   *
   * @param description uma descrio para a causa da falha.
   */
  public void changeDescription(String description) {
    // No altera a descrio padro.
  }
}
