/*
 * $Id: ProjectFileType.java 175310 2016-08-03 14:38:03Z fpina $
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;

import csbase.remote.ClientRemoteLocator;
import csbase.util.StringUtils;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * A classe <code>ProjectFileType</code> mantm todos os tipos de arquivos
 * conhecidos pelo sistema.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectFileType implements Serializable,
  Comparable<ProjectFileType> {

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        ProjectFileType.fileTypes = null;
      }
    });
  }

  /**
   * Tipo indicativo de diretrio.
   */
  public static final String DIRECTORY_TYPE = "DIRECTORY_TYPE";

  /**
   * Tipo do arquivo que indica "Arquivo Desconhecido"
   */
  public static final String UNKNOWN = "UNKNOWN";

  /**
   * Guarda as instncias dos tipos de arquivo carregados.
   */
  private static Map<String, ProjectFileType> fileTypes;

  /**
   * Prottipo que serve para criar a instncia de ProjectFileType apropriada
   * (cliente ou lgica)
   */
  protected static ProjectFileType projectFileTypePrototype =
    new ProjectFileType();

  /**
   * Descrio do tipo
   */
  private String description;

  /**
   * As extenses de um tipo de arquivo
   */
  private List<String> extensions;

  /**
   * MimeType do tipo
   */
  private String mimeType;

  /**
   * Cdigo do tipo
   */
  private String typeCode;

  /**
   * Se  diretrio <code>true</code> ou arquivo <code>false</code>.
   */
  private boolean isDirectory;
  
  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(final ProjectFileType fileType) {
    if (this.description == null) {
      return -1;
    }
    if (fileType == null) {
      return -1;
    }
    final String desc = fileType.getDescription();
    if (desc == null) {
      return -1;
    }
    return StringUtils.compare(description, desc);
  }

  /**
   * Cria um ProjectFileType. (Ser sobreescrito na classe cliente para criar o
   * objeto apropriado.)
   * 
   * @param info .
   * 
   * @return .
   */
  protected ProjectFileType createProjectFileType(final ProjectFileTypeInfo info) {
    return new ProjectFileType(info);
  }

  /**
   * Obtm o cdigo do tipo.
   * 
   * @return .
   */
  public String getCode() {
    return this.typeCode;
  }

  /**
   * Obtm a descrio do tipo.
   * 
   * @return .
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Consulta as extenses do arquivo.
   * 
   * @return as extenses
   */
  public List<String> getExtensions() {
    return Collections.unmodifiableList(this.extensions);
  }

  /**
   * Obtm o mimeType
   * 
   * @return .
   */
  public String getMimeType() {
    return this.mimeType;
  }

  /**
   * Atribui a descricao.
   * 
   * @param description .
   */
  public void setDescription(final String description) {
    this.description = description;
  }

  /**
   * Indica se o tipo pode ser atribuido a diretrios <code>true</code> ou
   * a arquivos <code>false</code>.
   *  
   * @return .
   */
  public boolean isDirectory() {
   return isDirectory;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    ProjectFileType other = (ProjectFileType) obj;
    if (!typeCode.equals(other.typeCode)) {
      return false;
    }
    return true;
  }

@Override
  public int hashCode() {
    return typeCode.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.description;
  }

  /**
   * Cria um prottipo do objeto ProjectFileType, para que o mtodo
   * createProjectFileType possa ser chamado de forma apropriada.
   */
  protected ProjectFileType() {
  }

  /**
   * Cria um ProjectFileType
   * 
   * @param info .
   */
  public ProjectFileType(final ProjectFileTypeInfo info) {
    this.typeCode = info.getCode();
    this.description = info.getDescription();
    this.mimeType = info.getMimeType();
    this.extensions = new ArrayList<String>();
    this.extensions.addAll(info.getExtensions());
    this.isDirectory = info.isDirectory();
  }

  /**
   * Obtm os tipos de arquivos aceitos pelo filtro.
   * 
   * @param filter O filtro (no pode ser null).
   * 
   * @return um array com todos os tipos de arquivos.
   */
  public static SortedSet<ProjectFileType> getFileTypes(ProjectFileTypeFilter filter) {
    if (filter == null) {
      throw new IllegalArgumentException("O parmetro filter  null.");
    }

    SortedSet<ProjectFileType> types = new TreeSet<ProjectFileType>();
    for (ProjectFileType type : fileTypes.values()) {
      if (filter.accept(type)) {
        types.add(type);
      }
    }
    return Collections.unmodifiableSortedSet(types);
  }

  /**
   * Obtm um tipo de arquivo que possui um determinado cdigo.
   * 
   * @return o tipo.
   */
  public static ProjectFileType getUnknownFileType() {
    return fileTypes.get(UNKNOWN);
  }

  /**
   * Obtm um tipo de arquivo que possui um determinado cdigo.
   * 
   * @param typeCode o cdigo do tipo.
   * 
   * @return o tipo.
   */
  public static ProjectFileType getFileType(final String typeCode) {
    ProjectFileType type = null;
    /*
     * Se o mapa de fileTypes no foi carregado, obtem do servidor, mas no
     * carrega o mapa local.
     */
    if (fileTypes == null) {
      ProjectFileTypeInfo info;
      try {
        info = ClientRemoteLocator.projectService.getFileType(typeCode);
      }
      catch (final Exception e) {
        return null;
      }
      type =
        (info != null) ? projectFileTypePrototype.createProjectFileType(info)
          : null;
    }
    else {
      type = fileTypes.get(typeCode);
      if (type == null) {
        type = fileTypes.get(UNKNOWN);
      }
    }
    return type;
  }

  /**
   * Consulta o tipo de arquivo sugerido dada uma extenso de arquivo.
   * 
   * @param extension extenso
   * @param isDirectory {@code true} para diretrio ou {@code false} para 
   * arquivo.
   *
   * @return tipo
   */
  public static ProjectFileType getProjectFileTypeFromExtension(
    final String extension, boolean isDirectory) {
    String unknownTypeCode = (isDirectory ? DIRECTORY_TYPE : UNKNOWN);
    ProjectFileType unknownType = ProjectFileType.getFileType(unknownTypeCode);
    if (extension == null) {
      return unknownType;
    }
    for (final ProjectFileType pft : fileTypes.values()) {
      if (pft.isDirectory() == isDirectory) {
        final List<String> exts = pft.getExtensions();
        for (final String ext : exts) {
          if (extension.equalsIgnoreCase(ext)) {
            return pft;
          }
        }
      }
    }
    return unknownType;
  }

  /**
   * Carrega os tipos de arquivos definidos no servidor.
   * 
   * @param locale definio de idioma
   * @throws RemoteException erro de rmi na chamada remota
   */
  public static void loadFileTypes(final Locale locale) throws RemoteException {
    if (fileTypes != null) {
      return;
    }
    fileTypes = new HashMap<String, ProjectFileType>();
    final Map<String, ProjectFileTypeInfo> fileTypesInfo =
      ClientRemoteLocator.projectService.getAllFileTypes(locale);
    for (final ProjectFileTypeInfo info : fileTypesInfo.values()) {
      final ProjectFileType pft =
        projectFileTypePrototype.createProjectFileType(info);
      fileTypes.put(info.getCode(), pft);
    }
  }
}
