package csbase.logic.algorithms.parameters;

import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.ClientSGAFile;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.ClientSGAFileValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * Parmetro que encapsula paths de arquivos que esto na mquina em que o SGA
 * est instalado.
 *
 * @author Tecgraf
 */
public class ClientSGAFileParameter extends SimpleParameter<ClientSGAFile> {

  /**
   * A representao textual do tipo do parmetro.
   */
  public static final String TYPE = "CLIENT_SGA_FILE";

  /** Define se o usurio pode escolher multiplos arquivos. */
  private boolean isMultipleSelection;

  /** Define se o usurio escolhe apenas arquivos. */
  private boolean showFiles;

  /** Define se o usurio escolhe arquivo(s) de entrada. */
  private boolean isInputMode;

  /** Lista com os tipo de arquivos usados no filtro do chooser. */
  private List<String> types;

  /**
   * Construtor padro.
   *
   * @param name - nome do parmetro.
   * @param label - rtulo do parmetro.
   * @param description - a descrio do parmetro.
   * @param defaultValue - valor default.
   * @param isOptional - flag que define se  opcional.
   * @param isVisible - flag que define se o parmetro  visvel.
   * @param commandLinePattern - O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   */
  public ClientSGAFileParameter(String name, String label, String description,
    ClientSGAFile defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SimpleParameterValidator<ClientSGAFile> createParameterValidator() {
    return new ClientSGAFileValidator(isOptional());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("null")
  public void setValueAsText(String parameterValue) throws ParseException {
    if (parameterValue == null) {
      setValue(null);
      return;
    }

    int begin = 0;
    int end = parameterValue.indexOf(",");

    String sgaName = parameterValue.substring(0, end);

    begin = end + 1;
    end = parameterValue.indexOf(";", begin);

    String separator = parameterValue.substring(begin, end);

    String[] splited = parameterValue.substring(end + 1).split(";");

    ClientSGAFile file = null;
    for (int i = 0; i < splited.length; i++) {
      String clientFileStr = splited[i];

      int count = 0;
      begin = 0;
      end = clientFileStr.indexOf(",");
      String[] values = new String[8];
      while (count < 7) {
        values[count] = clientFileStr.substring(begin, end);
        begin = end + 1;
        end = clientFileStr.indexOf(",", begin);
        count++;
      }
      values[count] = clientFileStr.substring(begin);

      if (i == 0) {
        file = buildClientSGAFile(sgaName, separator, values);
      }
      else {
        ClientSGAFile brother = buildClientSGAFile(sgaName, separator, values);
        file.getBrothers().add(brother);
      }
    }

    setValue(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCommandValue(CommandLineContext context) {
    StringBuilder paths = new StringBuilder();

    ClientSGAFile value = getValue();
    if (value == null) {
      return null;
    }

    paths.append(value.getStringPath());

    for (ClientSGAFile brother : value.getBrothers()) {
      paths.append(",");
      paths.append(brother.getStringPath());
    }

    return paths.toString();
  }

  /**
   * Define se o usurio puder escolher multiplos arquivos.
   *
   * @param isMultipleSelection - true se o usurio puder escolher multiplos
   *        arquivos; false caso contrrio.
   */
  public void setMultipleSelection(boolean isMultipleSelection) {
    this.isMultipleSelection = isMultipleSelection;
  }

  /**
   * Define se o usurio puder escolher multiplos arquivos.
   *
   * @return true se o usurio puder escolher multiplos arquivos; false caso
   *         contrrio.
   */
  public boolean isMultipleSelection() {
    return isMultipleSelection;
  }

  /**
   * Define se o usurio escolhe apenas arquivos.
   *
   * @param showFiles - true se o usurio escolhe apenas arquivos; false define
   *        o caso em que o usurio escolhe apenas diretrios.
   */
  public void setShowFiles(boolean showFiles) {
    this.showFiles = showFiles;
  }

  /**
   * Define se o usurio escolhe apenas arquivos.
   *
   * @return true se o usurio escolhe apenas arquivos; false define o caso em
   *         que o usurio escolhe apenas diretrios.
   */
  public boolean isShowFiles() {
    return showFiles;
  }

  /**
   * Define se o usurio escolhe arquivo(s) de entrada.
   *
   * @param isInputMode - true se o usurio escolhe arquivo(s) de entrada; false
   *        define que o usurio escolhe um arquivo de saida.
   */
  public void setInputMode(boolean isInputMode) {
    this.isInputMode = isInputMode;
  }

  /**
   * Define se o usurio escolhe arquivo(s) de entrada.
   *
   * @return true se o usurio escolhe arquivo(s) de entrada; false define que o
   *         usurio escolhe um arquivo de saida.
   */
  public boolean isInputMode() {
    return isInputMode;
  }

  /**
   * Retorna os tipos de arquivos permitidos.
   *
   * @return tipos de arquivos permitidos.
   */
  public List<String> getTypes() {
    return types;
  }

  /**
   * Tipos de arquivos usados no filtro do chooser.
   *
   * @param types - tipos de arquivos usados no filtro do chooser.
   */
  public void setTypes(List<String> types) {
    this.types = types;
  }

  // --------------------- Mtodos privados ------------------------------

  /**
   * Mtodo que constri um objeto {@link ClientSGAFile}.
   *
   * @param sgaName - nome do SGA.
   * @param separator - separador usado nos paths.
   * @param values - valores que compem o objeto {@link ClientSGAFile}.
   * @return objeto {@link ClientSGAFile}.
   */
  private ClientSGAFile buildClientSGAFile(String sgaName, String separator,
    String[] values) {
    ClientSGAFile sgaFile = new ClientSGAFile(sgaName, values[6]);
    sgaFile.setSeparator(separator);
    sgaFile.setDir(Boolean.valueOf(values[0]));
    sgaFile.setSymbolicLink(Boolean.valueOf(values[1]));
    sgaFile.setCanRead(Boolean.valueOf(values[2]));
    sgaFile.setCanWrite(Boolean.valueOf(values[3]));
    sgaFile.setCanExecute(Boolean.valueOf(values[4]));
    sgaFile.setSize(Long.valueOf(values[5]));
    sgaFile.setLinkPath(values[7]);

    return sgaFile;
  }
}
