package csbase.logic.applicationservice;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Properties;

import tecgraf.javautils.core.lng.LNG;

/**
 * Categoria de aplicaes.
 * @author Tecgraf/PUC-Rio
 */
public class ApplicationCategory implements Serializable {

  /**
   * Nome da propriedade que indica se a categoria ser exibida com group frame.
   */
  private static final String SHOWN_AT_APP_PANEL_PROPERTY =
    "shown.at.application.panel";

  /**
   * Nome da propridade que indica se o categoria ser exibida no menu de
   * aplicaes.
   */
  private static final String SHOWN_AT_MENU_PROPERTY = "shown.at.application.menu";

  /**
   * Ids das aplicaoes contidas na categoria
   */
  final private ArrayList<String> appIds = new ArrayList<String>();

  /**
   * Id
   */
  final private String id;

  /**
   * Imagem 16
   */
  final private byte[] img16;

  /**
   * Imagem 32
   */
  final private byte[] img32;

  /**
   * Propriedades.
   */
  final private Properties properties;

  /**
   * Construtor
   * 
   * @param id id
   * @param properties propriedades internas
   * @param img16 imagem 16
   * @param img32 imagem 32
   */
  public ApplicationCategory(String id, Properties properties, byte[] img16,
    byte[] img32) {
    this.id = id;
    this.img16 = img16;
    this.img32 = img32;
    this.properties = properties;
  }

  /**
   * Consulta o nome de uma aplicao.
   * 
   * @param locale locale.
   * @return o nome
   */
  public final String getCategoryName(final Locale locale) {
    return getLocaledProperty("name", locale);
  }

  /**
   * Consulta de propriedade formatada.
   * 
   * @param propertyName o nome da tag desejada.
   * @param locale locale
   * @return o texto com o valor da propriedade desejada.
   */
  private String getLocaledProperty(final String propertyName,
    final Locale locale) {
    final String language = locale.getLanguage();
    final String country = locale.getCountry();
    final Object[] args = new Object[] { language, country };
    final String key = propertyName + ".{0}.{1}";
    final String fmtKey = MessageFormat.format(key, args);
    final String value = getStringSpecificProperty(fmtKey);
    return value;
  }

  /**
   * Obtm uma propriedade especfica.
   * 
   * @param propName nome da propriedade.
   * 
   * @return propriedade especfica.
   */
  final public String getStringSpecificProperty(final String propName) {
    final String tag = id + "." + propName;
    final String value = properties.getProperty(tag);
    if (value == null || value.isEmpty()) {
      final String err = String.format( LNG.get(
    		  "csbase.logic.applicationservice.UndefinedProperty"),
    		  new Object[] { propName, getId() } );
      throw new IllegalArgumentException(err);
    }
    return value.trim();
  }

  /**
   * Obtm uma propriedade especfica.
   * 
   * @param propName nome da propriedade.
   * 
   * @return propriedade especfica.
   */
  final public boolean getBooleanSpecificProperty(final String propName) {
    final String str = getStringSpecificProperty(propName);
    final String[] accepts = new String[] { "true", "false" };
    for (final String acc : accepts) {
      if (acc.equals(str.trim())) {
        final boolean value = Boolean.parseBoolean(str);
        return value;
      }
    }
    final String err = String.format( LNG.get(
    		"csbase.logic.applicationservice.BadBooleanFormat"),
    		new Object[] { propName, getId() } );
    throw new IllegalArgumentException(err);
  }

  /**
   * @return o campo id
   */
  final public String getId() {
    return id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public String toString() {
    return getId();
  }

  /**
   * Consulta se aplicao aparece em menu.
   * 
   * @return um flag indicativo.
   */
  final public boolean isShownAtApplicationMenu() {
    return getBooleanSpecificProperty(SHOWN_AT_MENU_PROPERTY);
  }

  /**
   * Consulta se aplicao aparece no app-panel
   * 
   * @return um flag indicativo.
   */
  final public boolean isShownAtApplicationPanel() {
    return getBooleanSpecificProperty(SHOWN_AT_APP_PANEL_PROPERTY);
  }

  /**
   * Consulta o valor de appIds
   * 
   * @return o valor
   */
  public final ArrayList<String> getApplicationIds() {
    return appIds;
  }

  /**
   * @param appId registro
   */
  public final void addApplicationId(final String appId) {
    appIds.add(appId);
  }

  /**
   * Consulta a imagem.
   * 
   * @return a imagem
   */
  final public byte[] getIcon() {
    return img16;
  }

  /**
   * Consulta a imagem.
   * 
   * @return a imagem
   */
  final public byte[] getImage() {
    return img32;
  }

}
