package csbase.logic;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.LinkedList;
import java.util.List;

import tecgraf.javautils.core.lng.LNG;
import csbase.exception.project.FileLockedException;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Define um arquivo que se encontra na mquina do SGA e que est visvel para o
 * cliente.
 * 
 * @author Tecgraf
 */
public class ClientSGAFile implements ClientFile, Serializable {

  /** Referncia para o pai. */
  private ClientSGAFile parent;

  /** Nome do SGA. */
  private String sgaName;

  /** Path do arquivo que est no SGA. */
  private String path;

  /** Flag que define se este path aponta para um diretrio. */
  private boolean isDir;

  /** Separador usado pelos paths da mquina do SGA. */
  private String separator;

  /** Flag que define se o arquivo pode ser lido. */
  private boolean canRead;

  /** Flag que define se o arquivo pode ser escrito. */
  private boolean canWrite;

  /** Flag que define se o arquivo pode ser executado. */
  private boolean canExecute;

  /** Tamanho do arquivo. */
  private long size;

  /** Path real para o arquivo caso o path seja um link simblico. */
  private String linkPath;

  /** Flag que define se o path  um link simblico. */
  private boolean isSymbolicLink;

  /** Lista com os arquivos irmos. */
  private List<ClientSGAFile> brothers;

  /**
   * Construtor padro que assume como path inicial "/".
   * 
   * @param sgaName - nome do SGA.
   */
  public ClientSGAFile(String sgaName) {
    this(null, sgaName, "/");
  }

  /**
   * Construtor padro.
   * 
   * @param sgaName - nome do SGA.
   * @param path - path do arquivo que est no SGA.
   */
  public ClientSGAFile(String sgaName, String path) {
    this(null, sgaName, path);
  }

  /**
   * Construtor padro.
   * 
   * @param parent - referncia para o pai.
   * @param sgaName - nome do SGA.
   * @param path - path do arquivo que est no SGA.
   */
  private ClientSGAFile(ClientSGAFile parent, String sgaName, String path) {
    if (sgaName == null) {
      throw new IllegalArgumentException(String.format(LNG.get(
    		  "csbase.logic.CannotBeNull"),"Nome do SGA"));
    }
    this.parent = parent;
    this.sgaName = sgaName;
    this.path = (path == null) ? "/" : path;
    if (this.path.equals("/")) {
      this.isDir = true;
    }
    this.separator = "/";
    this.canRead = true;
    this.canWrite = true;
    this.canExecute = true;
    this.size = 0;
    this.linkPath = "";
    this.isSymbolicLink = false;
    this.brothers = new LinkedList<ClientSGAFile>();
  }

  /**
   * Nome do SGA.
   * 
   * @return nome do SGA.
   */
  public String getSGAName() {
    return sgaName;
  }

  /**
   * Retorna true se o path apontar para um diretrio, false caso contrrio.
   * 
   * @return true se o path apontar para um diretrio, false caso contrrio.
   */
  public boolean isDir() {
    return isDir;
  }

  /**
   * True se o path apontar para um diretrio, false caso contrrio.
   * 
   * @param isDir - true se o path apontar para um diretrio, false caso
   *        contrrio.
   */
  public void setDir(boolean isDir) {
    this.isDir = isDir;
  }

  /**
   * Separador usado pelos paths da mquina do SGA.
   * 
   * @param separator - separador usado pelos paths da mquina do SGA.
   */
  public void setSeparator(String separator) {
    this.separator = separator;
  }

  /**
   * Separador usado pelos paths da mquina do SGA.
   * 
   * @return - separador usado pelos paths da mquina do SGA.
   */
  public String getSeparator() {
    return separator;
  }

  /**
   * True se o arquivo pode ser lido, false caso contrrio.
   * 
   * @param canRead - true se o arquivo pode ser lido, false caso contrrio.
   */
  public void setCanRead(boolean canRead) {
    this.canRead = canRead;
  }

  /**
   * True se o arquivo pode ser escrito, false caso contrrio.
   * 
   * @param canWrite - true se o arquivo pode ser escrito, false caso contrrio.
   */
  public void setCanWrite(boolean canWrite) {
    this.canWrite = canWrite;
  }

  /**
   * True se o arquivo pode ser executado, false caso contrrio.
   * 
   * @param canExecute - true se o arquivo pode ser executado, false caso
   *        contrrio.
   */
  public void setCanExecute(boolean canExecute) {
    this.canExecute = canExecute;
  }

  /**
   * Tamanho do arquivo.
   * 
   * @param size - tamanho do arquivo.
   */
  public void setSize(long size) {
    this.size = size;
  }

  /**
   * Retorna o path real para o arquivo caso o path seja um link simblico.
   * 
   * @return path real para o arquivo caso o path seja um link simblico.
   */
  public String getLinkPath() {
    return linkPath;
  }

  /**
   * Retorna true se o path for um link simblico, false caso contrrio.
   * 
   * @return true se o path for um link simblico, false caso contrrio.
   */
  public boolean isSymbolicLink() {
    return isSymbolicLink;
  }

  /**
   * Path real para o arquivo caso o path seja um link simblico.
   * 
   * @param linkPath - path real para o arquivo caso o path seja um link
   *        simblico.
   */
  public void setLinkPath(String linkPath) {
    this.linkPath = linkPath;
  }

  /**
   * True se o path for um link simblico, false caso contrrio.
   * 
   * @param isSymbolicLink - true se o path for um link simblico, false caso
   *        contrrio.
   */
  public void setSymbolicLink(boolean isSymbolicLink) {
    this.isSymbolicLink = isSymbolicLink;
  }

  /**
   * Retorna lista com os irmos selecionados.
   * 
   * @return lista com os irmos selecionados.
   */
  public List<ClientSGAFile> getBrothers() {
    return brothers;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((path == null) ? 0 : path.hashCode());
    result = prime * result + ((sgaName == null) ? 0 : sgaName.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    ClientSGAFile other = (ClientSGAFile) obj;
    if (path == null) {
      if (other.path != null) {
        return false;
      }
    }
    else if (!path.equals(other.path)) {
      return false;
    }
    if (sgaName == null) {
      if (other.sgaName != null) {
        return false;
      }
    }
    else if (!sgaName.equals(other.sgaName)) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append("(sga=");
    builder.append(sgaName);
    builder.append(",path=");
    builder.append(path);
    builder.append(")");

    return builder.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return canRead;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return canWrite;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return canExecute;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean exists() throws IOException {
    SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    ClientSGAFile file = sgaService.getFile(sgaName, path);
    return file != null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFile[] getChildren() throws Exception {
    SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    List<ClientSGAFile> children = sgaService.getChildren(sgaName, path);
    if (children == null) {
      throw new IOException(LNG.get("csbase.logic.OperFailSGANotAvailable"));
    }

    ClientFile[] result = new ClientFile[children.size()];
    for (int i = 0; i < children.size(); i++) {
      ClientSGAFile child = children.get(i);
      child.setParent(this);
      result[i] = child;
    }
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFileType getClientFileType() {
    return ClientFileType.SGA;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    String[] splited = FileUtils.splitPath(path, separator);
    final int length = splited.length;
    if (length <= 0) {
      return path;
    }
    return splited[length - 1];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFile getParent() {
    if (parent != null) {
      return parent;
    }
    if ("/".equals(path) || path.isEmpty()) {
      return null;
    }

    try {
      int i = path.substring(0, path.length() - 1).lastIndexOf(separator);
      if (i != -1) {
        String parentPath = path.substring(0, i + 1);

        SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
        ClientSGAFile parentFile = sgaService.getFile(sgaName, parentPath);
        return parentFile;
      }
    }
    catch (RemoteException e) {
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    String[] splited = FileUtils.splitPath(path, separator);
    return splited;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    return path;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    final String fileExtension = FileUtils.getFileExtension(getName());
    final ProjectFileType pft =
      ProjectFileType.getProjectFileTypeFromExtension(fileExtension, isDirectory());
    return pft.getCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDirectory() {
    return isDir;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long size() {
    return size;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate() {
    return 0L;
  }

  /**
   * Define o objeto pai deste objeto.
   * 
   * @param parent - objeto pai deste objeto.
   */
  private void setParent(ClientSGAFile parent) {
    this.parent = parent;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws IOException {
    throw new UnsupportedOperationException(LNG.get(
		"csbase.logic.UnsupportedOperCloseFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperReadFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperWriteFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(boolean readOnly) throws Exception {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperReadFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long position() throws IOException {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperReadFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperReadFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, long position) throws Exception {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperReadFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len, long position)
    throws Exception {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperReadFile"));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperWriteFile"));

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position)
    throws IOException, FileLockedException {
    throw new UnsupportedOperationException(LNG.get(
      "csbase.logic.UnsupportedOperWriteFile"));
  }

}
