package csbase.logic;

/**
 * Permisso para indicar se um usurio, em um determinado sistema, pode
 * escrever no determinado projeto.
 * 
 * Essa permisso possui dois tipos de atributos:
 * 
 * - {@code sistema=<id_do_sistema>} : representa a identificao (string nica) do
 * sistema que est autorizado a escrever no projeto. Para representar o prprio
 * sistema atual, voc deve criar um atributo "sistema=."
 * 
 * - {@code owner=<nome do servidor>} : representa o nome do servidor dono atual do
 * projeto.
 * 
 */
public class ReadOnlyProjectPermission extends AttributesPermission {
  /** Atributo que define o nome do servidor dono do projeto */
  public static final String OWNER_ID = "owner=";

  /**
   * Atributo que define o identificador (string) nico do sistema em que essa
   * permisso vai ser vlida.
   */
  public static final String SYSTEM_ID = "sistema=";

  /**
   * Identificador do sistema corrente, caso nenhum sistema tenha sido
   * especificado. Esse sistema indica que a chamada remota foi originada no
   * sistema atual.
   */
  private final static String CURRENT_SYSTEM_ID = ".";

  /**
   * Constri uma permisso para escrita no projeto.
   */
  public ReadOnlyProjectPermission() {
    super();
  }

  /**
   * Constri uma permisso para escrita no projeto.
   * 
   * @param name nome da permisso
   * @param description descrio
   * @param attributes atributos da permisso
   */
  public ReadOnlyProjectPermission(String name, String description,
    String[] attributes) {
    super(name, description, attributes);
  }

  /**
   * Verifica se o usurio, no sistema especificado, tem permisso para escrever
   * no projeto.
   * 
   * @param user usurio que solicita a permisso de execuo
   * @param systemId identificador do sistema, em que foi originada a chamada
   *        remota
   * @param ownerId nome do servidor dono do projeto
   * 
   * @return retorna true se o usurio, no sistema especificado, tem permisso
   *         para escrever no determinado projeto, caso contrrio, retorna false
   * @throws Exception erro ocorrido durante a busca pela permisso (ex: de rmi)
   * 
   */
  public static boolean checkSystemAndOwnerPermission(User user,
    String systemId, String ownerId) throws Exception {
    boolean hasPermission = false;
    // Cria o atributo que indica o sistema que originou a chamada
    String systemAttribute =
      ReadOnlyProjectPermission.SYSTEM_ID
        + ((systemId == null) ? CURRENT_SYSTEM_ID : systemId);

    // Cria o atributo que indica o nome do servidor dono do projeto
    String ownerAttribute = ReadOnlyProjectPermission.OWNER_ID + ownerId;

    String[] attributes = new String[] { systemAttribute, ownerAttribute };
    ReadOnlyProjectPermission readOnlyProjectPermission;
    readOnlyProjectPermission =
      (ReadOnlyProjectPermission) user.getMatchAttributesPermission(
        ReadOnlyProjectPermission.class, attributes);
    if (readOnlyProjectPermission != null) {
      hasPermission = true;
    }
    return hasPermission;
  }
}