package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;
import java.util.List;

import csbase.exception.algorithms.ExpressionFunctionExecutionException;
import tecgraf.javautils.core.lng.LNG;

/**
 * Assinatura de um mtodo que trabalha com listas.
 */
public abstract class ListExpressionFunction extends ExpressionFunction {

  /**
   * Construtor.
   * 
   * @param name O nome do mtodo.
   * @param parameterTypes O array com os tipos dos parmetros.
   */
  public ListExpressionFunction(String name, Class<?>... parameterTypes) {
    super(name, createParameterTypes(parameterTypes));
  }

  /**
   * Mtodo que executa a operao.
   * 
   * @param values a lista de valores.
   * @param arguments os parmetros.
   * @return o resultado da operao
   * 
   * @throws ExpressionFunctionExecutionException em caso de falha durante a
   *         execuo.
   */
  protected abstract Object doOperation(List<?> values, Object... arguments)
    throws ExpressionFunctionExecutionException;

  /**
   * {@inheritDoc}
   */
  @Override
  protected final Object doOperation(SimpleAlgorithmConfigurator configurator,
    Object... parameters) throws ExpressionFunctionExecutionException {
    if (configurator == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
		  "csbase.logic.algorithms.nullParameter"),
		  "configurator"));
    }
    if (parameters == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
		  "csbase.logic.algorithms.nullParameter"),
		  "parameters"));
    }
    if (parameters.length == 0) {
      throw new ExpressionFunctionExecutionException(LNG.get(
		  "csbase.logic.algorithms.parameters.MissingListName"),this);
    }
    if (!(parameters[0] instanceof String)) {
      throw new ExpressionFunctionExecutionException(LNG.get(
		  "csbase.logic.algorithms.parameters.InvalidListParameter"),
		  this, parameters[0]);
    }
    String parameterName = (String) parameters[0];
    SimpleParameter<?> parameter =
      configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ExpressionFunctionExecutionException(LNG.get(
		  "csbase.logic.algorithms.parameters.ListNotFoundInConfigurator"),
		  parameterName, configurator, this);
    }
    if (!(parameter instanceof ListParameter)) {
      throw new ExpressionFunctionExecutionException(LNG.get(
		  "csbase.logic.algorithms.parameters.InvalidListConfigurator"),
		  parameterName, configurator, this);
    }
    ListParameter<?> listParameter = (ListParameter<?>) parameter;
    Object[] newParameters = new Object[parameters.length - 1];
    for (int i = 0; i < newParameters.length; i++) {
      newParameters[i] = parameters[i + 1];
    }
    List<?> values = listParameter.getValue();
    return doOperation(values, newParameters);
  }

  /**
   * Cria um novo array de tipos dos parmetros do mtodo a partir do array
   * original. Acrescenta os tipos dos parmetros necessrios para os mtodos
   * sobre listas.
   * 
   * @param parameterTypes o array original de tipos.
   * 
   * @return o novo array de tipos.
   */
  private static Class<?>[] createParameterTypes(Class<?>[] parameterTypes) {
    if (parameterTypes == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
		  "csbase.logic.algorithms.nullParameter"),
		  "parameterTypes"));
    }
    Class<?>[] newParameterTypes = new Class[parameterTypes.length + 1];
    newParameterTypes[0] = String.class;
    for (int i = 1; i < newParameterTypes.length; i++) {
      newParameterTypes[i] = parameterTypes[i - 1];
    }
    return newParameterTypes;
  }
}
