package csbase.logic.algorithms.parameters;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.ListParameterValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.parsers.exception.AutomatonException;

import java.text.MessageFormat;
import java.util.*;

/**
 * Parmetro do Tipo Lista.
 * 
 * @param <E> O tipo do elemento armazenado na lista.
 * 
 * @author lmoreira
 */
public abstract class ListParameter<E> extends SimpleParameter<List<E>> {

  /**
   * Indica se os elementos precisam ser ordenados.
   */
  protected boolean mustSort;

  /**
   * Indica se o parmetro aceita elementos repetidos,
   */
  protected boolean acceptDuplicates;

  /**
   * Cria o parmetro.
   * 
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param mustSort Indica se os caminhos para os arquivos precisam ser
   *        ordenados.
   * @param acceptDuplicates Indica se o parmetro aceita elementos repetidos.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   */
  protected ListParameter(String name, String label, String description,
    List<E> defaultValue, boolean isOptional, boolean isVisible,
    boolean mustSort, boolean acceptDuplicates, String commandLinePattern) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    this.mustSort = mustSort;
    this.acceptDuplicates = acceptDuplicates;
  }

  /**
   * <p>
   * Adiciona um elemento ao valor deste parmetro.
   * </p>
   * 
   * <p>
   * Gera o evento {@link ParameterListener#valueWasChanged(Parameter)}.
   * </p>
   * 
   * @param element O elemento (No aceita {@code null}).
   * @return {@code true} se o elemento foi adicionado com sucesso.
   */
  public final boolean addElement(E element) {
    if (element == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "element"));
    }
    List<E> values = getValue();
    List<E> newValues;
    if (values == null) {
      newValues = new LinkedList<E>();
    }
    else {
      newValues = new LinkedList<E>(values);
    }
    if (!acceptDuplicates && newValues.contains(element)) {
      return false;
    }
    newValues.add(element);
    if (mustSort) {
      Collections.sort(newValues, getItemComparator());
    }
    return setValue(newValues);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAsText() {
    List<E> values = getValue();
    if (values == null || values.isEmpty()) {
      return null;
    }
    StringBuffer buffer = new StringBuffer();
    buffer.append(ListParameterParser.START_LINE_CHAR);
    Iterator<E> valueIterator = values.iterator();
    String separator = "";
    while (valueIterator.hasNext()) {
      buffer.append(separator);
      buffer.append(getItemValueAsText(valueIterator.next()));
      separator =
        Character.toString(ListParameterParser.ELEMENT_SEPARATOR_CHAR);
    }
    buffer.append(ListParameterParser.END_LINE_CHAR);
    return buffer.toString();
  }

  /**
   * Remove elemento especfico.
   * 
   * @param element elemento
   * @return indicativo conforme {@link #removeElement(int)};
   */
  public final boolean removeElementObj(E element) {
    if (element == null) {
      return false;
    }
    List<E> values = getValue();
    final int index = values.indexOf(element);
    if (index < 0) {
      return false;
    }
    return removeElement(index);
  }

  /**
   * <p>
   * Remove um elemento especfico.
   * </p>
   * 
   * <p>
   * Gera o evento {@link ParameterListener#valueWasChanged(Parameter)}.
   * </p>
   * 
   * @param index O ndice do elemento (A faixa aceita  de 0 a quantidade de
   *        elementos - 1).
   * @return {@code true} se o elemento foi removido com sucesso.
   */
  public final boolean removeElement(int index) {
    List<E> values = getValue();
    if (values == null) {
      return false;
    }
    if (index < 0 || index >= values.size()) {
      return false;
    }
    if (values.size() == 1) {
      setValue(null);
    }
    else {
      List<E> newValues = new LinkedList<E>(values);
      newValues.remove(index);
      setValue(newValues);
    }
    return true;
  }

  /**
   * <p>
   * Troca um elemento por outro.
   * </p>
   * 
   * <p>
   * Gera o evento {@link ParameterListener#valueWasChanged(Parameter)}.
   * </p>
   * 
   * @param index O ndice do elemento (A faixa aceita  de 0 a quantidade de
   *        elementos - 1).
   * @param newElement O novo elemento (Aceita {@code null}).
   * 
   * 
   * @return {@code true} em caso de sucesso ou {@code false} se o elemento que
   *         estava armazenado na posio informada  igual ao elemento
   *         fornecido.
   */
  public final boolean setElement(int index, E newElement) {
    List<E> values = getValue();
    if (values == null) {
      throw new UnsupportedOperationException(LNG.get(
        "csbase.logic.algorithms.parameters.EmptyElementList"));
    }
    if (index < 0 || index >= values.size()) {
      String message =
        MessageFormat.format(LNG.get(
          "csbase.logic.algorithms.parameters.IndexOutOfBounds"),
          new Object[] { new Integer(index), new Integer(values.size()) });
      throw new IllegalArgumentException(message);
    }
    Object oldElement = values.get(index);
    if (oldElement.equals(newElement)) {
      return false;
    }
    List<E> newValues = new LinkedList<E>(values);
    newValues.set(index, newElement);
    setValue(newValues);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if (parameterValue != null) {
      ListParameterParser parser = new ListParameterParser();
      List<E> values = new LinkedList<E>();
      List<String> textValues;
      try {
        textValues = parser.parseText(parameterValue);
      }
      catch (AutomatonException e) {
        throw new ParseException(e, LNG.get(
    		"csbase.logic.algorithms.parameters.InvalidToList"),
    		new Object[] { parameterValue, this });
      }
      Iterator<String> textValueIterator = textValues.iterator();
      while (textValueIterator.hasNext()) {
        String textValue = textValueIterator.next();
        values.add(getItemValueFromText(textValue));
      }
      setValue(values);
    }
    else {
      setValue(null);
    }
  }

  /**
   * Decodifica um texto criando um elemento.
   * 
   * @param elementAsText O valor do elemento codificado (Aceita {@code null}).
   * 
   * @return O valor do elemento.
   * 
   * @throws ParseException Se o valor codificado no representar um elemento
   *         que possa ser armazenado neste parmetro.
   */
  public abstract E getItemValueFromText(String elementAsText)
    throws ParseException;

  /**
   * Codifica um elemento criando um texto.
   * 
   * @param element O elemento a ser codificado (Aceita {@code null}).
   * 
   * @return O texto codificado ({@code null} se o valor do elemento for
   *         {@code null}).
   */
  protected String getItemValueAsText(E element) {
    if (element == null) {
      return null;
    }
    return element.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCommandValue(CommandLineContext context) {
    List<E> values = getValue();
    if (values == null) {
      return null;
    }
    StringBuffer buffer = new StringBuffer();
    String separator = "";
    Iterator<E> valueIterator = values.iterator();
    while (valueIterator.hasNext()) {
      E value = valueIterator.next();
      buffer.append(separator);
      buffer.append(getCommandItemValue(context, value));
      separator = ",";
    }
    return buffer.toString();
  }

  /**
   * Obtm o valor de um tem da lista pronto para ser utilizado na linha de
   * comando.
   * 
   * @param context Contexto para gerao da linha de comando.
   * @param itemValue O valor (No aceita {@code null}).
   * 
   * @return O valor convertido.
   */
  protected String getCommandItemValue(CommandLineContext context, E itemValue) {
    return itemValue.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ListParameterValidator<E> createParameterValidator() {
    return new ListParameterValidator<E>(createItemValidator());
  }

  /**
   * Cria o validador dos itens da lista.
   * 
   * @return O validador.
   */
  protected abstract SimpleParameterValidator<E> createItemValidator();

  /**
   * Indica se precisa ordenar a lista de valores.
   * 
   * @return true se precisa e false caso contrrio.
   */
  public boolean mustSort() {
    return mustSort;
  }

  /**
   * Cria um comparador de itens da lista.
   * 
   * @return o comparador.
   */
  protected Comparator<E> getItemComparator() {
    return new Comparator<E>() {
      @Override
      public int compare(E o1, E o2) {
        if (o1 == null && o2 == null) {
          return 0;
        }
        if (o1 != null && o2 == null) {
          return 1;
        }
        if (o1 == null && o2 != null) {
          return -1;
        }
        return getItemValueAsText(o1).compareTo(getItemValueAsText(o2));
      }
    };
  }
}
