package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.exception.algorithms.FormulaCreationException;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.ValidationExpression;
import tecgraf.javautils.core.lng.LNG;

/**
 * <p>
 * Analisador de {@link ValidationExpression}.
 * </p>
 * 
 * <p>
 * Este parser l os atributos que representam as expresses de validao. O
 * elemento corrente do {@link XmlParser analisador de XML} precisa ser um
 * elemento {@link #EXPRESSION_ELEMENT}.
 * </p>
 */
public class ValidationExpressionParser {

  /**
   * <p>
   * O elemento {@value #EXPRESSION_ELEMENT}: descreve as proriedades de um
   * {@link ValidationExpression validador por expresso} a ser utilizado no
   * configurador de algoritemos.
   * </p>
   * <p>
   * O seu contedo  obrigatrio e representa a expresso que ser utilizada no
   * processo de validao.
   * </p>
   * <p>
   *  elemento-filho do elemento
   * {@link SimpleAlgorithmParser#ALGORITHM_ELEMENT}.
   * </p>
   */
  static final String EXPRESSION_ELEMENT = "expressao";

  /**
   * <p>
   * O atributo {@value #EXPRESSION_ELEMENT_ERROR_ATTRIBUTE} do elemento
   * {@link #EXPRESSION_ELEMENT}. Indica a mensagem de erro que ser exibida ao
   * usurio se a expresso for avaliada como falsa.  obrigatria e o seu tipo
   *  string.
   * </p>
   */
  private static final String EXPRESSION_ELEMENT_ERROR_ATTRIBUTE = "erro";

  /**
   * <p>
   * Carrega uma {@link ValidationExpression}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser
   * {@link #EXPRESSION_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador(No aceita {@code null}).
   * @param configurator O configurador de algoritmo (No aceita {@code null}).
   * 
   * @throws ParseException Em caso de erro no XML.
   */
  public void loadExpression(XmlParser parser,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    String errorMessage =
      parser.extractAttributeValue(EXPRESSION_ELEMENT_ERROR_ATTRIBUTE);
    String expressionText = parser.getElementValue(null);
    if (expressionText == null) {
      throw new ParseException(LNG.get(
        "csbase.logic.algorithms.parsers.MissingExprText"),
        new Object[] { EXPRESSION_ELEMENT });
    }
    parser.checkAttributes();
    parser.checkChildElements();

    ValidationExpression expression;
    try {
      expression =
        new ValidationExpression(configurator, expressionText, errorMessage);
    }
    catch (FormulaCreationException e) {
      throw new ParseException(e, e.getLocalizedMessage());
    }
    if (!configurator.addExpression(expression)) {
      throw new ParseException(LNG.get(
        "csbase.logic.algorithms.parsers.DuplcatedExprInAlgo"), new Object[] {
            expression, configurator });
    }
  }

}
