package csbase.logic.algorithms.parsers.triggers;

import java.util.Collection;

import csbase.exception.OperationFailureException;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.triggers.Trigger;

/**
 * <p>
 * Classe abstrata para simplificar a criao de {@link TriggerFactory fbricas
 * de gatilhos} de {@link ParameterGroup parmetros grupos}.
 * </p>
 * 
 * <p>
 * Ela implementa os mtodos requiridos deixando mtodos abstratos nos pontos
 * que so especficos de acordo com o tipo de fbrica.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstractGroupTriggerFactory extends AbstracTriggerFactory {
  /**
   * <p>
   * Atributo {@value #GROUP_ATTRIBUTE}: ele  um atributo para os elementos que
   * define os gatilhos.
   * </p>
   * 
   * <p>
   *  uma das formas de indicar o nome do grupo que ser manipulado pelo
   * gatilho. A outra forma  {@link #GROUP_ELEMENT}.
   * </p>
   * 
   * <p>
   *  opcional e  do tipo string. Ele  opcional, pois pode-se utilizar o
   * elemento {@link #GROUP_ELEMENT} com o mesmo sentido, portanto pelo menos um
   * dos 2 tem que existir.*
   * </p>
   */
  private static final String GROUP_ATTRIBUTE = "grupo";

  /**
   * <p>
   * Elemento {@value #GROUP_ELEMENT}:
   * </p>
   * 
   * <p>
   *  uma das formas de indicar o nome do grupo que ser manipulado pelo
   * gatilho. A outra forma  {@link #GROUP_ATTRIBUTE}.
   * </p>
   * 
   * <p>
   *  uma das formas de indicar o nome do grupo que ser manipulado pelo
   * gatilho.
   * </p>
   */
  private static final String GROUP_ELEMENT = "grupo";

  /**
   * O atributo {@value #GROUP_ELEMENT_ID_ATTRIBUTE} do elemento
   * {@link #GROUP_ELEMENT}: indica o nome do grupo manipulado pelos gatilhos. 
   * obrigatrio e  do tipo string.
   */
  private static final String GROUP_ELEMENT_ID_ATTRIBUTE = "id";

  /**
   * Cria a fbrica.
   * 
   * @param elementName O nome do elemento que define as propriedades do
   *        gatilho.
   * @param allowMultiples Indica se essa fbrica  capaz de criar mltiplas
   *        instncias do mesmo {@link Trigger gatilhos}.
   */
  protected AbstractGroupTriggerFactory(String elementName,
    boolean allowMultiples) {
    super(elementName, GROUP_ELEMENT, GROUP_ELEMENT_ID_ATTRIBUTE,
      GROUP_ATTRIBUTE, allowMultiples);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final ParameterGroup findParameter(
    SimpleAlgorithmConfigurator configurator, String parameterName)
    throws OperationFailureException {
    ParameterGroup triggerGroup =
      findGroupByName(configurator.getGroups(), parameterName);
    return triggerGroup;
  }

  /**
   * Busca recursivamente um grupo pelo nome a partir de um conjunto de grupos.
   * 
   * @param groups o conjunto de grupos.
   * @param groupName o nome do grupo a ser buscado.
   * 
   * @return parameter o grupo com o nome especificado ou nulo, caso no seja
   *         encontrado nenhum grupo com o nome.
   * 
   * @throws OperationFailureException em caso de se encontrar mais de um grupo
   *         com o mesmo nome.
   */
  private ParameterGroup findGroupByName(Collection<ParameterGroup> groups,
    String groupName) throws OperationFailureException {
    ParameterGroup groupFound = null;
    for (ParameterGroup group : groups) {
      if (group.getName().equals(groupName)) {
        if (groupFound == null) {
          groupFound = group;
        }
        else {
          throw new OperationFailureException(
            "Foi encontrado mais de um grupo com o identificador \"{0}\".",
            groupName);
        }
      }
      ParameterGroup childGroup = findGroupByName(group.getGroups(), groupName);
      if (childGroup != null) {
        if (groupFound == null) {
          groupFound = childGroup;
        }
        else {
          throw new OperationFailureException(
            "Foi encontrado mais de um grupo com o identificador \"{0}\".",
            groupName);
        }
      }
    }
    return groupFound;
  }

}
