package csbase.logic.algorithms.parsers.triggers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.DoubleListParameter;
import csbase.logic.algorithms.parameters.DoubleParameter;
import csbase.logic.algorithms.parameters.IntegerListParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.conditions.Condition;
import csbase.logic.algorithms.parameters.triggers.ChangeMinimumForDoublesTrigger;
import csbase.logic.algorithms.parameters.triggers.ChangeMinimumForIntegersTrigger;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.logic.algorithms.parsers.DoubleListParameterFactory;
import csbase.logic.algorithms.parsers.DoubleParameterFactory;
import csbase.logic.algorithms.parsers.IntegerListParameterFactory;
import csbase.logic.algorithms.parsers.IntegerParameterFactory;
import csbase.logic.algorithms.parsers.XmlParser;

/**
 * Fbrica de {@link ChangeMinimumForDoublesTrigger} ou
 * {@link ChangeMinimumForIntegersTrigger}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ChangeMinimumTriggerFactory extends
  AbstractSimpleTriggerFactory {
  /**
   * Atributo {@value #INCLUDE_NEW_MINIMUM_ATTRIBUTE}: indica se o valor mnimo
   * indicado por {@link #NEW_MINIMUM_ATTRIBUTE} se  inclusivo ({@code true})
   * ou  exclusivo ({@code false}).  opcional (o valor-padro 
   * {@link #INCLUDE_NEW_MINIMUM_DEFAULT_VALUE}).  do tipo booleano. S 
   * aplicvel se o parmetro for tipo do tipo real.
   */
  private static final String INCLUDE_NEW_MINIMUM_ATTRIBUTE =
    "incluir_novo_minimo";

  /**
   * <p>
   * O valor-padro para o atributo {@link #INCLUDE_NEW_MINIMUM_DEFAULT_VALUE}.
   * </p>
   * <p>
   * O seu valor  {@value #INCLUDE_NEW_MINIMUM_DEFAULT_VALUE}.
   * </p>
   */
  private static final boolean INCLUDE_NEW_MINIMUM_DEFAULT_VALUE = true;

  /**
   * Atributo {@value #NEW_MINIMUM_ATTRIBUTE}: indica o valor mnimo. 
   * obrigatrio. O seu tipo depende do tipo do parmetro.
   */
  private static final String NEW_MINIMUM_ATTRIBUTE = "novo_minimo";

  /**
   * Cria a fbrica.
   */
  public ChangeMinimumTriggerFactory() {
    super("trocar_minimo", false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Trigger<?> doCreateTrigger(XmlParser parser, Condition condition,
    Parameter<?> parameter, SimpleAlgorithmConfigurator configurator)
      throws ParseException {
    Trigger<?> trigger;
    if (parameter instanceof DoubleParameter) {
      Double minimum = parser.extractAttributeValueAsDouble(
        NEW_MINIMUM_ATTRIBUTE, null);
      boolean isMinimumIncluded = parser.extractAttributeValueAsBoolean(
        INCLUDE_NEW_MINIMUM_ATTRIBUTE, INCLUDE_NEW_MINIMUM_DEFAULT_VALUE);
      DoubleParameter doubleParameter = (DoubleParameter) parameter;
      trigger = new ChangeMinimumForDoublesTrigger(doubleParameter, condition,
        minimum, isMinimumIncluded);
    }
    else if (parameter instanceof DoubleListParameter) {
      Double minimum = parser.extractAttributeValueAsDouble(
        NEW_MINIMUM_ATTRIBUTE, null);
      boolean isMinimumIncluded = parser.extractAttributeValueAsBoolean(
        INCLUDE_NEW_MINIMUM_ATTRIBUTE, true);
      DoubleListParameter doubleListParameter = (DoubleListParameter) parameter;
      trigger = new ChangeMinimumForDoublesTrigger(doubleListParameter,
        condition, minimum, isMinimumIncluded);
    }
    else if (parameter instanceof IntegerParameter) {
      Integer minimum = parser.extractAttributeValueAsInteger(
        NEW_MINIMUM_ATTRIBUTE, null);
      IntegerParameter integerParameter = (IntegerParameter) parameter;
      trigger = new ChangeMinimumForIntegersTrigger(integerParameter, condition,
        minimum);
    }
    else if (parameter instanceof IntegerListParameter) {
      Integer minimum = parser.extractAttributeValueAsInteger(
        NEW_MINIMUM_ATTRIBUTE, null);
      IntegerListParameter integerListParameter =
        (IntegerListParameter) parameter;
      trigger = new ChangeMinimumForIntegersTrigger(integerListParameter,
        condition, minimum);
    }
    else {
      throw new ParseException("O parmetro {0} no  de um tipo vlido.\n"
        + "Tipos permitidos:\n{1};\n{2};\n{3};\n{4}..", parameter.getName(),
        DoubleParameterFactory.DOUBLE_PARAMETER_ELEMENT,
        DoubleListParameterFactory.DOUBLE_LIST_PARAMETER_ELEMENT,
        IntegerParameterFactory.INTEGER_PARAMETER_ELEMENT,
        IntegerListParameterFactory.INTEGER_LIST_PARAMETER_ELEMENT);
    }
    return trigger;

  }

}
