/*
 * Detalhes da ltima alterao:
 *
 * $Author: fpina $ $Date: 2016-09-22 18:12:51 -0300 (Thu, 22 Sep 2016) $
 * $Revision: 176168 $
 */
package csbase.remote;

import java.io.Serializable;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.security.PublicKey;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import csbase.logic.EncryptedPassword;
import csbase.logic.PreLoginData;
import csbase.logic.ServerURI;
import csbase.logic.Session;
import csbase.logic.UserOutline;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.util.rmi.IPingable;

/**
 * Define as funcionalidades de um servidor CSBase que so visveis para o
 * cliente. A partir desta interface remota que deve ser feito lookup no
 * servidor.
 *
 * <p>
 * Ex: <br>
 * <code>  ServerEntryPoint server = (ServerEntryPoint)
 * Naming.lookup("rmi://"+serverName+"/"+ServerEntryPoint.LOOKUP); </code>
 * </p>
 */
public interface ServerEntryPoint extends IPingable {
  /**
   * Constante utilizada para fazer lookup no servidor
   */
  public static String LOOKUP = "Server";

  /**
   * Informa a verso do servidor em execuo.
   *
   * @return A verso do servidor.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String getVersionName() throws RemoteException;

  /**
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha. Esse token pode ser utilizado como parmetro em endereos urls.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   *
   * @return Dados sobe o <i>preLogin</i> realizado.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #login(String)
   */
  public PreLoginData preLogin(String login, String password, Locale locale)
    throws RemoteException;

  /**
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha. Esse token pode ser utilizado como parmetro em endereos urls.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @param attributes Atributos da sesso do usurio.
   *
   * @return Dados sobe o <i>preLogin</i> realizado.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #login(String)
   */
  public PreLoginData preLogin(String login, String password, Locale locale,
    Map<String, Serializable> attributes) throws RemoteException;

  /**
   * <p>
   * Autentica um usurio no sistema e cria um token para ser usado como
   * identificao em um posterior login onde no h a informao de usurio e
   * senha. Esse token pode ser utilizado como parmetro em endereos urls.
   * </p>
   *
   * <p>
   * O usurio  autenticado por um super-usurio, que  um usurio com
   * permisso de super-usurio({@link csbase.logic.SuperUserPermission}).
   * Autentica e cria no servidor uma sesso do usurio delegado.
   * </p>
   *
   * <p>
   * OBS: O Administrador(admin) no precisa de permisso de super-usurio para
   * delegar um outro usurio e, por questes de segurana, no pode ser um
   * usurio delegado por nenhum super-usurio.
   * </p>
   *
   *
   * @param login o identificador do super-usurio.
   * @param password a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param delegatedLogin login do usurio que ser delegado pelo super-usurio
   *        sem a necessidade de senha.
   * @param control Objeto de controle.
   * @param userData Identificao do usurio.
   *
   * @return Dados sobe o <i>preLogin</i> realizado.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #login(String)
   */
  PreLoginData preLogin(String login, String password, Locale locale,
    String delegatedLogin, Remote control, Serializable userData)
    throws RemoteException;

  /**
   * Autentica um usurio no sistema especificado e cria um token para ser usado
   * como identificao em um posterior login onde no h a informao de
   * usurio e senha. Esse token pode ser utilizado como parmetro em endereos
   * urls.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @param systemName O nome do sistema onde o usurio deseja se autenticar.
   *
   * @return Uma string que  o token para identificao de um pr-login vlido.
   *         <code>null</code> se o login e/ou senha forem invlidos.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #login(String)
   */
  public String preLogin2Web(String login, String password, Locale locale,
    String systemName) throws RemoteException;

  /**
   * Autentica um usurio no sistema especificado e cria um token para ser usado
   * como identificao em um posterior login onde no h a informao de
   * usurio e senha. Esse token pode ser utilizado como parmetro em endereos
   * urls.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @param delegatedLogin login do usurio que ser delegado pelo super-usurio
   *        sem a necessidade de senha.
   * @param systemName O nome do sistema onde o usurio deseja se autenticar.
   *
   * @return Uma string que  o token para identificao de um pr-login vlido.
   *         <code>null</code> se o login e/ou senha forem invlidos.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #login(String)
   */
  public String preLogin2Web(String login, String password, Locale locale,
    String delegatedLogin, String systemName) throws RemoteException;

  /**
   * Autentica um usurio no sistema e retorna uma url de acesso ao mesmo, que
   * permitir abrir o sistema sem necessidade de perguntar ao usurio as
   * informaes de login e senha. Para se concatenar parmetros nessa url
   * deve-se usar {@code "&"}.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   *
   * @return Uma url para carga do cliente do sistema. <code>null</code> se o
   *         login e/ou senha forem invlidos.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String preLogin2Web(String login, String password, Locale locale)
    throws RemoteException;

  /**
   * Autentica um usurio no sistema e retorna uma url de acesso ao mesmo, que
   * permitir abrir o sistema sem necessidade de perguntar ao usurio as
   * informaes de login e senha. Para se concatenar parmetros nessa url
   * deve-se usar {@code "&"}.
   *
   * @param login O identificador do usurio que deseja fazer seu login.
   * @param password A senha do respectivo usurio.
   * @param locale A Locale que o usurio deseja utilizar no posterior login.
   * @param attributes Atributos da sesso do usurio.
   *
   * @return Uma url para carga do cliente do sistema. <code>null</code> se o
   *         login e/ou senha forem invlidos.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String preLogin2Web(String login, String password, Locale locale,
    Map<String, Serializable> attributes) throws RemoteException;

  /**
   * Executa o login de um usurio via usurio e senha.
   * <p>
   * Pode ser utilizado para se logar a primeira vez ou aps uma queda do
   * servidor.
   * </p>
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do usurio que deseja fazer seu login.
   * @param encryptedPassword a senha do respectivo usurio.
   * @param locale A Locale que o usurio escolheu no login.
   *
   * @return Uma chave que identifica um usurio logado. <code>null</code> se
   *         chave e/ou senha foram invlidos
   *
   * @throws RemoteException se ocorrer algum problema na comunicao com o
   *         servidor.
   *
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale) throws RemoteException;

  /**
   * Executa o login de um usurio via usurio e senha.
   * <p>
   * Pode ser utilizado para se logar a primeira vez ou aps uma queda do
   * servidor.
   * </p>
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do usurio que deseja fazer seu login.
   * @param encryptedPassword a senha do respectivo usurio.
   * @param locale A Locale que o usurio escolheu no login.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   *
   * @return Uma chave que identifica um usurio logado. <code>null</code> se
   *         chave e/ou senha foram invlidos
   *
   * @throws RemoteException se ocorrer algum problema na comunicao com o
   *         servidor.
   *
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, TimeZone timeZone) throws RemoteException;

  /**
   * Executa o login de um usurio via login e senha com a possibilidade de
   * delegar o login a um outro usurio, para isso o usurio dever ser um
   * super-usurio do sistema que significa ter permisso de super-usurio(
   * {@link csbase.logic.SuperUserPermission}). Autentica e cria no servidor uma
   * sesso do usurio delegado. Pode ser utilizado para se logar a primeira vez
   * ou aps uma queda do servidor. OBS: O Administrador(admin) no precisa de
   * permisso de super-usurio para delegar um outro usurio e, por questes de
   * segurana, no pode ser um usurio delegado por nenhum super-usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param encryptedPassword a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param delegatedLogin login do usurio que ser delegado pelo super-usurio
   *        sem a necessidade de senha.
   *
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   *         invlidos
   *
   * @throws RemoteException se ocorrer algum problema na comunicao com o
   *         servidor.
   *
   * @see csbase.logic.SuperUserPermission
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, String delegatedLogin) throws RemoteException;

  /**
   * Executa o login de um usurio via login e senha com a possibilidade de
   * delegar o login a um outro usurio, para isso o usurio dever ser um
   * super-usurio do sistema que significa ter permisso de super-usurio(
   * {@link csbase.logic.SuperUserPermission}). Autentica e cria no servidor uma
   * sesso do usurio delegado. Pode ser utilizado para se logar a primeira vez
   * ou aps uma queda do servidor. OBS: O Administrador(admin) no precisa de
   * permisso de super-usurio para delegar um outro usurio e, por questes de
   * segurana, no pode ser um usurio delegado por nenhum super-usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param encryptedPassword a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param tz O TimeZone que ser usado no login do usurio delegado
   * @param delegatedLogin login do usurio que ser delegado pelo super-usurio
   *        sem a necessidade de senha.
   *
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   *         invlidos
   *
   * @throws RemoteException se ocorrer algum problema na comunicao com o
   *         servidor.
   *
   * @see csbase.logic.SuperUserPermission
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, TimeZone tz, String delegatedLogin) throws RemoteException;

  /**
   * Executa o login de um usurio via login e senha com a possibilidade de
   * delegar o login a um outro usurio, para isso o usurio dever ser um
   * super-usurio do sistema que significa ter permisso de super-usurio (
   * {@link csbase.logic.SuperUserPermission}). Autentica e cria no servidor uma
   * sesso do usurio delegado. Pode ser utilizado para se logar a primeira vez
   * ou aps uma queda do servidor. OBS: O Administrador(admin) no precisa de
   * permisso de super-usurio para delegar um outro usurio e, por questes de
   * segurana, no pode ser um usurio delegado por nenhum super-usurio.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param login o identificador do super-usurio.
   * @param encryptedPassword a senha do super-usurio.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param tz O TimeZone que ser usado no login do usurio delegado
   * @param delegatedLogin login do usurio que ser delegado pelo super-usurio
   *        sem a necessidade de senha.
   * @param params Parmetros que sero copiados para a sesso do usurio
   *
   * @return A sesso do usurio. <code>null</code> se chave e/ou senha foram
   *         invlidos
   *
   * @throws RemoteException se ocorrer algum problema na comunicao com o
   *         servidor.
   *
   * @see csbase.logic.SuperUserPermission
   * @see #logout(Object)
   */
  public Session login(String login, EncryptedPassword encryptedPassword,
    Locale locale, TimeZone tz, String delegatedLogin,
    Map<String, Serializable> params) throws RemoteException;

  /**
   * Executa o login de um usurio que fez anteriormente um preLogin.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param token Identificao de um preLogin.
   *
   * @return Uma chave que identifica um usurio logado. <code>null</code> se
   *         chave e/ou senha foram invlidos
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #preLogin(String, String, Locale)
   * @see #logout(Object)
   */
  public Session login(String token) throws RemoteException;

  /**
   * Executa o login de um usurio que fez anteriormente um preLogin.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param token Identificao de um preLogin.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   *
   * @return Uma chave que identifica um usurio logado. <code>null</code> se
   *         chave e/ou senha foram invlidos
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #preLogin(String, String, Locale)
   * @see #logout(Object)
   */
  public Session login(String token, TimeZone timeZone) throws RemoteException;

  /**
   * Executa o login de um usurio conectado ao OpenBus.
   *
   * @param token O token de login do OpenBus.
   * @param locale A Locale que ser usado no login do usurio delegado.
   * @param timeZone TimeZone do cliente pelo qual o usurio se logou.
   *
   * @return Uma chave que identifica um usurio logado. <code>null</code> se
   *         chave e/ou senha foram invlidos
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #logout(Object)
   */
  public Session login(OpenBusLoginToken token, Locale locale,
    TimeZone timeZone) throws RemoteException;

  /**
   * Executa o login de um servidor local.
   * <p>
   * <b>Ateno:</b> Para cada ao de login deve haver uma ao de logout
   * associada.
   * </p>
   *
   * @param localServerName O nome do servidor local.
   * @param signedLocalServerName O nome do servidor local, assinado.
   * @param locale O locale do servidor local.
   *
   * @return Uma chave que identifica um usurio logado. <code>null</code> se
   *         chave e/ou senha foram invlidos
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   *
   * @see #logout(Object)
   */
  public Session login(String localServerName, byte[] signedLocalServerName,
    Locale locale) throws RemoteException;

  /**
   * Executa o login de um usurio que  validado atravs de uma referncia de
   * outro servidor onde ele deve ter uma sesso vlida. O servidor deve possuir
   * o certificado do servidor referenciado para garantir a relao de
   * confiana. Se o servidor referenciado for o mesmo (mesmo host e porta deste
   * servidor) ento basta verificar se a sesso  valida.
   *
   * @param referedServerURI A URI do servidor de referncia para validao
   * @param attr Atributos para serem armazenados na sesso do usurio
   * @param sessionkey A chave da sesso a ser validada no servidor de
   *        referncia
   * @param copyServerSessionAttrs Se true copia os attributos de sesso do
   *        servidor de referncia para a nova sesso criada nesse servidor
   * @param login O login do usurio
   * @param delegatedLogin O login para delegao ou null se no for o caso
   * @param locale O Locale
   * @param tz O Timezone
   * @return A sesso do usurio ou null caso o usurio no seja validado no
   *         servidor de referncia ou ocorra algum erro durante o login.
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public Session login(ServerURI referedServerURI,
    Map<String, Serializable> attr, boolean copyServerSessionAttrs,
    Object sessionkey, String login, String delegatedLogin, Locale locale,
    TimeZone tz) throws RemoteException;

  /**
   * Faz o logout de um usurio do sistema.
   *
   * @param sessionKey A chave de sesso do usurio que est efetuando o logout.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public void logout(Object sessionKey) throws RemoteException;

  /**
   * Obtm os usurios atualmente conectados ao sistema.
   *
   * @return usurios atualmente conectados ao sistema.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public UserOutline[] getLoggedUsers() throws RemoteException;

  /**
   * Indica se o mtodo de autenticao do servidor  local ou no.
   *
   * @return <code>true</code> se os usurios forem autenticados localmente,
   *         <code>false</code> caso contrrio.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public boolean isLocalLogin() throws RemoteException;

  /**
   * Informa se o servidor aceite troca de senhas. A troca pode no ser aceita
   * quando o mtodo de autenticao utilizado pelo servidor acesse um servidor
   * externo. Atualmente, este  o caso da autenticao via LDAP.
   *
   * @return verdadeiro caso o servidor aceite trocar senhas, falso caso
   *         contrrio.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public boolean canChangePasswords() throws RemoteException;

  /**
   * Obtem as referncias para os servios especificados. O mapa de servios tem
   * como chave os nomes dos servios conforme as constantes
   * <code>XxxxServiceInterface.SERVICE_NAME</code>.
   *
   * @param sessionKey Chave da sesso de um usurio.
   * @param servicesNames Nome dos servios.
   *
   * @return Mapa com os servios disponveis. <code>null</code> se a sesso for
   *         invlida.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public Map<String, ServiceInterface> fetchServices(Object sessionKey,
    Set<String> servicesNames) throws RemoteException;

  /**
   * Obtem a referncia para um determinado servio disponvel
   * 
   * @param <T> Tipo que implementa ServiceInterface
   * @param sessionKey Chave da sesso de um usurio.
   * @param serviceName Nome do Servio. Conforme constante
   *        {@code XxxxServiceInterface.SERVICE_NAME}.
   *
   * @return Referncia para o servio especificado. {@code null} caso o servio
   *         solicitado no esteja disponvel ou a sesso for invlida.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public <T extends ServiceInterface> T fetchService(Object sessionKey,
    String serviceName) throws RemoteException;

  /**
   * Trata o caso especial dos servio HttpService que ainda no foi segmentado
   * e migrado para o padro de proxies para servios.
   * <p>
   * <b>OBS:</b> <br>
   * <i>Esse mtodo deve ser utilizado somente em casos onde  necessrio obter
   * uma referncia para esse servio sem estar logado. Servlets por
   * exemplo.</i>
   * </p>
   *
   * @return O servio de Http.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public HttpServiceInterface fetchHttpService() throws RemoteException;

  /**
   * Define o nome do sistema que est sendo usado como cliente.
   *
   * @param sessionKey A chave de sesso de um usurio.
   * @param systemName O nome do sistema cliente.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public void setSystemName(Object sessionKey, String systemName)
    throws RemoteException;

  /**
   * Consulta o nome do sistema.
   *
   * @return o nome
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String getSystemName() throws RemoteException;

  /**
   * Consulta nome do charset default do servidor
   *
   * @return o nome do charset
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String getSystemDefaultCharsetName() throws RemoteException;

  /**
   * Obtm a URL de um determinado sistema, acrescido do parmetro da porta RMI
   * <code>(server_port)</code>.
   *
   * @return a URL de acesso para determinado sistema.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String getSystemURLWithRMIPort() throws RemoteException;

  /**
   * Obtm a URL de um determinado sistema, acrescido do parmetro da porta RMI
   * <code>(server_port)</code> e o parmetro da pgina <code>(page)</code>.
   *
   * @param systemName o nome do sistema que se deseja obter a URL. Este nome 
   *        usado para definir o parmetro <code>page</code> da URL, que ser
   *        acrescido da extenso do tipo de aplicativo web. (i.e.: ".jnlp")
   *
   * @return a URL de acesso para determinada aplicao.
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String getSystemURLWithRMIPort(String systemName)
    throws RemoteException;

  /**
   * Define o valor de uma propriedade na sesso do usurio no servidor.
   *
   * @param sessionKey A chave da sesso do usurio.
   * @param propertyName O nome da propriedade.
   * @param propertyValue O novo valor da propriedade.
   *
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public void setServerSessionProperty(Object sessionKey, String propertyName,
    Serializable propertyValue) throws RemoteException;

  /**
   * Valida a sesso de um usurio. O servidor deve verificar a assinatura do
   * nome do servidor requisitando a validao.
   *
   * @param sessionKey A sesso a ser validada
   * @param serverName O nome do servidor requisitando a validao
   * @param signedServerName O nome assinado do servidor requisitando a
   *        validao
   * @return O mapa de atributos do servidor se a sesso  valida e o servidor
   *         que requisitou a validao  confivel, null caso contrrio.
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public Map<String, Serializable> isValidSession(Object sessionKey,
    String serverName, byte[] signedServerName) throws RemoteException;

  /**
   * Retorna a chave pblica para criptografar a senha do login.
   *
   * @return chave pblica para criptografia da senha do login.
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public PublicKey getPublicKey() throws RemoteException;

  /**
   * Retorna os atributos da sesso para um determinado login
   *
   * @param sessionKey login do usurio para o qual se quer os atributos de
   *        sesso
   * @return atributos da sesso para um determinado login
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public Map<String, Serializable> getSessionAttributes(Object sessionKey)
    throws RemoteException;

  /**
   * Gera uma URL para carregar um novo desktop a partir de um token de
   * autenticao, locale e nome de sistema (pgina html alternativa)
   *
   * @param token token de autenticao para um novo usurio
   * @param locale locale utilizado
   * @param systemName nome do sistema (pgina html alternativa a ser carregada)
   * @return URL para carregar novo desktop
   * @throws RemoteException Caso ocorra algum problema na comunicao com o
   *         servidor.
   */
  public String generateSystemURL(String token, Locale locale,
    String systemName) throws RemoteException;
}
