/*
 * $Id$
 */
package csbase.logic.algorithms.xml.algorithmspack;

import java.io.IOException;
import java.io.Writer;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;

import tecgraf.javautils.xml.ImprovedXMLListIterator;
import tecgraf.javautils.xml.XMLBasicElement;
import tecgraf.javautils.xml.XMLElementInterface;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * Esta classe implementa o elemento XML que representa um algoritmo.
 * 
 */
public class XmlAlgorithmElement extends XMLBasicElement {

  /** Mantm o algoritmo corrente que est sendo escrito */
  private AlgorithmInfo currentAlgo;

  /**
   * Constri o elemento xml que descreve um algoritmo.
   */
  public XmlAlgorithmElement() {
    super();
    setTag(XmlAlgorithmsPackUtils.ALGORITHM_TAG);
  }

  /**
   * Construtor utilizado no processo de escrita do arquivo xml de algoritmos.
   * 
   * @param algoInfo algoritmo a ser escrito
   */
  public XmlAlgorithmElement(AlgorithmInfo algoInfo) {
    this();
    currentAlgo = algoInfo;
  }

  /**
   * Executado quando a tag de fim do elemento XML  encontrada. Deve-se criar o
   * objeto correspondente na camada da aplicao.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#endTag(java.util.List)
   */
  @Override
  public void endTag(List<XMLElementInterface> childrenList) {
    String id = null, name = null, desc = null;
    Hashtable<String, String> propertiesValues = null;
    Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> versions = null;
    List<String> categoryFullNames = null;

    ImprovedXMLListIterator iterator =
      new ImprovedXMLListIterator(childrenList);
    XMLElementInterface nextElem = iterator.next(ImprovedXMLListIterator.ANY);
    while (nextElem != null) {
      //L o id do algoritmo
      if (isIdElement(nextElem)) {
        id = nextElem.getStrValue();
      }
      //L o nome do algoritmo
      if (isNameElement(nextElem)) {
        name = nextElem.getStrValue();
      }

      //L a descrio do algoritmo
      if (isDescElement(nextElem)) {
        desc = nextElem.getStrValue();
      }

      //L as propriedades do algoritmo
      if (isAlgoPropertiesElement(nextElem)) {
        propertiesValues =
          ((XMLPropertySetElement) nextElem).getPropertyValues();
      }

      //L as verses do algoritmo
      if (isAlgoVersionsElement(nextElem)) {
        versions = ((XMLVersionSetElement) nextElem).getVersionsValues();
      }

      //L as categorias do algoritmo
      if (isAlgoCategoriesElement(nextElem)) {
        categoryFullNames =
          ((XMLCategorySetElement) nextElem).getCategoryFullNameSet();
      }

      nextElem = iterator.next(ImprovedXMLListIterator.ANY);
    }

    currentAlgo =
      createAlgorithmInfo(id, name, null, null, propertiesValues, versions);
    currentAlgo.setDescription(desc);
    currentAlgo.setAlgoPackCategoryFullNames(categoryFullNames);
    //    printAlgoData(currentAlgo);
  }

  /**
   * Imprime os dados do algoritmo criado a partir da leitura do xml.
   * 
   * @param algo algoritmo criado
   */
  protected void printAlgoData(AlgorithmInfo algo) {
    System.out.println("\nDados do Algoritmo criado a partir do xml: ");
    System.out.println("Id: " + algo.getId());
    System.out.println("Nome: " + algo.getName());
    System.out.println("Descrio: " + algo.getDescription());

    //Propriedades do algoritmo
    Hashtable<String, String> propertyValues = algo.getPropertyValues();
    Set<Entry<String, String>> entrySet = propertyValues.entrySet();
    System.out.println("\n>> Propriedades do algoritmo: " + entrySet.size());
    for (Entry<String, String> entry : entrySet) {
      System.out.println(entry.getKey() + " = " + entry.getValue());
    }

    //Verses do algoritmo
    Vector<AlgorithmVersionInfo> versions = algo.getVersions();
    System.out.println("\n>> Verses do algoritmo: " + versions.size());
    for (AlgorithmVersionInfo algoVersionInfo : versions) {
      System.out.println("\n-> Verso: " + algoVersionInfo);
      System.out.println("descrio: " + algoVersionInfo.getDescription());
      Map<String, String> properties = algoVersionInfo.getPropertyValues();
      entrySet.clear();
      entrySet = properties.entrySet();
      System.out.println("\nPropriedades da verso do algoritmo: "
        + entrySet.size());
      for (Entry<String, String> entry : entrySet) {
        System.out.println(entry.getKey() + " = " + entry.getValue());
      }
    }

    //Categorias do algoritmo
    List<String> categoryFullNames = algo.getAlgoPackCategoryFullNames();
    System.out.println("\n>> Categorias do algoritmo: "
      + categoryFullNames.size());
    for (String categoryFullName : categoryFullNames) {
      System.out.println("\n-> Nome completo: " + categoryFullName);
    }

  }

  /**
   * Cria as informaes do algoritmo a partir dos dados especificados.
   * 
   * @param id identificador do algoritmo
   * @param name nome do algoritmo
   * @param dir diretrio do algoritmo
   * @param algorithmRepositoryPath caminho do repositrio de algoritmos
   * @param propertyValues valores das propriedades do algoritmo
   * @param versions verses do algoritmo
   * 
   * @return retorna o algoritmo com suas informaes
   */
  private AlgorithmInfo createAlgorithmInfo(String id, String name, String dir,
    String algorithmRepositoryPath, Hashtable<String, String> propertyValues,
    Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> versions) {
    return new AlgorithmInfo(id, name, dir, algorithmRepositoryPath,
      propertyValues, versions);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(Writer arg0, String arg1) throws IOException {
    // TODO Auto-generated method stub

  }

  /**
   * Obtm o objeto da aplicao algoritmo que foi criado a partir desse
   * elemento xml.
   * 
   * @return o algoritmo correspondente ao elemento xml
   */
  public AlgorithmInfo getAlgorithmInfo() {
    return currentAlgo;
  }

  /**
   * Verifica se  o elemento que representa o identificador do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento  o identificador do algoritmo, caso
   *         contrrio, retorna false
   */
  private boolean isIdElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.ALGORITHM_ID_TAG);
  }

  /**
   * Verifica se  o elemento que representa o nome do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento  o nome do algoritmo, caso contrrio,
   *         retorna false
   */
  private boolean isNameElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.ALGORITHM_NAME_TAG);
  }

  /**
   * Verifica se  o elemento que representa a descrio do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento  a descrio do algoritmo, caso
   *         contrrio, retorna false
   */
  private boolean isDescElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.DESCRIPTION_TAG);
  }

  /**
   * Verifica se  o elemento que representa as propriedades do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento representa as propriedades do algoritmo,
   *         caso contrrio, retorna false
   */
  private boolean isAlgoPropertiesElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.PROPERTIES_TAG);
  }

  /**
   * Verifica se  o elemento que representa as verses do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento representa as verses do algoritmo, caso
   *         contrrio, retorna false
   */
  private boolean isAlgoVersionsElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.VERSIONS_TAG);
  }

  /**
   * Verifica se  o elemento que representa as categorias do algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true se o elemento representa as categorias do algoritmo,
   *         caso contrrio, retorna false
   */
  private boolean isAlgoCategoriesElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.CATEGORIES_TAG);
  }

}
