/*
 * $Id$
 */

package csbase.logic;

import java.io.File;
import java.io.FilenameFilter;
import java.io.Serializable;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.util.FileSystemUtils;

/**
 * @author Tecgraf / PUC-Rio
 * 
 * Classe usada para representar um arquivo.<br>
 * Ela contm as informaes bsicas de um arquivo como nome, pai, filhos e se 
 * um diretrio e pode ser serializada, permitindo passar essas informaes
 * entre cliente e servidor.<br>
 */
public class FileInfo implements Serializable, Comparable<FileInfo> {

  // Filtro de nomes vlidos.
  private static final FilenameFilter NAME_FILTER;

  // Representa o pai do arquivo.
  private FileInfo parent;
  // Nome do arquivo.
  private final String name;
  // Representa os filhos do arquivo.
  private final List<FileInfo> children;
  /*
   * Verso inaltervel da lista de filhos. Esta lista  utilizada para ser
   * repassada ao usurio.
   */
  private final List<FileInfo> unmodifiableChildren;
  // Indica se o arquivo  um diretrio.
  private final boolean isDirectory;

  static {
    NAME_FILTER = new FilenameFilter() {
      @Override
      public boolean accept(File dir, String name) {
        return isNameValid(name);
      }
    };
  }

  /**
   * Construtor.
   * 
   * @param path caminho de diretrios at o arquivo sendo representado.
   * @param name Nome do arquivo que esta instncia ir representar.
   * @param isDirectory <tt>true</tt> se o arquivo representado por esta
   *        instncia  um diretrio.
   * 
   * @throws IllegalArgumentException se o nome do arquivo ou de algum de seus
   *         diretrios presente no path for invlido.
   * @see #isNameValid(String)
   */
  public FileInfo(String[] path, String name, boolean isDirectory) {
    this(name, isDirectory);

    FileInfo child = this;

    for (int inx = path.length - 1; inx >= 0; inx--) {
      String parentName = path[inx];
      FileInfo parent = new FileInfo(parentName, true);
      child.setParent(parent);
      child = parent;
    }
  }

  /**
   * Construtor.
   * 
   * 
   * @param file arquivo a ser representado por esta instncia.
   * 
   * @throws IllegalArgumentException se o nome do arquivo for invlido.
   * @see #isNameValid(String)
   */
  public FileInfo(File file) {
    this(file.getName(), file.isDirectory());

    List<FileInfo> executables = createFilesInfo(file);
    for (FileInfo executable : executables) {
      addChild(executable);
    }
    Collections.sort(children);
  }

  /**
   * Construtor.
   * 
   * @param name Nome do arquivo que esta instncia ir representar.
   * @param isDirectory <tt>true</tt> se o arquivo representado por esta
   *        instncia  um diretrio.
   * 
   * @throws IllegalArgumentException se o nome do arquivo for invlido.
   * @see #isNameValid(String)
   */
  public FileInfo(String name, boolean isDirectory) {

    if (!isNameValid(name))
      throw new IllegalArgumentException("Invalid name " + name);

    this.name = name.trim();
    this.isDirectory = isDirectory;

    this.parent = null;
    if (isDirectory) {
      this.children = new ArrayList<FileInfo>();
    }
    else {
      this.children = Collections.emptyList();
    }
    unmodifiableChildren = Collections.unmodifiableList(children);
  }

  /**
   * <p>
   * Cria uma lista de {@link FileInfo} a partir dos arquivos presentes no
   * diretrio passado como argumento.
   * </p>
   * <p>
   * O diretrio <b>no</b> ser levado em considerao e assim no ser criado
   * um {@link FileInfo} representando ele como pai dos demais. Caso este seja o seu
   * objetivo, basta criar um FileInfo do diretrio a partir do
   * {@link #FileInfo(File)} e pedir a lista de filhos dele atravs do mtodo
   * {@link #getChildren()}.
   * </p>
   * <p>
   * Obs. Arquivos com nome invlido sero ignorados.
   * </p>
   * 
   * @param dir diretrio que detm os arquivos a serem representados como
   *        instncias de {@link FileInfo}
   * 
   * @return uma lista de {@link FileInfo} a partir dos arquivos presentes no
   *         diretrio passado como argumento.
   * 
   * @see #isNameValid(String)
   */
  public static List<FileInfo> createFilesInfo(File dir) {
    if (dir.isDirectory()) {
      File[] childrenFile = dir.listFiles(NAME_FILTER);
      if (0 < childrenFile.length) {
        List<FileInfo> executables = new ArrayList<FileInfo>();
        for (File childFile : childrenFile) {
          FileInfo executable = new FileInfo(childFile);
          executables.add(executable);
        }
        return executables;
      }
    }

    return Collections.emptyList();
  }

  /**
   * Checa se o nome do arquivo  vlido.<br>
   * O nome do arquivo NO deve:
   * <ol>
   * <li>Ser nulo.</li>
   * <li>Ser igual ao nome do
   * {@link FileSystemUtils#VERSION_CONTROL_DIR diretrio do controle de verso}.</li>
   * </ol>
   * 
   * @param name nome do arquivo a ser checado.
   * 
   * @return <tt>true</tt> se o nome for vlido para criar um {@link FileInfo}.
   */
  public static boolean isNameValid(String name) {
    return null != name
      && !name.trim().equalsIgnoreCase(FileSystemUtils.VERSION_CONTROL_DIR);
  }

  /**
   * Obtm o nome do arquivo representado por esta instncia.
   * 
   * @return o nome do arquivo representado por esta instncia.
   */
  public String getName() {
    return name;
  }

  /**
   * Obtm uma instncia de {@link FileInfo} representando o diretrio em que o
   * arquivo representado pro esta instncia se encontra.
   * 
   * @return uma instncia de {@link FileInfo} representando o diretrio aonde o
   *         arquivo se encontra.
   */
  public FileInfo getParent() {
    return parent;
  }

  /**
   * Obtm uma lista de {@link FileInfo} representando os arquivos presentes no
   * diretrio representado por esta instncia. Caso esta instncia no
   * represente um diretrio uma lista vazia ser retornada.<br>
   * Obs. A lista retornada no pode ser modificada.
   * 
   * @return uma lista de {@link FileInfo} representando os arquivos presentes
   *         no diretrio representado por esta instncia. Caso esta instncia
   *         no represente um diretrio uma lista vazia ser retornada.<br>
   *         Obs. A lista retornada no pode ser modificada.
   */
  public List<FileInfo> getChildren() {
    return unmodifiableChildren;
  }

  /**
   * Adiciona um {@link FileInfo} representando um filho do arquivo e se pe
   * como pai daquela instncia.
   * 
   * @param child representao do filho do arquivo.
   * 
   * @return <tt>true</tt> se este representa um diretrio e se o filho foi
   *         inserido com sucesso.
   * 
   * @see #setParent(FileInfo)
   */
  public boolean addChild(FileInfo child) {
    if (isDirectory) {
      if (children.add(child)) {
        child.setParent(this);
        return true;
      }
    }
    return false;
  }

  /**
   * Indica se esta instncia representa um diretrio.
   * 
   * @return <tt>true</tt> se esta instncia representar um diretrio.
   */
  public boolean isDirectory() {
    return isDirectory;
  }

  /**
   * Obtm o caminho relativo at este executvel.
   * 
   * @return o caminho relativo at este executvel.
   */
  public String getPath() {
    return IPathFactory.DEFAULT.getPath(this);
  }

  /**
   * Obtm o caminho relativo at este executvel.
   * 
   * @return o caminho relativo at este executvel.
   */
  public String[] getPathAsArray() {
    List<String> path = _getPath();
    return (String[]) path.toArray(new String[path.size()]);
  }

  /**
   * Obtm o caminho relativo at este executvel.<br>
   * Este mtodo  privado e  utilizado pelo {@link #getPath()} para manter o
   * padro do cdigo em que um caminho  apresentado sempre como uma
   * {@link String} ou um array de {@link String}.
   * 
   * @return o caminho relativo at este executvel.
   */
  private List<String> _getPath() {
    List<String> path;
    if (null == getParent()) {
      path = new ArrayList<String>();
    }
    else {
      path = getParent()._getPath();
    }
    path.add(getName());
    return path;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + (isDirectory ? 1231 : 1237);
    result = prime * result + ((name == null) ? 0 : name.hashCode());
    result = prime * result + ((parent == null) ? 0 : parent.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    final FileInfo other = (FileInfo) obj;
    if (isDirectory != other.isDirectory)
      return false;
    if (name == null) {
      if (other.name != null)
        return false;
    }
    else if (!name.equals(other.name))
      return false;
    if (parent == null) {
      if (other.parent != null)
        return false;
    }
    else if (!parent.equals(other.parent))
      return false;
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(FileInfo o) {
    if (this.isDirectory() && !o.isDirectory()) {
      return -1;
    }
    else if (!this.isDirectory() && o.isDirectory()) {
      return 1;
    }
    else {
      return Collator.getInstance().compare(this.getName(), o.getName());
    }
  }

  /**
   * Obtm a representao desta instncia na forma de uma {@link String}.<br>
   * Essa instncia  representada como uma {@link String}, atravs de seu
   * nome.
   * 
   * @return a representao desta instncia na forma de uma {@link String}.
   */
  @Override
  public String toString() {
    return getName();
  }

  /**
   * Atribui um pai a esta instncia. Se ela j tinha um pai, o pai anterior a
   * perde como filha.
   * 
   * @param parent Novo pai.
   */
  private void setParent(FileInfo parent) {
    if (null != this.parent) {
      this.parent.children.remove(this);
    }
    this.parent = parent;
  }
}
