/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.validators.IntegerParameterValidator;
import csbase.logic.algorithms.parameters.validators.ListParameterValidator;

/**
 * Parmetro do Tipo Lista de Inteiros.
 *
 * @author lmoreira
 */
public final class IntegerListParameter extends ListParameter<Integer> {

  /** Texto que representa o tipo desse parmetro */
  public static final String TYPE = "LIST_OF_INTEGERS";

  /**
   * Os observadores.
   */
  private transient List<IntegerListParameterListener> listeners;

  /**
   * O valor mximo considerado vlido.
   */
  private Integer maximum;
  /**
   * O valor mnimo considerado vlido.
   */
  private Integer minimum;

  /**
   * Cria um parmetro.
   *
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se  o valor  opcional/obrigatrio.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param maximum O mximo (Aceita {@code null}; se o mnimo existir ele tem
   *        que ser maior ou igual a ele).
   * @param minimum O mnimo (Aceita {@code null}; se o mximo existir ele tem
   *        que ser menor ou igual a ele).
   */
  public IntegerListParameter(String name, String label, String description,
    List<Integer> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, Integer maximum, Integer minimum) {
    super(name, label, description, defaultValue, isOptional, isVisible, false,
      true, commandLinePattern);
    this.listeners = new LinkedList<IntegerListParameterListener>();
    setMaximum(maximum);
    setMinimum(minimum);
  }

  /**
   * Adiciona um observador a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public void addIntegerListParameterListener(
    IntegerListParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Obtm o valor mximo.
   *
   * @return O valor mximo ou {@code null} se ele no existir.
   */
  public Integer getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o valor mnimo.
   *
   * @return O valor mnimo ou {@code null} se ele no existir.
   */
  public Integer getMinimum() {
    return this.minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * Atribui o mximo a este parmetro.
   *
   * @param maximum O mximo (Aceita {@code null}; se o mnimo existir ele tem
   *        que ser maior ou igual a ele).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o valor mximo
   *         atual for igual ao valor mximo fornecido.
   */
  public boolean setMaximum(Integer maximum) {
    if (this.minimum != null) {
      if (maximum != null && maximum.compareTo(this.minimum) < 0) {
        throw new IllegalArgumentException(
          "O valor mximo no pode ser menor do que o valor mnimo.");
      }
    }
    if ((this.maximum == null) && (maximum == null)) {
      return false;
    }
    if ((this.maximum != null) && (this.maximum.equals(maximum))) {
      return false;
    }
    this.maximum = maximum;
    updateValidator();

    for (IntegerListParameterListener listener : this.listeners) {
      listener.maximumWasChanged(this);
    }
    return true;
  }

  /**
   * Atualiza os valores utilizados para validar o parmetro.
   */
  private void updateValidator() {
    ListParameterValidator<Integer> listParameterValidator =
      (ListParameterValidator<Integer>) getParameterValidator();
    IntegerParameterValidator validator =
      (IntegerParameterValidator) listParameterValidator.getItemValidator();
    validator.setMaximum(this.maximum);
    validator.setMinimum(this.minimum);
  }

  /**
   * Atribui o mnimo a este parmetro.
   *
   * @param minimum O mnimo (Aceita {@code null}; se o mximo existir ele tem
   *        que ser menor ou igual a ele).
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o valor mnimo
   *         atual for igual ao valor mnimo fornecido.
   */
  public boolean setMinimum(Integer minimum) {
    if (this.maximum != null) {
      if (minimum != null && minimum.compareTo(this.maximum) > 0) {
        throw new IllegalArgumentException(
          "O valor mnimo no pode ser maior do que o valor mximo.");
      }
    }
    if ((this.minimum == null) && (minimum == null)) {
      return false;
    }
    if ((this.minimum != null) && (this.minimum.equals(minimum))) {
      return false;
    }
    this.minimum = minimum;
    updateValidator();

    for (IntegerListParameterListener listener : this.listeners) {
      listener.minimumWasChanged(this);
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Integer getItemValueFromText(String itemValue)
    throws ParseException {
    if (itemValue == null) {
      return null;
    }
    try {
      int intValue = Integer.parseInt(itemValue);
      return new Integer(intValue);
    }
    catch (NumberFormatException e) {
      throw new ParseException(e,
        "O valor {0} foi atribuido ao parmetro {1}, "
          + "porm ele no  um valor vlido para um nmero inteiro.",
        itemValue, this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected IntegerParameterValidator createItemValidator() {
    return new IntegerParameterValidator(isOptional(), maximum, minimum);
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in Leitor de objetos
   *
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<IntegerListParameterListener>();
  }
}
