package csbase.logic.algorithms.parameters;

import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.FileParameterValue;

/**
 * <p>
 * Parmetro do tipo lista de arquivo de sada.
 * </p>
 * <p>
 * Ele possui 3 modos de funcionamento: aceitar apenas arquivos regulares,
 * aceitar apenas diretrios ou aceitar arquivos regulares e diretrios (veja
 * {@link FileParameterMode}).
 * </p>
 */
public class OutputFileListParameter extends FileListParameter {

  /** Texto que representa o tipo desse parmetro */
  public static final String TYPE = "OUTPUT_FILE";

  /** Indica se o arquivo deve ter uma das extenses pr-definidas. */
  private boolean mustForceExtension;

  /**
   * Cria um parmetro do tipo arquivo.
   *
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileType O tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}).
   * @param mustSort Indica se os caminhos para os arquivos precisam ser
   *        ordenados.
   * @param mustForceExtension Indica se o parmetro deve forar o uso da
   *        extenso no arquivo (Se ele no tiver, ele colocar a extenso).
   * @param usesFilter Indica se um painel de filtro pode ser usado.
   */
  public OutputFileListParameter(String name, String label, String description,
    List<FileParameterValue> defaultValue, boolean isOptional,
    boolean isVisible, String commandLinePattern, String fileType,
    FileParameterMode mode, boolean mustSort, boolean usesFilter,
    boolean mustForceExtension) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern, fileType, mode, mustSort, usesFilter, false);
    this.mustForceExtension = mustForceExtension;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE;
  }

  /**
   * Indica se deve forar a extenso do arquivo.
   *
   * @return .
   */
  public boolean mustForceExtension() {
    return this.mustForceExtension;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setValue(List<FileParameterValue> files) {
    checkExtension(files);
    return super.setValue(files);
  }

  /**
   * Verifica se deve forar a extenso do arquivo e a coloca caso necessrio.
   *
   * @param files lista de arquivos de sada
   */
  private void checkExtension(List<FileParameterValue> files) {
    if ((files != null) && mustForceExtension()) {
      for (FileParameterValue file : files) {
        String filePath = file.getPath();
        String fileExtension = FileUtils.getFileExtension(filePath);
        /* Obtm a lista de possveis extenses para o arquivo de sada */
        String type = this.getFileType();
        ProjectFileType projectFileType = ProjectFileType.getFileType(type);
        if (projectFileType == null) {
          return;
        }
        List<String> extensions = projectFileType.getExtensions();
        if (extensions.isEmpty()) {
          return;
        }
        /*
         * Se arquivo no tem extenso ou a extenso no confere com o tipo do
         * arquivo, fora a extenso
         */
        if ((fileExtension == null) || !extensions.contains(fileExtension)) {
          filePath += '.' + extensions.get(0);
          /* Renomeia o arquivo. */
          file.setPath(filePath);
        }
      }
    }
  }

}
