package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parameters.TextColumn;
import csbase.logic.algorithms.parameters.TextListParameter;
import csbase.logic.algorithms.parameters.TextParameter;

/**
 * Classe abstrata que serve de base para analisadores de parmetros <T> com
 * valores do tipo texto como {@link TextParameter} e {@link TextListParameter}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractTextParameterParser<T extends SimpleParameter<?>>
  extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #TEXT_ELEMENT_MAXIMUM_CHARACTERS_ATTRIBUTE} dos
   * elementos:
   * <ul>
   * <li>{@link TextColumn} de um {@link TableParameter}</li>
   * <li>{@link TextParameter}</li>
   * <li>{@link TextListParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Informa a quantidade mxima de caracteres que  considerada vlida para os
   * valores da {@link TextColumn coluna do tipo texto} e dos parmetros:
   * {@link TextParameter parmetro do tipo texto} e {@link TextListParameter
   * parmetro do tipo lista de texto}.  opcional,  do tipo inteiro e o seu
   * valor mnimo  {@link #TEXT_ELEMENT_MAXIMUM_CHARACTERS_MINIMUM_VALUE}.
   * </p>
   */
  static final String TEXT_ELEMENT_MAXIMUM_CHARACTERS_ATTRIBUTE =
    "maximo_de_caracteres";

  /**
   * <p>
   * O valor mnimo para o atributo
   * {@link #TEXT_ELEMENT_MAXIMUM_CHARACTERS_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link TextColumn} de um {@link TableParameter}</li>
   * <li>{@link TextParameter}</li>
   * <li>{@link TextListParameter}</li>
   * </ul>
   * </p>
   * <p>
   * O seu valor  {@value #TEXT_ELEMENT_MAXIMUM_CHARACTERS_MINIMUM_VALUE}.
   * </p>
   */
  static final int TEXT_ELEMENT_MAXIMUM_CHARACTERS_MINIMUM_VALUE = 1;

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser, String name, String label,
    String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group) throws ParseException {
    Integer maxCharacters =
      parser.extractAttributeValueAsInteger(
        TEXT_ELEMENT_MAXIMUM_CHARACTERS_ATTRIBUTE, null, null,
        TEXT_ELEMENT_MAXIMUM_CHARACTERS_MINIMUM_VALUE);
    return createTextParameter(parser, name, label, description, isOptional,
      isVisible, commandLinePattern, maxCharacters);
  }

  /**
   * Cria uma instncia do parmetro de tipo <T>, a partir dos atributos bsicos
   * de um parmetro do tipo texto. As subclasses devem implementar esse mtodo,
   * fazendo a extrao dos demais atributos do parmetro.
   * 
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param maxCharacters Nmero mximo de caracteres que o valor do parmetro
   *        pode ter.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  protected abstract T createTextParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, Integer maxCharacters) throws ParseException;

}
