package csbase.rest.adapter.algorithm.v1;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import org.json.JSONArray;

import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;
import ibase.common.ServiceUtil;
import ibase.rest.model.algorithm.v1.Requirement;

/**
 * Created by mjulia on 14/07/16.
 */
public class RequirementsBuilder {

    /** Language bundle */
    public static final String RESOURCE_BUNDLE = "RequirementsBuilder";

    /** Lista de requisitos */
    public final List<Requirement> requirements;

    /** Locale para traduo dos nomes dos requisitos padronizados */
    private Locale locale;

    /** Chave para os requisitos padronizado: CPU */
    private static final String CPU_KEY = "cpu";

    /** Chave para os requisitos padronizado: MEMORY */
    private static final String MEMORY_KEY = "memory";

    /** Chave para os requisitos padronizado: PLATFORMS */
    private static final String PLATFORMS_KEY = "platforms";

    /** Chave para os requisitos padronizado: REQUIREMENTS */
    private static final String REQUIREMENTS_KEY = "requirements";

    private ResourceBundle getBundle(Locale locale) {
        ResourceBundle bundle = ResourceBundle.getBundle(RESOURCE_BUNDLE, locale, this.getClass().getClassLoader());
        return bundle;
    }

    public RequirementsBuilder(String algorithmId, AlgorithmVersionId algoVersionId, Locale currentLocale) {
        this.locale = currentLocale;
        requirements = new ArrayList<>();
        AlgorithmServiceInterface service = ClientRemoteLocator.algorithmService;
        try {
            AlgorithmInfo algoInfo = service.getInfo((Object) algorithmId);
            Map<String, String> propertyValues = algoInfo.getPropertyValues();
            if (propertyValues!=null) {
                List<Requirement> requirements = new ArrayList<>();
                propertyValues.forEach((k,v)->loadRequirement(k,v));
            }
            AlgorithmVersionInfo versionInfo = algoInfo.getVersionInfo(algoVersionId);
            List<String> supportedPlatforms = versionInfo.getSupportedPlatforms();
            if (supportedPlatforms!=null) {
                JSONArray plats = new JSONArray();
                supportedPlatforms.stream().forEach(p -> plats.put(p));
                loadRequirement("platforms", plats.toString());
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void loadRequirement(String key, String value) {
        String name = ServiceUtil.getTranslator(getBundle(locale)).message(key);
        switch (key) {
            case CPU_KEY :
            case MEMORY_KEY:
                Double.parseDouble(value);
                requirements.add(buildRequirement(key, name, Double.parseDouble(value), Requirement.TypeEnum.DOUBLE, false));
                break;
            case REQUIREMENTS_KEY:
                Arrays.stream(value.split(",")).forEach(v -> requirements.add(buildRequirement(v, v, Boolean.valueOf(true), Requirement.TypeEnum.BOOLEAN, false)));
                break;
            case PLATFORMS_KEY:
                requirements.add(buildRequirement(key, name, value, Requirement.TypeEnum.ENUMERATION, true));
                break;
        }
    }

    private Requirement buildRequirement (String key, String name, Object value, Requirement.TypeEnum type, boolean editable) {
        Requirement req = new Requirement();
        req.setKey(key);
        req.setName(name);
        req.setValue(value);
        req.setType(type);
        req.setEditable(editable);
        return req;
    }
}
