package csbase.rest.adapter.project.v1;

import java.util.HashSet;
import java.util.Set;

import csbase.logic.CommonClientProject;
import csbase.logic.ProjectPermissions;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import ibase.exception.InternalServiceException;
import ibase.rest.api.project.v1.adapter.AccessType;
import ibase.rest.api.project.v1.adapter.Project;
import ibase.rest.api.project.v1.adapter.SharingType;

/**
 * Implementação do Gateway para um projeto no CSBase.
 *
 * @author Tecgraf/PUC-Rio
 */
public class CSBaseProject implements Project {

  /** A chave do atributo com o tipo do projeto */
  public static final String PROJECT_TYPE = "ibase_project_type";

  /**
   * O projeto no CSBase.
   */
  private final CommonClientProject project;

  /**
   * Construtor.
   * 
   * @param project o projeto no CSBase
   */
  CSBaseProject(CommonClientProject project) {
    this.project = project;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return (String) project.getId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return project.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getOwnerId() {
    return (String) project.getUserId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDescription() {
    return project.getDescription();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return (String) project.getAttribute(PROJECT_TYPE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getLastModificationDate() {
    return project.getLastModificationDate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SharingType getSharingType() {
    ProjectPermissions.SharingType sharingType = project.getSharingType();
    switch (sharingType) {
      case ALL_RO:
        return SharingType.PUBLIC_RO;
      case ALL_RW:
        return SharingType.PUBLIC_RW;
      case PARTIAL:
        return SharingType.SELECTIVE;
      case PRIVATE:
        return SharingType.PRIVATE;
      default:
        return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getTeamMembers(AccessType accessType) {
    Set<Object> users;
    switch (accessType) {
      case READ_ONLY:
        users = project.getUsersRO();
        break;
      case READ_WRITE:
        users = project.getUsersRW();
        break;
      default:
        return new HashSet<>();
    }
    HashSet<String> userIds = new HashSet<>(users.size());
    users.forEach(i -> userIds.add(i.toString()));
    return userIds;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addTeamMember(AccessType accessType, String userId) {
    Set<Object> membersRO = project.getUsersRO();
    Set<Object> membersRW = project.getUsersRW();
    Set<Object> updatedMembersRO = new HashSet<Object>();
    Set<Object> updatedMembersRW = new HashSet<Object>();
    updatedMembersRO.addAll(membersRO);
    updatedMembersRW.addAll(membersRW);
    if (!membersRO.contains(userId) && !membersRW.contains(userId)) {
      // Se o usuário não está entre os membros do projeto, deve ser adicionado
      switch (accessType) {
        case READ_ONLY:
          updatedMembersRO.add(userId);
          break;
        case READ_WRITE:
          updatedMembersRW.add(userId);
          break;
      }
    }
    else {
      // Se o usuário já esté entre os membros do projeto, apenas seu nivel de acesso é alterado
      switch (accessType) {
        case READ_ONLY:
          if (membersRW.contains(userId)) {
            updatedMembersRW.remove(userId);
            updatedMembersRO.add(userId);
          }
          break;
        case READ_WRITE:
          if (membersRO.contains(userId)) {
            updatedMembersRO.remove(userId);
            updatedMembersRW.add(userId);
          }
          break;
      }
    }
    try {
      ProjectServiceInterface service = ClientRemoteLocator.projectService;
      service.updateUsers(project.getId(), ProjectPermissions.getSharingType(project.getInfo()), updatedMembersRO, updatedMembersRW);
      //project.updateUsers(updatedMembersRO, updatedMembersRW);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AccessType getUserAccessType(String userId) {
    Set<Object> membersRO = project.getUsersRO();
    Set<Object> membersRW = project.getUsersRW();
    if (membersRW.contains(userId)) {
      return AccessType.READ_WRITE;
    }
    if (membersRO.contains(userId)) {
      return AccessType.READ_ONLY;
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getCreationDate() {
    return project.getCreationDate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Boolean isWritableByUser(String userId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      return service.userCanWrite(project.getId(), userId);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeTeamMember(String userId) {
    try {
      return project.removeUser(userId);
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

}