package csbase.server.services.restservice;

import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.DispatcherType;
import javax.servlet.FilterRegistration;
import javax.ws.rs.core.UriBuilder;

import org.glassfish.grizzly.http.server.CLStaticHttpHandler;
import org.glassfish.grizzly.http.server.HttpServer;
import org.glassfish.grizzly.http.server.NetworkListener;
import org.glassfish.grizzly.servlet.ServletRegistration;
import org.glassfish.grizzly.servlet.WebappContext;
import org.glassfish.grizzly.ssl.SSLContextConfigurator;
import org.glassfish.grizzly.ssl.SSLEngineConfigurator;
import org.glassfish.grizzly.websockets.WebSocketAddOn;
import org.glassfish.grizzly.websockets.WebSocketEngine;
import org.glassfish.hk2.api.Factory;
import org.glassfish.hk2.utilities.binding.AbstractBinder;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.media.multipart.MultiPartFeature;
import org.glassfish.jersey.servlet.ServletContainer;

import csbase.exception.ParseException;
import csbase.logic.User;
import csbase.remote.RestServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.restservice.websocket.messenger.CSBaseMessenger;
import csbase.server.services.restservice.websocket.notificationcenter.CSBaseNotificationCenter;
import ibase.common.DAOFactory;
import ibase.common.ServiceAdapter;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.swagger.jaxrs.config.BeanConfig;
import io.swagger.jaxrs.listing.ApiListingResource;
import io.swagger.jaxrs.listing.SwaggerSerializers;

/**
 * Implementacao da interface RestService
 *
 * @author Tecgraf/PUC-Rio
 */
public class RestService extends Service implements RestServiceInterface {

  /** Servidor do jetty */
  private HttpServer httpServer;

  /**
   * Segredo para a criao e validao do token
   */
  private SecretKey privateKey;

  /**
   * Flag para client developer mode
   */
  private boolean clientDevMode;
  /**
   * Mapa de URLs de servidores HTTP dos clientes, indexados pelo token do
   * cliente.
   */
  private final Map<String, String> clientHttpServers;

  /**
   * @throws ServerException
   */
  protected RestService() throws ServerException {
    super(SERVICE_NAME);
    setEnabled(this.getBooleanProperty("enabled"));
    clientDevMode=this.getBooleanProperty("client.developer.mode");
    clientHttpServers = new HashMap<>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initService() throws ServerException {
    loadPrivateKey();
    String[] packages = loadPackages();
    loadDAOFactories();

    buildSwagger(packages);

    Map<Class<? extends Factory<ServiceAdapter>>, Class<?>> factories =
            loadAdapterFactories();
    CSBaseResourceConfig config = new CSBaseResourceConfig(packages);
    config.register(new AbstractBinder() {
      @Override
      protected void configure() {
        factories.forEach((k, v) -> bindFactory(k).to(v));
      }
    });
    config.register(ApiListingResource.class);
    config.register(SwaggerSerializers.class);
    config.register(CSBaseRequestFilter.class);
    config.register(CSBaseResponseFilter.class);
    config.register(MultiPartFeature.class);
    config.register(CSJsonProvider.class);
    config.register(JacksonFeature.class);
    config.register(new CSBaseRuntimeExceptionMapper());

    try {

      httpServer = createHttpServer();

      ServletContainer container = new ServletContainer(config);
      WebappContext context = new WebappContext("WebappContext", "/v1");
      ServletRegistration registration = context.addServlet("ServletContainer",
        container);
      registration.addMapping("/*");

      FilterRegistration corsFilter = context.addFilter("CorsFilter",
        new CorsServletFilter());
      corsFilter.addMappingForUrlPatterns(EnumSet.allOf(DispatcherType.class),
        true, "/*");

      // Registering Grizzly WebSocket
      httpServer.getListener("grizzly").registerAddOn(new WebSocketAddOn());
      WebSocketEngine.getEngine().register("", "/messenger", new CSBaseMessenger());
      WebSocketEngine.getEngine().register("", "/notification", new CSBaseNotificationCenter());

      httpServer.start();
      context.deploy(httpServer);

      // Inicializa o swagger
      CLStaticHttpHandler staticHttpHandler = new CLStaticHttpHandler(
        CSBaseResourceConfig.class.getClassLoader(), "swagger-ui/", "docs/");
      httpServer.getServerConfiguration().addHttpHandler(staticHttpHandler,
        "/docs");
    }
    catch (Exception e) {
      throw new ServerException(e);
    }
  }

    private HttpServer createHttpServer() throws Exception {
      boolean secure = getBooleanProperty("SSL.enable");
      URI url = getBaseURI(secure, getIntProperty("service.port"));
      final NetworkListener listener = new NetworkListener("grizzly",
              url.getHost(), url.getPort());
      HttpServer server = HttpServer.createSimpleServer(url.getScheme() + url.getHost(), url.getPort());
      if (secure) {
        this.logInfoMessage("--- Servidor HTTP com SSL habilitado");
        listener.setSecure(secure);
        SSLEngineConfigurator sslEngineConfigurator = createSSLConfig();
        listener.setSSLEngineConfig(sslEngineConfigurator);
      }
      server.addListener(listener);
      return server;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unregisterClientHttpServer(String token) {
    clientHttpServers.remove(token);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isClientDeveloperMode() {
    return clientDevMode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void registerClientHttpServer(String token, String httpServerURL) {
    clientHttpServers.put(token, httpServerURL);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getClientHttpServer(String token) {
     return clientHttpServers.get(token);
  }

  /**
   * Retorna a URL bsica do servico.
   *
   * @param port Porta que o servico esta rodando
   *
   * @return URL do Servico.
   */
  private static URI getBaseURI(boolean secure, int port) {
    String protocol = secure ? "https" : "http";
    return UriBuilder.fromUri(protocol + "://0.0.0.0").port(port).path("/").build();
  }

  /**
   * Cria uma configurao SSL sem autenticao do cliente.
   * @return
   * @throws Exception
     */
  private SSLEngineConfigurator createSSLConfig()
          throws Exception {
    String keystore = getStringProperty("SSL.keystore");
    String keystorePass = getStringProperty("SSL.keystore_password");
    final SSLContextConfigurator sslContextConfigurator = new SSLContextConfigurator();
      sslContextConfigurator.setKeyStoreFile(keystore);
      sslContextConfigurator.setKeyStorePass(keystorePass);
    SSLEngineConfigurator result = new SSLEngineConfigurator(
            sslContextConfigurator.createSSLContext(), false, false, false);
    return result;
  }

  /**
   * Carrega a chave usada no token do servio REST.
   * 
   * @throws ServerException
   */
  private void loadPrivateKey() throws ServerException {
    String privateKeyFilePath = getStringProperty("private.key.file");
    try {
      this.privateKey = readKeyFromFile(privateKeyFilePath);
    }
    catch (Exception e) {
      throw new ServerException(e);
    }
  }

  /**
   * Carrega os pacotes com os servios REST das APIs
   * 
   * @return a lista dos pacotes
   * @throws ServerException
   */
  private String[] loadPackages() throws ServerException {
    List<String> classList = getStringListProperty("resources.service.class");
    //classList.forEach(System.out::println);
    List<String> packages = new ArrayList<String>();
    for (String c : classList) {
      try {
        Class<?> cl = Class.forName(c);
        packages.add(cl.getPackage().getName());
        Logger.getLogger(cl.getSimpleName()).addHandler(
          new CSBaseLoggerHandler());
      }
      catch (ClassNotFoundException e) {
        throw new ServerException(
          "A classe configurada em resources.service.class precisam estar disponvel",
          e);
      }
    }
    return packages.toArray(new String[0]);
  }

  /**
   * Carrega as fbricas com os objetos a serem injetados nos servios.
   * 
   * @return o mapa das fbricas com suas interfaces
   * @throws ServerException
   */
  private Map<Class<? extends Factory<ServiceAdapter>>, Class<?>> loadAdapterFactories()
    throws ServerException {
    List<String> factorys = getStringListProperty("adapter.service.factory");
    Map<Class<? extends Factory<ServiceAdapter>>, Class<?>> factories = new HashMap<>();
    for (int i = 0; i < factorys.size(); i++) {
      try {
        @SuppressWarnings("unchecked")
        Class<? extends Factory<ServiceAdapter>> factoryClass =
          (Class<? extends Factory<ServiceAdapter>>) Class.forName(factorys
            .get(i));
        Type[] genericInterfaces = factoryClass.getGenericInterfaces();
        ParameterizedType genericType = (ParameterizedType)genericInterfaces[0];
        Type type = genericType.getActualTypeArguments()[0];
        Class<?> interfaceClass = Class.forName(type.getTypeName());
        factories.put(factoryClass, interfaceClass);
      }
      catch (Throwable e) {
        throw new ServerException(e);
      }
    }
    return factories;
  }

  private void loadDAOFactories() {
    for (int i = 1;; i++) {
      String index = String.valueOf(i);
      if (!this.hasProperty("RestService.dao.service.factory.".concat(index))) {
        break;
      }
      String factoryClassName = this.getStringProperty("dao.service.factory.".concat(index));
      try {
        Class<?> factoryClass = Class.forName(factoryClassName);
        Class<?> interfaceClass = factoryClass.getInterfaces()[0];
        DAOFactory factory = DAOFactory.class.cast(factoryClass.newInstance());
        if (this.hasProperty("RestService.dao.service.properties.".concat(index))) {
          Properties properties = this.getExternalPropertyFile("dao.service.properties.".concat(index));
          factory.setProperties(properties);
        }
        ServiceAdapter.addDAOFactory(interfaceClass, factory);
      } catch (ClassNotFoundException e) {
        throw new RuntimeException(e);
      } catch (InstantiationException e) {
        throw new RuntimeException(e);
      } catch (IllegalAccessException e) {
        throw new RuntimeException(e);
      }
    }
  }

  /**
   * Configura o Swagger Ui
   * 
   * @param packageNames
   */
  private static void buildSwagger(String[] packageNames) {
    BeanConfig beanConfig = new BeanConfig();
    beanConfig.setVersion("1.0.0");
    StringBuilder builder = new StringBuilder();
    int i = 0;
    for (String p : packageNames) {
      if (i > 0) {
        builder.append(",");
      }
      builder.append(p.trim());
      i++;
    }
    beanConfig.setResourcePackage(builder.toString());
    beanConfig.setScan(true);
    beanConfig.setBasePath("/v1");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void shutdownService() throws ServerException {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdown();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return true;
  }

  /**
   * Constri a instncia do servio.
   * 
   * @throws ServerException caso ocorra falha na inicializao
   */
  public static void createService() throws ServerException {
    new RestService();
  }

  /**
   * Retorna a nica instancia do servico
   * 
   * @return nica instancia do servico
   */
  public static RestService getInstance() {
    return (RestService) getInstance(SERVICE_NAME);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String createToken(String subject, Map<String, Object> attributes,
    Date expirationDate, Date issuedDate) {
    //System.out.println("Data de emissao -- REST SERVICE createToken:" + issuedDate + "--" + issuedDate.getTime());
    Claims claims = Jwts.claims();
    if (subject != null) {
      claims.setSubject(subject);
    }
    if (attributes != null) {
      claims.putAll(attributes);
    }
    String token = Jwts.builder().setClaims(claims).setExpiration(
      expirationDate).setIssuedAt(issuedDate).signWith(SignatureAlgorithm.HS512, this.privateKey)
      .compact();
    return token;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String parserToken(String token, Map<String, Object> outAttributes)
    throws ParseException {
    String subject = null;
    try {
      Claims claims = Jwts.parser().setSigningKey(this.privateKey)
        .parseClaimsJws(token).getBody();
      subject = claims.getSubject();
      if (outAttributes != null) {
        outAttributes.putAll(claims);
      }
      AdministrationService adminService =  AdministrationService.getInstance();
      User user = adminService.getUser(subject);
      if (user!=null) {
        Date tokenIssued = claims.getIssuedAt();
        Date lastUpdate = new Date(user.getLastUpdate());
        //System.out.println("tokenIssued="+tokenIssued + ": "+ tokenIssued.getTime());
        //System.out.println("lastUpdate="+lastUpdate + ": "+ lastUpdate.getTime());
        long issueSeconds = tokenIssued.getTime()/1000;
        long lastSeconds = lastUpdate.getTime()/1000;
        //System.out.println("resultado:" + (issueSeconds-lastSeconds) );
        if ((issueSeconds-lastSeconds)>=0) {
          //if(tokenIssued.compareTo(lastUpdate)>=0){
          //System.out.println("OK! Token emitido aps a data da ultima alteracao dos dados do usuario");
        } else {
          //System.out.println("FALHA! Token emitido antes da data da ultima alteracao dos dados do usuario");
          throw new ParseException(getString("RestService.invalid.token.error"));
        }
      }
    }
    catch (Exception e) {
      throw new ParseException(getString("RestService.invalid.token.error"));
    }
    return subject;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getExternalURL() {
    return getStringProperty("external.url");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logSevereMessage(String msg, Throwable t) {
    Server.logSevereMessage(msg, t);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void logInfoMessage(String msg) {
    Server.logInfoMessage(msg);
  }

  /**
   * Recupera a chave privada contida no arquivo fornecido.
   * 
   * @param privateKeyFileName o path para o arquivo.
   * @return A chave privada RSA.
   * @throws IOException
   * @throws InvalidKeySpecException
   * @throws NoSuchAlgorithmException
   */
  public SecretKey readKeyFromFile(String privateKeyFileName)
    throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
    FileInputStream fis = new FileInputStream(privateKeyFileName);
    try {
      FileChannel channel = fis.getChannel();
      ByteBuffer buffer = ByteBuffer.allocate((int) channel.size());
      int size = channel.read(buffer);
      if (size != (int) channel.size()) {
        throw new IOException("No foi possvel ler todo o arquivo.");
      }
      return new SecretKeySpec(buffer.array(), "HmacSHA512");
    }
    finally {
      fis.close();
    }
  }
}
