/*
 * HttpService.java
 * 
 * $Id: HttpService.java 175498 2016-08-17 16:59:27Z tatimf $
 */
package csbase.server.services.httpservice;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.BindException;
import java.net.URLEncoder;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;
import java.util.Random;
import java.util.TreeMap;

import csbase.exception.HttpServiceException;
import csbase.exception.InvalidRequestException;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;
import csbase.remote.HttpServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.projectservice.ProjectService;

/**
 * A classe <code>HttpService</code> implementa o servio de download e upload
 * de arquivos. Cada requisicao de dowload e' inserida na requestList, para um
 * usuario, projeto e filePath Nulo, alem de a data de criacao do registro. O
 * filePath e' nulo no download pois o arquivo pode ter referencias para outros
 * arquivos que possuem caminhos distintos, sendo que estes precisam de um mesmo
 * codigo de acesso.
 */
public class HttpService extends Service implements HttpServiceInterface {
  /** Tipo da operao upload */
  private static final String UPLOAD = "upload";
  /** Tipo da operao download */
  private static final String DOWNLOAD = "download";

  /** Mensagem para parmetros invlidos/inadequados passsados pelo cliente. */
  private static final String INVALID_REQUEST_MSG = "invalid_request_msg";

  /** Hashtable com as requisicoes dos arquivo */
  private Map<String, RequestInfo> requestList = null;

  /** Servidor http */
  private String httpServiceURL = null;

  /** Gerador de numeros magicos para o download */
  private Random numberGenerator;

  /** Thread para limpeza das requisicoes antigas */
  private Cleaner cleaner;

  /** Flag usado para desativar a thread de limpeza */
  private boolean shutdown;

  /**
   * Representacao de usuario, projeto e arquivo. whenDone indica a data de
   * criacao de uma instancia de registro.
   */
  private class RequestInfo {
    /** Id do usurio */
    Object userId;
    /** Id do projeto */
    Object projectId;
    /** Caminho para o arquivo */
    String filePath;
    /** Tipo de arquivo */
    String fileType;
    /** Tipo de download */
    int downloadType;
    /** Momento no qual a requisio terminou */
    public long whenDone = 0;
    /** Booleano que indica se o arquivo deve ser removido */
    boolean removeFile = false;
    /** Caminho para upload */
    String uploadPresentationPath;
    /** Caminho do arquivo de resultado do upload */
    String uploadResultPath;
    /** Texto que ser exibido no navegador */
    String text;
    /** Parmetros passados para o relatrio */
    Map<String, String> parameters;
    /** Nome da classe do driver JDBC */
    String jdbcDriverClassName;
    /** URL de acesso para criao da conexo usada no relatrio */
    String jdbcURL;
    /** Usurio de acesso para criao da conexo usada no relatrio */
    String jdbcUser;
    /** Senha de acesso */
    String jdbcPassword;
    /** Observador que  notificado quando o upload termina. */
    Observer observer;
    /** Classe que realiza o upload */
    public UploadHandler fileHandler;
  }

  /**
   * Objeto que permite a observao do servidor da classe HttpService. Quando o
   * upload e' finalizado, os observadores sao notificados. E' interessante
   * quando outros servicos querem saber quando o upload terminou.
   */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Constri a instncia do servio.
   * 
   * @throws ServerException em caso de erro.
   */
  public static void createService() throws ServerException {
    new HttpService();
  }

  /**
   * Construtor.
   * 
   * @throws ServerException falha na construo do servio.
   */
  protected HttpService() throws ServerException {
    super(SERVICE_NAME);
    int port = Server.getInstance().getRMIExportPort();
    try {
      UnicastRemoteObject.exportObject(this, port);
    }
    catch (RemoteException e) {
      Throwable cause = e.getCause();
      if (cause instanceof BindException) {
        throw new ServerException("Porta RMI " + port + " j em uso", e);
      }
    }
    catch (Exception e) {
      throw new ServerException("Falha na construo de: " + SERVICE_NAME, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDownloadURL(Object userId, String password, Object projectId,
    String[] filePath) throws RemoteException {
    User user = LoginService.getInstance().checkLogin((String) userId,
      password);
    if (user == null) {
      throw new RemoteException("Falha na autenticao do usurio.\n"
        + "Verifique se a chave e a senha so vlidos.");
    }
    return getURL(userId, projectId, filePath, null, DOWNLOAD,
      FILE_DOWNLOAD_TYPE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getUploadURL(Object userId, String password, Object projectId,
    String[] filePath, String fileType) throws RemoteException {
    User user = LoginService.getInstance().checkLogin((String) userId,
      password);
    if (user == null) {
      throw new RemoteException("Falha na autenticao do usurio.\n"
        + "Verifique se a chave e a senha so vlidos.");
    }
    try {
      ProjectService projectService = ProjectService.getInstance();
      if (!projectService.existsFile(projectId, filePath)) {
        String pathStr = Arrays.toString(filePath);
        throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
          "Caminho " + pathStr + " no encontrado na rvore do projeto.");
      }
      String[] dirPath = new String[filePath.length - 1];
      for (int i = 0; i < filePath.length - 1; i++) {
        dirPath[i] = filePath[i];
      }
      String fileName = filePath[filePath.length - 1];
      ClientProjectFile cpf = projectService.getChild(projectId, dirPath,
        fileName);
      projectService.setUnderConstruction(projectId, filePath, true);
    }
    catch (InvalidRequestException e) {
      Server.logSevereMessage("Argumentos invlidos do cliente: (projectId="
        + projectId + "), " + "(filePath=" + filePath + ")");
    }
    return getURL(userId, projectId, filePath, fileType, UPLOAD);
  }

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado.
   * 
   * @param userId Identificador do usurio no servidor http remoto.
   * @param projectId Identificador do projeto.
   * @param filePath Caminho do arquivo no projeto.
   * @param fileType Tipo do arquivo.
   * @param type Indica se a operao  de upload ou download.
   * @param downloadType Indica o tipo de download.
   * @return Texto que representa a URL do arquivo.
   * @throws RemoteException
   */
  private String getURL(Object userId, Object projectId, String[] filePath,
    String fileType, String type, int downloadType) throws RemoteException {
    String path = filePathtoString(filePath);
    if (path == null) {
      String msg = "Path do arquivo invalido";
      Server.logSevereMessage(msg);
      throw new RemoteException(msg);
    }
    RequestInfo requestInfo = new RequestInfo();
    requestInfo.userId = userId;
    requestInfo.projectId = projectId;
    requestInfo.filePath = path;
    requestInfo.fileType = fileType;
    requestInfo.downloadType = downloadType;
    /* Verifica se existe codigo para o usuario e projeto */
    String accessCode = null;
    if (type.equals(DOWNLOAD)) {
      accessCode = findCode(userId, projectId, null);
    }
    else {
      accessCode = findCode(userId, projectId, path);
      requestInfo.uploadPresentationPath = getStringProperty(
        "uploadPresentation");
      requestInfo.uploadResultPath = getStringProperty("uploadResult");
    }
    if (accessCode == null) {
      /* Gerando um numero magico */
      accessCode = "" + Math.abs(numberGenerator.nextLong());
      accessCode = accessCode + Math.abs(numberGenerator.nextLong());
    }
    /* Coloca a hora atual no registro */
    requestInfo.whenDone = System.currentTimeMillis();
    /* adiciona a lista de requisicoes */
    synchronized (requestList) {
      requestList.put(accessCode, requestInfo);
    }
    String url;
    if (type.equals(DOWNLOAD)) {
      url = httpServiceURL + "/" + type + "/" + accessCode + "/" + type + "/"
        + path;
    }
    else {
      url = httpServiceURL + "/" + type + "/" + accessCode + "/" + type;
    }
    Server.logInfoMessage("URL: " + url + " - filePath: " + requestInfo.filePath
      + " - user: " + userId);
    return url;
  }

  /**
   * Codifica o texto para UTF-8.
   * 
   * @param text texto a ser codificado.
   * @return texto codificado.
   */
  private String encode(String text) {
    try {
      return URLEncoder.encode(text, "UTF-8");
    }
    catch (Exception e) {
      Server.logSevereMessage("No foi possvel codificar o texto: " + text);
      return null;
    }
  }

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado.
   * 
   * @param userId Identificador do usurio no servidor http remoto.
   * @param projectId Identificador do projeto.
   * @param filePath Caminho do arquivo no projeto.
   * @param fileType Tipo do arquivo.
   * @param type Indica se a operao  de upload ou download.
   * @return Texto que representa a URL do arquivo.
   * @throws RemoteException erro ao obter a url.
   */
  private String getURL(Object userId, Object projectId, String[] filePath,
    String fileType, String type) throws RemoteException {
    return getURL(userId, projectId, filePath, fileType, type,
      UNKNOWN_DOWNLOAD_TYPE);
  }

  /**
   * Verifica se jah exite codigo para o usuario, projeto e filePath, no caso do
   * upload.
   * 
   * @param userId .
   * @param projectId .
   * @param filePath .
   * 
   * @return accessCode, caso exista. Null caso contrario.
   */
  private String findCode(Object userId, Object projectId, String filePath) {
    synchronized (requestList) {
      for (Map.Entry<String, RequestInfo> entry : requestList.entrySet()) {
        String accessCode = entry.getKey();
        RequestInfo requestInfo = entry.getValue();
        if ((userId.equals(requestInfo.userId)) && (projectId.equals(
          requestInfo.projectId)) && ((filePath == null) || ((filePath != null)
            && (filePath.equals(requestInfo.filePath))))) {
          return accessCode;
        }
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getFileType(String accessCode, String filePath)
    throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo requestInfo;
    synchronized (requestList) {
      requestInfo = requestList.get(accessCode);
    }
    if (requestInfo == null) {
      return null;
    }
    String mimeType = "application/octet-stream";
    String[] path;
    if (filePath == null) {
      path = requestInfo.filePath.split("/");
    }
    else {
      String auxPath = expandPath(filePath);
      if (auxPath == null) {
        return null;
      }
      path = auxPath.split("/");
    }
    // Se no tem projeto associado infere o tipo a partir do nome.
    if (requestInfo.projectId == null) {
      mimeType = getMimeTypeFromName(path[path.length - 1]);
    }
    else {
      try {
        ProjectService projectService = ProjectService.getInstance();
        if (!projectService.existsFile(requestInfo.projectId, path)) {
          String pathStr = Arrays.toString(path);
          throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
            "Caminho " + pathStr + " no encontrado na rvore do servidor");
        }
        String[] dirPath = new String[path.length - 1];
        for (int i = 0; i < path.length - 1; i++) {
          dirPath[i] = path[i];
        }
        String fileName = path[path.length - 1];
        ClientProjectFile cpf = projectService.getChild(requestInfo.projectId,
          dirPath, fileName);
        String fileType = cpf.getType();
        mimeType = projectService.getMimeType(fileType);
      }
      catch (Exception e) {
        Server.logSevereMessage("Falha ao obter tipo do arquivo: "
          + requestInfo.filePath, e);
      }
    }
    return mimeType;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getDownloadType(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return UNKNOWN_DOWNLOAD_TYPE;
    }
    RequestInfo requestInfo;
    synchronized (requestList) {
      requestInfo = requestList.get(accessCode);
    }
    if (requestInfo == null) {
      return UNKNOWN_DOWNLOAD_TYPE;
    }
    return requestInfo.downloadType;
  }

  /**
   * Obtm o mimeType a partir da terminacao do arquivo.
   * 
   * @param fileName .
   * 
   * @return .
   */
  private String getMimeTypeFromName(String fileName) {
    int index = fileName.lastIndexOf(".");
    String fileExtention = null;
    if (index != -1) {
      fileExtention = fileName.substring(index + 1);
    }
    if (fileExtention == null) {
      return "application/zip";
    }
    fileExtention = fileExtention.toLowerCase();
    if (fileExtention.equals("doc")) {
      return "application/msword";
    }
    if (fileExtention.equals("xls")) {
      return "application/vnd.ms-excel";
    }
    if (fileExtention.equals("pdf")) {
      return "application/pdf";
    }
    if (fileExtention.equals("txt")) {
      return "text/plain";
    }
    if (fileExtention.equals("log")) {
      return "text/plain";
    }
    if (fileExtention.equals("jpg")) {
      return "image/jpeg";
    }
    if (fileExtention.equals("gif")) {
      return "image/gif";
    }
    if (fileExtention.equals("tif")) {
      return "image/tiff";
    }
    if (fileExtention.equals("rob")) {
      return "text/plain";
    }
    if (fileExtention.equals("html")) {
      return "text/html";
    }
    return "application/zip";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean createFile(String accessCode, String fileName)
    throws RemoteException {
    if (accessCode == null) {
      return false;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return false;
    }
    if (reqInfo.projectId == null) {
      if (createFileOutProject(reqInfo.filePath, fileName)) {
        reqInfo.filePath = reqInfo.filePath + File.separator + fileName;
        return true;
      }
      else {
        return false;
      }
    }
    String[] dirPath = reqInfo.filePath.split("/");
    try {
      ProjectService projectService = ProjectService.getInstance();
      if (!projectService.existsFile(reqInfo.projectId, dirPath)) {
        return false;
      }
      projectService.createFile(reqInfo.projectId, dirPath, fileName,
        reqInfo.fileType);
      /* Troca o caminho do diretorio, pelo do arquivo para que */
      /* possa atualizar o estado na finalizacao do upload */
      String fPath = "";
      String[] filePath = new String[dirPath.length + 1];
      for (int i = 0; i < dirPath.length; i++) {
        fPath = fPath + dirPath[i] + "/";
        filePath[i] = dirPath[i];
      }
      fPath = fPath + fileName;
      filePath[filePath.length - 1] = fileName;
      reqInfo.filePath = fPath;
      projectService.setUnderConstruction(reqInfo.projectId, filePath, true);
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha ao criar o arquivo para upload: "
        + fileName, e);
      return false;
    }
    return true;
  }

  /**
   * Cria um arquivo vazio que no est na rea de projetos. Retorna true se o
   * arquivo j existir, ou se conseguir criar.
   * 
   * @param filePath caminho do arquivo.
   * @param fileName nome do arquivo.
   * @return true, caso o arquivo tenha sido criado, ou false, caso tenha
   *         ocorrido algum erro ao criar o arquivo.
   */
  private boolean createFileOutProject(String filePath, String fileName) {
    try {
      File file = new File(filePath + File.separator + fileName);
      if (file.exists()) {
        return true;
      }
      return file.createNewFile();
    }
    catch (Exception e) {
      Server.logSevereMessage("createFileOutProject: " + e.getMessage());
      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getFilePath(String accessCode) throws RemoteException {
    return getFilePath(accessCode, null);
  }

  /**
   * Transforma um caminho de arquivo para o formato utilizado por ambientes
   * Unix.
   * 
   * @param path caminho do arquivo.
   * @return caminho transformado.
   */
  private static String unixPath(String path) {
    return path.replaceAll("\\\\", "/");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getFilePath(String accessCode, String filePath)
    throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    // Se no tem projeto associado retorna o prprio caminho.
    if (reqInfo.projectId == null) {
      return reqInfo.filePath;
    }
    String path[];
    if (filePath == null) {
      path = reqInfo.filePath.split("/");
    }
    else {
      String auxPath = expandPath(filePath);
      if (auxPath == null) {
        return null;
      }
      path = auxPath.split("/");
    }
    try {
      ProjectService projectService = ProjectService.getInstance();
      if (!projectService.existsFile(reqInfo.projectId, path)) {
        String pathStr = Arrays.toString(path);
        throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
          "Caminho " + pathStr + " no encontrado na rvore do servidor");
      }
      return unixPath(projectService.getAbsolutePath(reqInfo.projectId, path));
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha obtencao do caminho do arquivo! " + "("
        + reqInfo.filePath + ") " + e);
      return null;
    }
  }

  /**
   * Transforma um caminho da forma como  lido para a forma em que ser
   * utilizado.
   * 
   * @param path caminho a ser transformado.
   * @return caminho vlido para ser utilizado.
   */
  private String expandPath(String path) {
    String newPath = path.replaceAll("//", "/");
    do {
      path = newPath;
      newPath = newPath.replaceFirst("^/", "");
      if (newPath.startsWith("../")) {
        return null;
      }
      newPath = newPath.replaceFirst("[^/]+/\\.\\.", "");
      newPath = newPath.replaceFirst("//", "/");
    } while (!newPath.equals(path));
    if (newPath.equals("..")) {
      return null;
    }
    return newPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getPresentationPath(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    try {
      File presentationFile = new File(reqInfo.uploadPresentationPath);
      return unixPath(presentationFile.getCanonicalPath());
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha na obteno do caminho do arquivo: "
        + reqInfo.uploadPresentationPath);
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getResultPath(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    try {
      File resultFile = new File(reqInfo.uploadResultPath);
      return unixPath(resultFile.getCanonicalPath());
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha na obteno do caminho do arquivo: "
        + reqInfo.uploadResultPath);
      return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDirectory(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return false;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return false;
    }
    try {
      if (reqInfo.projectId == null) {
        File f = new File(reqInfo.filePath);
        return f.isDirectory();
      }
      String[] path = reqInfo.filePath.split("/");
      ProjectService projectService = ProjectService.getInstance();
      if (!projectService.existsFile(reqInfo.projectId, path)) {
        String pathStr = Arrays.toString(path);
        throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
          "Caminho " + pathStr + " no encontrado na rvore do projeto.");
      }
      String[] dirPath = new String[path.length - 1];
      for (int i = 0; i < path.length - 1; i++) {
        dirPath[i] = path[i];
      }
      String fileName = path[path.length - 1];
      ClientProjectFile cpf = projectService.getChild(reqInfo.projectId,
        dirPath, fileName);
      return cpf.isDirectory();
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha no metodo isDirectory do upload! " + "("
        + reqInfo.filePath + ") ", e);
      return false;
    }
  }

  /**
   * Remove o upload da lista de requisicoes e informa que o arquivo nao esta'
   * mais em construcao, se foi um upload de arquivo de projeto. Notifica os
   * observadores que o upload terminou.
   * 
   * @param accessCode cdigo de acesso.
   * @param fileName O nome original do arquivo que sofreu upload.
   * 
   * @return boolean se a operacao foi realizada com sucesso.
   * 
   * @throws HttpServiceException em caso de erro.
   */
  @Override
  public boolean finishUpload(String accessCode, String fileName)
    throws HttpServiceException {
    if (accessCode == null) {
      return false;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return false;
    }
    if (reqInfo.observer != null) {
      UploadResult result = new UploadResult(accessCode, fileName,
        reqInfo.filePath);
      reqInfo.observer.update(observable, result);
    }
    else if (reqInfo.fileHandler != null) {
      reqInfo.fileHandler.execute(reqInfo.filePath);
    }
    if (reqInfo.projectId == null) {
      synchronized (requestList) {
        requestList.remove(accessCode);
      }
      return true;
    }
    String[] path = reqInfo.filePath.split("/");
    try {
      ProjectService projectService = ProjectService.getInstance();
      if (!projectService.existsFile(reqInfo.projectId, path)) {
        String pathStr = Arrays.toString(path);
        throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
          "Caminho " + pathStr + " no encontrado na rvore do servidor");
      }
      String[] dirPath = new String[path.length - 1];
      for (int i = 0; i < path.length - 1; i++) {
        dirPath[i] = path[i];
      }
      String name = path[path.length - 1];
      ClientProjectFile cpf = projectService.getChild(reqInfo.projectId,
        dirPath, name);
      projectService.setUnderConstruction(cpf.getProjectId(), cpf.getPath(),
        false);
      synchronized (requestList) {
        requestList.remove(accessCode);
      }
      return true;
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha na finalizacao do upload! " + "(" + path
        + ") ", e);
      return false;
    }
  }

  /**
   * Transforma array de String em uma String separada por "/" e codificada
   * (UTF-8).
   * 
   * @param filePath path
   * 
   * @return texto com resultado.
   */
  private String filePathtoString(String[] filePath) {
    if (filePath == null) {
      return null;
    }
    if (filePath.length == 0) {
      return null;
    }
    String result = "";
    try {
      String pathEncoded;
      for (int i = 0; i < (filePath.length - 1); i++) {
        pathEncoded = URLEncoder.encode(filePath[i], "UTF-8");
        result = result + pathEncoded + "/";
      }
      pathEncoded = URLEncoder.encode(filePath[filePath.length - 1], "UTF-8");
      result = result + pathEncoded;
    }
    catch (UnsupportedEncodingException e) {
      Server.logSevereMessage("O caminho do arquivo nao pode ser codificado.");
      return null;
    }
    return result;
  }

  /**
   * Obtm uma URL para exibio de um arquivo do sistema de arquivos (NFS) em
   * um navegador web.
   * 
   * @param filePath caminho absoluto do arquivo no sistema de arquivos.
   * 
   * @return url para exibio do arquivo em um navegador.
   * 
   * @throws RemoteException em caso de erro.
   */
  public String getFileDownloadURL(String filePath) throws RemoteException {
    return getURL(filePath, DOWNLOAD, null, null, FILE_DOWNLOAD_TYPE, null,
      null, null, null, null, null, null);
  }

  /**
   * Obtm a URL para a exibio de um relatrio no formato pdf em um navegador
   * web.
   * 
   * @param filePath caminho absoluto do arquivo .jasper que representa o layout
   *        do relatrio
   * @param parameters parmetros passados para o relatrio
   * @param jdbcDriverClassName nome da classe do driver jdbc para criao da
   *        conexo usada no relatrio
   * @param jdbcURL URL de acesso para criao da conexo usada no relatrio
   * @param jdbcUser usurio de acesso para criao da conexo usada no
   *        relatrio
   * @param jdbcPassword senha de acesso para criao da conexo usada no
   *        relatrio
   * 
   * @return url para exibio do arquivo em um navegador.
   * 
   * @throws RemoteException em caso de erro.
   */
  public String getDownloadPDFReportURL(String filePath,
    Map<String, String> parameters, String jdbcDriverClassName, String jdbcURL,
    String jdbcUser, String jdbcPassword) throws RemoteException {
    return getURL(filePath, DOWNLOAD, null, null, PDF_REPORT_DOWNLOAD_TYPE,
      null, parameters, jdbcDriverClassName, jdbcURL, jdbcUser, jdbcPassword,
      null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> getParameters(String accessCode)
    throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    return reqInfo.parameters;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getJdbcDriverClassName(String accessCode)
    throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    return reqInfo.jdbcDriverClassName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getJdbcURL(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    return reqInfo.jdbcURL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getJdbcUser(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    return reqInfo.jdbcUser;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getJdbcPassword(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    return reqInfo.jdbcPassword;
  }

  //
  // Mtodos relativos ao download e upload de arquivos do VGE e aplicaes
  // que no dependem de projeto.
  //

  /**
   * Obtem a url para upload do arquivo.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param presentationPath Path do arquivo html para upload.
   * @param resultPath Path do arquivo html resultado do upload.
   * 
   * @return String que representa a url do arquivo
   * 
   * @throws RemoteException em caso de erro.
   */
  public String getUploadURL(String filePath, String presentationPath,
    String resultPath) throws RemoteException {
    return getURL(filePath, UPLOAD, presentationPath, resultPath, null);
  }

  /**
   * Obtem a url para upload do arquivo.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param presentationPath Path do arquivo html para upload.
   * @param resultPath Path do arquivo html resultado do upload.
   * @param fileHandler Classe que realiza o upload.
   * 
   * @return String que representa a url do arquivo
   * 
   */
  public String getUploadURL(String filePath, String presentationPath,
    String resultPath, UploadHandler fileHandler) {
    if (filePath == null) {
      throw new IllegalArgumentException("O parmetro filePath est nulo.");
    }
    if (presentationPath == null) {
      throw new IllegalArgumentException(
        "O parmetro presentationPath est nulo.");
    }
    if (resultPath == null) {
      throw new IllegalArgumentException("O parmetro resultPath est nulo.");
    }
    if (fileHandler == null) {
      throw new IllegalArgumentException("O parmetro fileHandler est nulo.");
    }
    if (httpServiceURL == null) {
      throw new IllegalStateException("O parmetro httpServiceURL est nulo.");
    }
    /* Verifica se existe codigo para o arquivo */
    String accessCode = getAccessCode(filePath);
    if (accessCode == null) {
      accessCode = "" + Math.abs(numberGenerator.nextLong());
      accessCode = accessCode + Math.abs(numberGenerator.nextLong());
    }
    /* adiciona a lista de requisicoes */
    RequestInfo req = new RequestInfo();
    req.userId = Service.getUser();
    req.downloadType = URL_DOWNLOAD_TYPE;
    req.filePath = filePath;
    req.removeFile = true;
    req.uploadPresentationPath = presentationPath;
    req.uploadResultPath = resultPath;
    req.fileHandler = fileHandler;
    req.whenDone = System.currentTimeMillis();
    synchronized (requestList) {
      requestList.put(accessCode, req);
    }
    String url = httpServiceURL + "/" + UPLOAD + "/" + accessCode + "/"
      + UPLOAD;
    return url;
  }

  /**
   * Obtem a url para upload do arquivo.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param presentationPath Path do arquivo html para upload.
   * @param resultPath Path do arquivo html resultado do upload.
   * @param obs Observador que  notificado quando o upload termina.
   * 
   * @return String que representa a url do arquivo
   * 
   */
  public String getUploadURL(String filePath, String presentationPath,
    String resultPath, Observer obs) {
    return getURL(filePath, UPLOAD, presentationPath, resultPath, obs);
  }

  /**
   * Obtem a url para download do arquivo.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * 
   * @return String que representa a url do arquivo
   * @throws RemoteException em caso de erro.
   * 
   */
  public String getDownloadURL(String filePath) throws RemoteException {
    return getURL(filePath, DOWNLOAD, (String) null, (String) null, null);
  }

  /**
   * Obtem a url para download do arquivo.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param parameters parmetros que sero armazenados na RequestInfo para uso
   *        geral.
   * 
   * @return String que representa a url do arquivo
   * @throws RemoteException em caso de erro.
   * 
   */
  public String getDownloadURL(String filePath, Map<String, String> parameters)
    throws RemoteException {
    return getURL(filePath, DOWNLOAD, null, null, null, parameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDownloadTextURL(Object userId, String password, String text)
    throws RemoteException {
    // Usurio que fez a chamada ao mtodo. Usar para a autenticao.
    User user = LoginService.getInstance().checkLogin((String) userId,
      password);
    if (user == null) {
      throw new RemoteException("Falha na autenticao do usurio. \n"
        + "Verifique se o login e a senha so vlidos");
    }
    return getDownloadTextURL(text);
  }

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for
   * TEXTO.
   *
   * @param text Texto que ser exibido no navegador
   *
   * @return a URL.
   *
   * @throws RemoteException em caso de erro.
   */
  public String getDownloadTextURL(String text) throws RemoteException {
    // s para no passar nulo e ser nico
    String filePath = "/" + Math.abs(numberGenerator.nextLong()) + "/"
      + "TEXT.txt";
    return getURL(filePath, DOWNLOAD, null, null, TEXT_DOWNLOAD_TYPE, text,
      null, null, null, null, null, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDownloadTextURL(Object userId, String password, String text,
    String fileName) throws RemoteException {
    // Usurio que fez a chamada ao mtodo. Usar para a autenticao.
    User user = LoginService.getInstance().checkLogin((String) userId,
      password);
    if (user == null) {
      throw new RemoteException("Falha na autenticao do usurio. \n"
        + "Verifique se o login e a senha so vlidos");
    }
    return getDownloadTextURL(text, fileName);
  }

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for
   * TEXTO.
   *
   * @param text texto que ser exibido no navegador caso o
   *        <code>downloadType</code> seja
   *        <code>HttpServiceInterface.TEXT_DOWNLOAD_TYPE</code>.
   * @param fileName caminho do arquivo local no projeto.
   * @return a URL.
   *
   * @throws RemoteException em caso de erro.
   */
  private String getDownloadTextURL(String text, String fileName)
    throws RemoteException {
    // s para no passar nulo e ser nico
    String filePath = "/" + Math.abs(numberGenerator.nextLong()) + "/"
      + fileName;
    return getURL(filePath, DOWNLOAD, null, null, TEXT_DOWNLOAD_TYPE, text,
      null, null, null, null, null, null);
  }

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for CSV.
   * 
   * @param userId identificao do usurio.
   * @param password senha do usurio.
   * @param text texto contendo o contedo do CSV.
   * 
   * @return URL de download do arquivo CSV.
   * 
   * @throws RemoteException em caso de erro.
   */
  @Override
  public String getDownloadCSVURL(Object userId, String password, String text)
    throws RemoteException {
    return getDownloadCSVURL(userId, password, text, "TEXT");
  }

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for CSV.
   * OBS: Esse mtodo no  exportado na interface remota, s pode ser chamada
   * de outro servio.
   * 
   * @param text texto contendo o contedo do CSV.
   * 
   * @return URL de download do arquivo CSV.
   * 
   * @throws RemoteException em caso de erro.
   */
  public String getDownloadCSVURL(String text) throws RemoteException {
    return getDownloadCSVURL(text, "TEXT");
  }

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for CSV.
   * OBS: Esse mtodo no  exportado na interface remota, s pode ser chamada
   * de outro servio.
   * 
   * @param text texto contendo o contedo do CSV.
   * @param fileName nome do arquivo gerado para download
   * 
   * @return URL de download do arquivo CSV.
   * 
   * @throws RemoteException em caso de erro.
   */
  public String getDownloadCSVURL(String text, String fileName)
    throws RemoteException {
    String filePath = "/" + Math.abs(numberGenerator.nextLong()) + "/"
      + fileName + ".csv";
    return getURL(filePath, DOWNLOAD, null, null, CSV_DOWNLOAD_TYPE, text, null,
      null, null, null, null, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getText(String accessCode) throws RemoteException {
    if (accessCode == null) {
      return null;
    }
    RequestInfo reqInfo;
    synchronized (requestList) {
      reqInfo = requestList.get(accessCode);
    }
    if (reqInfo == null) {
      return null;
    }
    return reqInfo.text;
  }

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param type se e' upload ou download
   * @param presentationPath Path do arquivo html para upload.
   * @param resultPath Path do arquivo html resultado do upload.
   * @param downloadType tipo de download.
   * @param text texto que ser exibido no navegador caso o
   *        <code>downloadType</code> seja
   *        <code>HttpServiceInterface.TEXT_DOWNLOAD_TYPE</code>.
   * @param parameters parmetros que sero armazenados na RequestInfo para uso
   *        geral.
   * @param jdbcDriverClassName nome da classe do driver JDBC para a criao de
   *        uma eventual conexo com um banco de dados a partir do
   *        <code>DownloadServlet</code>.
   * @param jdbcURL URL JDBC para a criao de uma eventual conexo com um banco
   *        de dados a partir do <code>DownloadServlet</code>.
   * @param jdbcUser usurio JDBC para a criao de uma eventual conexo com um
   *        banco de dados a partir do <code>DownloadServlet</code>.
   * @param jdbcPassword senha JDBC para a criao de uma eventual conexo com
   *        um banco de dados a partir do <code>DownloadServlet</code>.
   * @param observer Observador que e' notificado quando o upload termina.
   * 
   * @return String que representa a url do arquivo
   * 
   */
  private String getURL(String filePath, String type, String presentationPath,
    String resultPath, int downloadType, String text,
    Map<String, String> parameters, String jdbcDriverClassName, String jdbcURL,
    String jdbcUser, String jdbcPassword, Observer observer) {
    if ((filePath == null) || (type == null)) {
      return null;
    }
    if (httpServiceURL == null) {
      return null;
    }
    /* Verifica se existe codigo para o arquivo */
    String accessCode = getAccessCode(filePath);
    if (accessCode == null) {
      accessCode = "" + Math.abs(numberGenerator.nextLong());
      accessCode = accessCode + Math.abs(numberGenerator.nextLong());
    }
    /* adiciona a lista de requisicoes */
    RequestInfo req = new RequestInfo();
    req.userId = Service.getUser();
    req.downloadType = downloadType;
    req.text = text;
    req.filePath = filePath;
    String fileName = null;
    if (type.equals(UPLOAD)) {
      req.removeFile = true;
      req.uploadPresentationPath = presentationPath;
      req.uploadResultPath = resultPath;
      req.observer = observer;
    }
    else {
      // Obtm o nome do arquivo para incluir na url de download
      String[] path = filePath.split("/");
      if (path == null) {
        fileName = filePath;
      }
      else {
        fileName = path[path.length - 1];
      }
      fileName = encode(fileName);

      // parametros da RequestInfo para gerao de relatrios em pdf
      req.parameters = parameters; // parametros passados para o relatrio
      req.jdbcDriverClassName = jdbcDriverClassName;
      req.jdbcURL = jdbcURL;
      req.jdbcUser = jdbcUser;
      req.jdbcPassword = jdbcPassword;
    }
    req.whenDone = System.currentTimeMillis();
    synchronized (requestList) {
      requestList.put(accessCode, req);
    }
    String url;
    if (fileName == null) {
      url = httpServiceURL + "/" + type + "/" + accessCode + "/" + type;
    }
    else {
      url = httpServiceURL + "/" + type + "/" + accessCode + "/" + type + "/"
        + fileName;
    }
    Server.logInfoMessage("URL: " + url + " - filePath: " + req.filePath
      + " - user: " + Service.getUser().getLogin());
    return url;
  }

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param type se e' upload ou download
   * @param presentationPath Path do arquivo html para upload.
   * @param resultPath Path do arquivo html resultado do upload.
   * @param obs Observador que  notificado quando o upload termina.
   * 
   * @return String que representa a url do arquivo
   */
  private String getURL(String filePath, String type, String presentationPath,
    String resultPath, Observer obs) {
    return getURL(filePath, type, presentationPath, resultPath,
      URL_DOWNLOAD_TYPE, null, null, null, null, null, null, obs);
  }

  /**
   * Obtem a url atraves da qual o arquivo deve ser acessado.
   * 
   * @param filePath caminho do arquivo local no projeto.
   * @param type se e' upload ou download
   * @param presentationPath Path do arquivo html para upload.
   * @param resultPath Path do arquivo html resultado do upload.
   * @param obs Observador que e' notificado quando o upload termina.
   * @param parameters parmetros que sero armazenados na RequestInfo para uso
   *        geral.
   * 
   * @return String que representa a url do arquivo
   * 
   * @throws RemoteException em caso de erro.
   */
  private String getURL(String filePath, String type, String presentationPath,
    String resultPath, Observer obs, Map<String, String> parameters)
    throws RemoteException {
    return getURL(filePath, type, presentationPath, resultPath,
      URL_DOWNLOAD_TYPE, null, parameters, null, null, null, null, obs);
  }

  /**
   * Verifica se jah exite codigo para o arquivo.
   * 
   * @param filePath path
   * 
   * @return codigo de acesso, caso exista. Null caso contrario.
   */
  private String getAccessCode(String filePath) {
    synchronized (requestList) {
      for (Map.Entry<String, RequestInfo> entry : requestList.entrySet()) {
        String fileTemp = entry.getValue().filePath;
        if (filePath.equals(fileTemp)) {
          return entry.getKey();
        }
      }
    }
    return null;
  }

  //
  // Fim dos mtodos relativos ao download e upload de arquivos do VGE.
  //
  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return true;
  }

  /**
   * Trmino do servio.
   * 
   * @throws ServerException em caso de erro.
   */
  @Override
  public void shutdownService() throws ServerException {
    shutdown = true;
    if (cleaner != null) {
      cleaner.interrupt();
    }
    requestList = null;
  }

  /**
   * Inicializao do servio.
   * 
   * @throws ServerException se ocorrer um erro na inicializao
   */
  @Override
  public void initService() throws ServerException {
    requestList = new TreeMap<String, RequestInfo>();
    httpServiceURL = getStringProperty("webapp");

    /* Criando o gerador de numeros magico */
    double seed = Math.random();
    long longSeed = (long) (seed * Long.MAX_VALUE);
    numberGenerator = new Random(longSeed);

    /* Obtem as propriedades para a thread de limpeza */
    this.shutdown = false;
    float maxSaveTime = (float) getDoubleProperty("maxSaveTime");
    float cleanTime = (float) getDoubleProperty("cleanTime");
    cleaner = new Cleaner(maxSaveTime, cleanTime);
  }

  /**
   * Obtm a instncia do servio.
   * 
   * @return instncia do servio.
   */
  public static HttpService getInstance() {
    return (HttpService) getInstance(SERVICE_NAME);
  }

  /**
   * Limpa a fila de requisies, retirando entradas relativas as requisies
   * atendidas a mais de <code>maxSaveTime</code> dias. O thread criado pelo
   * objeto desta classe faz a limpeza na fila a cada <code>cleanTime</code>
   * dias. Tanto <code>maxSaveTime</code> quanto <code>cleanTime</code> so
   * propriedades do servio. O loop pode ser interrompido atravs do campo
   * <code>shutdown</code>.
   */
  private class Cleaner extends Thread {
    /** Dia em milissegundos */
    private static final long DAY_IN_MS = 1000 * 60 * 60 * 24;
    /** Tempo mximo para o armazenamento de entradas de requisies */
    private final float maxSaveTime;
    /** Intervalo de tempo em que a limpeza deve ser realizada */
    private final float cleanTime;

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      while (!shutdown) {
        try {
          sleep((long) (cleanTime * DAY_IN_MS));
        }
        catch (InterruptedException e) {
        }
        if (shutdown) {
          break;
        }
        Server.logFineMessage("Thread iniciada: cleaner");
        long now = System.currentTimeMillis();
        synchronized (requestList) {
          LinkedList<String> toDelete = new LinkedList<String>();
          for (Map.Entry<String, RequestInfo> entry : requestList.entrySet()) {
            String key = entry.getKey();
            RequestInfo reqInfo = entry.getValue();
            if ((reqInfo.whenDone > 0) && (reqInfo.whenDone < (now
              - (maxSaveTime * DAY_IN_MS)))) {
              toDelete.add(key);
              if (reqInfo.removeFile) {
                (new File(reqInfo.filePath)).delete();
              }
            }
          }
          for (String toDelKey : toDelete) {
            requestList.remove(toDelKey);
          }
        }
        Server.logFineMessage("Thread terminada: cleaner");
      }
    }

    /**
     * Construtor.
     * 
     * @param maxSaveTime Tempo mximo para o armazenamento de entradas de
     *        requisies.
     * @param cleanTime Intervalo de tempo em que a limpeza deve ser realizada.
     */
    public Cleaner(float maxSaveTime, float cleanTime) {
      this.maxSaveTime = maxSaveTime;
      this.cleanTime = cleanTime;
      setDaemon(true);
      this.start();
    }
  }

  /**
   * Pega a URL que ser chamada pelo browser quando o tipo de dowload for CSV.
   * 
   * @param userId identificao do usurio.
   * @param password senha do usurio.
   * @param text texto contendo o contedo do CSV.
   * @param fileName nome do arquivo que ser gerado
   * 
   * @return URL de download do arquivo CSV.
   * 
   * @throws RemoteException em caso de erro.
   */
  @Override
  public String getDownloadCSVURL(Object userId, String password, String text,
    String fileName) throws RemoteException {
    // Usurio que fez a chamada ao mtodo. Usar para a autenticao.
    User user = LoginService.getInstance().checkLogin((String) userId,
      password);
    if (user == null) {
      throw new RemoteException("Falha na autenticao do usurio. \n"
        + "Verifique se o login e a senha so vlidos");
    }
    return getDownloadCSVURL(text, fileName);
  }

  /**
   * Obtm o nmero de requisies de download retidos em memria, aguardando
   * limpeza peridica.
   * 
   * @return tamanho.
   * 
   * @throws RemoteException em caso de erro.
   */
  @Override
  public int getRequestListSize() throws RemoteException {
    synchronized (requestList) {
      return requestList.size();
    }
  }

}
