package csbase.server.services.projectservice;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.commons.io.IOUtils;

import csbase.exception.ServiceFailureException;
import csbase.logic.ProjectFileType;
import csbase.logic.ProjectFileTypeInfo;
import csbase.server.Server;

/**
 * Repositrio de tipos de arquivo de projeto {@link ProjectFileTypeInfo}:
 * responsvel por ler o arquivo de tipos de arquivos de projeto.
 * 
 * O arquivo tipos de arquivos de projetos  um arquivo cujo formato 
 * {@link Properties}. Ele descreve todos os tipos de arquivos de projeto .
 * 
 * Um exemplo de trecho deste arquivo :
 * {@code filetype.0.typeCode=UNKNOWN}
 * {@code filetype.UNKNOWN.description_pt_BR=Desconhecido}
 * {@code filetype.UNKNOWN.description_en_US=Unknown}
 * {@code filetype.UNKNOWN.description_es_AR=Desconocido}
 * {@code filetype.UNKNOWN.baseIcon=/resources/filetypes/other.gif}
 * {@code filetype.UNKNOWN.ucIcon=/resources/filetypes/other_uc.gif}
 * {@code filetype.UNKNOWN.cutIcon=/resources/filetypes/other_cut.gif}
 * {@code filetype.UNKNOWN.mimeType=application/zip}
 *
 * {@code filetype.1.typeCode=TEXT}
 * {@code filetype.TEXT.description_pt_BR=Texto}
 * {@code filetype.TEXT.description_en_US=Text}
 * {@code filetype.TEXT.description_es_AR=Texto}
 * {@code filetype.TEXT.baseIcon=/resources/filetypes/text.gif}
 * {@code filetype.TEXT.ucIcon=/resources/filetypes/text_uc.gif}
 * {@code filetype.TEXT.cutIcon=/resources/filetypes/text_cut.gif}
 * {@code filetype.TEXT.mimeType=text/plain}
 * {@code filetype.TEXT.extensions=txt}
 * 
 * {@code filetype.2.typeCode=HTML} {@code filetype.HTML.description_pt_BR=HTML}
 * {@code filetype.HTML.description_en_US=HTML}
 * {@code filetype.HTML.description_es_AR=HTML}
 * {@code filetype.HTML.baseIcon=/resources/filetypes/html.gif}
 * {@code filetype.HTML.ucIcon=/resources/filetypes/html_uc.gif}
 * {@code filetype.HTML.cutIcon=/resources/filetypes/html_cut.gif}
 * {@code filetype.HTML.mimeType=text/html}
 * {@code filetype.HTML.extensions=html,htm}
 * 
 * Todas as chaves comeam por {@code filetype} seguidas por um nmero
 * sequencial ou o cdigo do tipo. O nmero sequencial comea por 0 e o
 * incremento  1. Logo no exemplo, temos 3 tipos cadastrados.
 * 
 * Formato Cdigo:  o formato cdigo que informam o cdigo de tipo de 
 * arquivo. Ele : {@code filetype.N.typeCode}, onde N  o nmero sequencial.
 * 
 * Exemplo: {@code TEXT}, que  o valor da chave {@code filetype.1.typeCode}, 
 * o cdigo de arquivos de projeto do tipo texto.
 * 
 * H 2 formatos possveis para as demais informaes. So eles:
 * <ul>
 * <li>
 * Formato Regular: {@code filetype.TYPE_CODE.INFO}, onde TYPE_CODE  o 
 * cdigo do tipo e INFO  o nome do atributo.
 * </li>
 * <li>
 * Formato com Idioma: {@code filetype.TYPE_CODE.INFO_LOCALE}, onde TYPE_CODE  
 * o cdigo do tipo, INFO  o nome do atributo e o LOCATE  o idioma. O idioma
 * </li>
 * </ul>
 * 
 * Informaes com Formato Regular:
 * <ul>
 * <li>baseIcon: informa o caminho para o arquivo contendo o cone base do tipo.
 *  obrigatria. Exemplo: {@code /resources/filetypes/text.gif}  o caminho
 * para o arquivo que  o cone base de arquivos tipo texto.</li>
 * <li>ucIcon: informa o caminho para o arquivo contendo o cone
 * "sob construo" do tipo.  obrigatria. Exemplo:
 * {@code /resources/filetypes/text_uc.gif}  o caminho para o arquivo que  o
 * cone "sob construo" de arquivos tipo texto.</li>
 * <li>cutIcon: informa o caminho para o arquivo contendo o cone "cortado" do
 * tipo.  obrigatria. Exemplo: {@code /resources/filetypes/text_uc.gif}  o
 * caminho para o arquivo que  o cone "cortado" de arquivos tipo texto.</li>
 * <li>mimeType: informa o MIME type para o tipo de arquivo.  opcional.
 * Exemplo: {@code text/html}  o MIME type para arquivos de projeto do tipo
 * HTML.</li>
 * <li>extensions: informa as extenses vlidas para o tipo de arquivo. O seu
 * valor  uma lista, cujos valores so separados por vrgula.  opcional.
 * Exemplo: {@code html} e {@code htm} so extenses para o arquivos tipo HTML.</li>
 * </ul>
 *
 * Informaes com Formato com Idioma:
 * <ul>
 * <li>
 * description_LOCALE: informa a descrio do tipo para um idioma especfico. 
 * obrigatria. Pode haver 1 ou mais. Haver uma chave para cada idioma.
 * Exemplo: {@code Texto}, que  o valor da chave
 * {@code filetype.TEXT.description_pt_BR},  a descrio para o arquivo tipo
 * texto, no idioma Portugus Brasileiro.
 * </li>
 * </ul>
 * 
 * As informaes para o tipo desconhecido {@link ProjectFileType#UNKNOWN} so
 * obrigatrias, ou seja, para que o repositrio funcione corretamente 
 * necessrio ter uma sesso semelhante a:
 * {@code filetype.0.typeCode=UNKNOWN}
 * {@code filetype.UNKNOWN.description_pt_BR=Desconhecido}
 * {@code filetype.UNKNOWN.description_en_US=Unknown}
 * {@code filetype.UNKNOWN.description_es_AR=Desconocido}
 * {@code filetype.UNKNOWN.baseIcon=/resources/filetypes/other.gif}
 * {@code filetype.UNKNOWN.ucIcon=/resources/filetypes/other_uc.gif}
 * {@code filetype.UNKNOWN.cutIcon=/resources/filetypes/other_cut.gif}
 * {@code filetype.UNKNOWN.mimeType=application/zip}
 * No  obrigatrio que este tipo seja o primeiro.
 * 
 * Na ausncia de informaes nos outros tipos, o sistema assume valores para
 * evitar a falha, porm gera aviso.
 * <ul>
 * <li>Na ausncia da descrio, o sistema utiliza o prprio cdigo do tipo;</li>
 * <li>Na ausncia do mime-type, o sistema utiliza o mime-type padro 
 * {@link #DEFAULT_MIME_TYPE} {@value #DEFAULT_MIME_TYPE};</li>
 * <li>Na ausncia de algum cone, o sistema utiliza o cone base do tipo
 * desconhecido {@link ProjectFileType#UNKNOWN}, na ausncia do cone base
 * do tipo desconhecido, ele utilizar um vetor 0 bytes.</li>
 * </ul>
 *
 * @author Tecgraf/PUC-Rio
 */
public final class ProjectFileTypeRepository {

  /**
   *  utilizado se o mime-type de um tipo no for informado.
   */
  static final String DEFAULT_MIME_TYPE = "application/octet-stream";

  private static final String PREFFIX_KEY = "filetype.%s.";
  private static final String CODE_KEY = PREFFIX_KEY + "typeCode";
  private static final String DESCRIPTION_KEY = PREFFIX_KEY + "description_%s";
  private static final String BASE_ICON_KEY = PREFFIX_KEY + "baseIcon";
  private static final String UNDER_CONSTRUCTION_ICON_KEY = PREFFIX_KEY
      + "ucIcon";
  private static final String CUT_ICON_KEY = PREFFIX_KEY + "cutIcon";
  private static final String MIME_TYPE_KEY = PREFFIX_KEY + "mimeType";
  private static final String EXTENSIONS_KEY = PREFFIX_KEY + "extensions";

  private static final String IS_DIRECTORY_KEY = PREFFIX_KEY + "isDirectory";

  /**
   * cone utilizado quando faltar informaes sobre um cone em algum tipo.
   */
  private byte [] defaultIcon;
  
  /**
   * Mapa que associa o idioma ao cdigo do tipo e aos tipos de arquivo de
   * projeto.
   */
  private Map<Locale, Map<String, ProjectFileTypeInfo>> infosByCodeByLocale;

  /**
   * O caminho para o arquivo que contm as informaes sobre os tipos de
   * arquivos de projeto.
   */
  private String path;

  /**
   * Cria um repositrio.
   * 
   * @param path
   *          O caminho para o arquivo que contm as informaes sobre os tipos
   *          de arquivos de projeto (No pode ser null).
   */
  public ProjectFileTypeRepository(String path) {
    infosByCodeByLocale = new HashMap<Locale, Map<String, ProjectFileTypeInfo>>();
    setPath(path);
  }

  /**
   * Obtm as informaes dos tipos de arquivos para um idioma especfico.
   * 
   * @param locale O idioma (no pode ser {@code null}).
   * 
   * @return Um mapa associativo que associa o cdigo e o tipo do arquivo.
   * O mapa retornado  imutvel.
   */
  public Map<String, ProjectFileTypeInfo> getInfos(Locale locale) {
    if (locale == null) {
      throw new IllegalArgumentException("O parmetro locale est nulo.");
    }

    Map<String, ProjectFileTypeInfo> infosByCode = infosByCodeByLocale
        .get(locale);
    if (infosByCode == null) {
      // Double check pattern para otimizar a sincronizao garantido 
      // a atomicidade do trecho de cdigo a seguir.
      synchronized (infosByCodeByLocale) {
        infosByCode = infosByCodeByLocale.get(locale);
        if (infosByCode == null) {
          infosByCode = loadInfos(locale);
          infosByCodeByLocale.put(locale, infosByCode);
        }
      }
    }
    return Collections.unmodifiableMap(infosByCode);
  }

  /**
   * Obtm informaes sobre o tipo de arquivo cujo cdigo foi informado, 
   * respeitando o idioma informado.
   * 
   * @param locale
   *          O idioma (No pode ser {@code null}).
   * @param code
   *          O cdigo. Se for {@code null} sero retornadas as informaes do
   *          tipo desconhecido ({@link ProjectFileType#UNKNOWN}).
   * 
   * @return As informaes do tipo de arquivo.
   * 
   * @throws ServiceFailureException
   *           Se houver um erro de E/S ao ler o arquivo {@code path}.
   *           Se for necessrio obter as informaes do tipo desconhecido 
   *           ({@link ProjectFileType#UNKNOWN}) e essas informaes no
   *           existirem.
   */
  public ProjectFileTypeInfo getInfo(Locale locale, String code) {
    Map<String, ProjectFileTypeInfo> infosByCode = getInfos(locale);

    if (code == null) {
      String msg = String.format(
          "Obtendo informaes sobre o tipo desconhecido (%s), j que foi "
              + "fornecido um cdigo null.", ProjectFileType.UNKNOWN);
      Server.logFineMessage(msg);
      code = ProjectFileType.UNKNOWN;
    }

    ProjectFileTypeInfo type = infosByCode.get(code);
    if (type == null) {
      String msg = String.format(
          "Obtendo informaes sobre o tipo desconhecido (%s), j que foi "
              + "fornecido o cdigo %s que no est presente no sistema.",
          ProjectFileType.UNKNOWN, code);
      Server.logFineMessage(msg);
      return getInfo(locale, ProjectFileType.UNKNOWN);
    }

    return type;
  }

  /**
   * Procura os tipos de arquivo que aceitam a extenso informada. 
   * 
   * @param locale
   *          O idioma (No pode ser {@code null}).
   * @param extension
   *          A extenso. (No pode ser {@code null}).
   * 
   * @return Um conjunto com os tipos de arquivos que aceitam a extenso.
   *         Caso nenhum tipo aceite a extenso, ser retornado um conjunto 
   *         vazio. O counjunto  imutvel.
   * 
   * @throws ServiceFailureException
   *           Se houver um erro de E/S ao ler o arquivo {@code path}.
   */
  public Set<ProjectFileTypeInfo> findInfosByExtension(Locale locale, String extension) {
    Map<String, ProjectFileTypeInfo> infosByCode = getInfos(locale);

    Set<ProjectFileTypeInfo> infos = new HashSet<ProjectFileTypeInfo>();
    
    for (ProjectFileTypeInfo info : infosByCode.values()) {
      if (info.getExtensions().contains(extension)) {
        infos.add(info);
      }
    }

    return Collections.unmodifiableSet(infos);
  }
  
  /**
   * Procura um tipo de arquivo que aceitam a extenso informada.
   * 
   * Se houver mais do que 1 tipo de arquivo que aceite a extenso, ele 
   * retornar 1 deles e registrar um log como aviso para alertar do
   * problema.
   * 
   * @param locale
   *          O idioma (No pode ser {@code null}).
   * @param extension
   *          A extenso. (No pode ser {@code null}).
   * 
   * @return Um conjunto com os tipos de arquivos que aceitam a extenso.
   *         Caso no haja extenso adequada, ele retornar o 
   *         {@link ProjectFileType#UNKNOWN Tipo Desconhecido}.
   * 
   * @throws ServiceFailureException
   *           Se houver um erro de E/S ao ler o arquivo {@code path}. Se for
   *           necessrio obter as informaes do tipo desconhecido (
   *           {@link ProjectFileType#UNKNOWN}) e essas informaes no
   *           existirem.
   */
  public ProjectFileTypeInfo findInfoByExtension(Locale locale, String extension) {
    Set<ProjectFileTypeInfo> infos = findInfosByExtension(locale, extension);
    if (infos.isEmpty()) {
      String msg =
        String.format("No h tipos cadastrados para a extenso '%s'."
            + "Ser utilizado o tipo 'Desconhecido'.",
            extension);
      Server.logWarningMessage(msg);
      return getInfo(locale, ProjectFileType.UNKNOWN);
    }

    if (infos.size() != 1) {
      String msg = String.format("H 2 ou mais tipos cadastrados para a extenso '%s'."
          + "So eles: %s."
          + "Ser retornado algum deles.",
          extension, infos);
      Server.logWarningMessage(msg);
    }

    ProjectFileTypeInfo info = infos.iterator().next();

    return info;
  }

  
  private void setPath(String path) {
    if (path == null) {
      throw new IllegalArgumentException("O parmetro path  nulo.");
    }
    this.path = path;
  }

  /**
   * Carrega as informaes sobre os tipos de arquivo de projeto para um locale
   * especfico.
   * 
   * L as informaes de tipos de arquivo de projetos. Quando ocorrer uma falha
   * ao ler essas informaes,  utilizada o mesmo tipo de informao do
   * tipo desconhecido ({@link ProjectFileType#UNKNOWN}. Por exemplo, 
   * se houver erro ao ler a descrio de um tipo,  utilizada a descrio do
   * tipo desconhecido.
   * 
   * @param locale
   *          O idioma (no pode ser {@code null}).
   *          
   * @return O mapa que associa os cdigos dos tipos aos tipos.
   * Se no houver tipos, retornar um mapa vazio.
   *
   * @throws ServiceFailureException
   *           Se no houver informaes sobre o tipo desconhecido
   *           ({@link ProjectFileType.UNKNOWN}).
   *           Se houver um erro de E/S ao ler o arquivo.
   */
  private Map<String, ProjectFileTypeInfo> loadInfos(Locale locale) {
    InputStream inputStream = getClass().getResourceAsStream(path);
    if (inputStream == null) {
      String msg = String.format(
          "No foi possvel encontrar o arquivo %s no classpath.", path);
      throw new ServiceFailureException(msg);
    }
    try {
      Properties properties = new Properties();
      try {
        properties.load(inputStream);
      } catch (IOException e) {
        String msg = String.format(
            "Erro de E/S ao ler o arquivo %s no classpath.", path);
        throw new ServiceFailureException(msg, e);
      }

      Map<String, ProjectFileTypeInfo> infosByCode = new HashMap<String, ProjectFileTypeInfo>();
      Set<String> codes = getCodes(properties);
      if (!codes.contains(ProjectFileType.UNKNOWN)) {
        String msg = String.format("No foi possvel encontrar informaes sobre o tipo %s no arquivo %s presente no classpath do sistema.\n",
        ProjectFileType.UNKNOWN, path);
        throw new ServiceFailureException(msg);
      }

      readDefaultIcon(properties);

      for (String code : codes) {
        infosByCode.put(code, readInfo(properties, code, locale));
      }
      return infosByCode;
    } finally {
      IOUtils.closeQuietly(inputStream);
    }
  }

  /**
   * Obtm todos os cdigos de tipo de arquivos de projeto.
   * 
   * As chaves comeam por {@code filetype} seguidas por um nmero sequencial. O
   * nmero sequencial comea por 0 e o incremento  1. O formato de chaves que
   * informam o cdigo de tipo de arquivo : {@code filetype.N.typeCode}, onde N
   *  o nmero sequencial.
   * 
   * @param properties
   *          As propriedades (no pode ser {@code null}).
   * 
   * @return Os cdigos. A lista retornada  imutvel. Se no houver cdigos, a
   *         lista retornada ser vazia. .
   */
  private Set<String> getCodes(Properties properties) {
    Set<String> codes = new HashSet<String>();
    int i = 0;
    String code = properties.getProperty(String.format(CODE_KEY, i));
    while (code != null) {
      codes.add(code);
      i++;
      code = properties.getProperty(String.format(CODE_KEY, i));
    }
    return Collections.unmodifiableSet(codes);
  }

  /**
   * L as informaes sobre um tipo de arquivo de projeto especfico.
   * 
   * Quando ocorrer uma falha ao ler essas informaes,  utilizada o mesmo tipo
   * de informao do tipo desconhecido ({@link ProjectFileType#UNKNOWN}.
   * Por exemplo, se houver erro ao ler a descrio de um tipo,  utilizada a 
   * descrio do tipo desconhecido.
   * 
   * @param properties
   *          As propriedades (no pode ser {@code null}).
   * @param code
   *          O cdigo (no pode ser {@code null}).
   * @param locale
   *          O locale (no pode ser {@code null}).
   * 
   * @return As informaes.
   * 
   * @throws ServiceFailureException
   *           Se houver um erro ao ler as informaes do tipo desconhecido.
   */
  private ProjectFileTypeInfo readInfo(Properties properties, String code,
      Locale locale) {
    String description = getDescription(properties, code, locale);
    if (description.isEmpty()) {
      String msg = String.format(
        "Faltando a descrio para o tipo %s."
        + "Usando o prprio cdigo do tipo como descrio."
        + "Corrija o arquivo %s disponvel no classpath do Sistema.", code, path);
      Server.logWarningMessage(msg);
      description = code;
    }

    String mimeType = getMimeType(properties, code);
    if (mimeType.isEmpty()) {
      String msg = String.format(
          "Faltando o mime-type para o tipo %s."
          + "Usando o prprio o mime-type padro %s."
          + "Corrija o arquivo %s disponvel no classpath do Sistema.", code,
          DEFAULT_MIME_TYPE, path);
      Server.logWarningMessage(msg);
      mimeType = DEFAULT_MIME_TYPE;
    }

    byte[] baseIcon = readBaseIcon(properties, code);
    if (baseIcon.length == 0) {
      String msg = String.format(
          "Faltando o cone base para o tipo %s."
          + "Usando o cone base do tipo %s.."
          + "Corrija o arquivo %s disponvel no classpath do Sistema.", code,
          ProjectFileType.UNKNOWN, path);
      Server.logWarningMessage(msg);
      baseIcon = Arrays.copyOf(defaultIcon, defaultIcon.length);
    }

    byte[] ucIcon = readUnderConstructionIcon(properties, code);
    if (ucIcon.length == 0) {
      String msg = String.format(
          "Faltando o cone em construo para o tipo %s."
          + "Usando o cone base do tipo %s.."
          + "Corrija o arquivo %s disponvel no classpath do Sistema.", code,
          ProjectFileType.UNKNOWN, path);
      Server.logWarningMessage(msg);
      ucIcon = Arrays.copyOf(defaultIcon, defaultIcon.length);
    }

    byte[] cutIcon = readCutIcon(properties, code);
    if (cutIcon.length == 0) {
      String msg = String.format(
          "Faltando o cone cortar para o tipo %s."
          + "Usando o cone base do tipo %s.."
          + "Corrija o arquivo %s disponvel no classpath do Sistema.", code,
          ProjectFileType.UNKNOWN, path);
      Server.logWarningMessage(msg);
      cutIcon = Arrays.copyOf(defaultIcon, defaultIcon.length);
    }

    Set<String> extensions = getExtensions(properties, code);

    boolean isDirectory = isDirectory(properties, code);

    return new ProjectFileTypeInfo(code, description, mimeType, baseIcon,
        ucIcon, cutIcon, extensions, isDirectory);
  }

  /**
   * Indica se o tipo de projeto especfico  aplicavel a diretrios ou a
   * arquivos.
   * 
   * O formato da chave da propriedade 
   * {@code filetype.CODE.isDirectory}, onde {@code CODE} deve ser
   * substitudo pelo valor do parmetro {@code code}.
   * 
   * Exemplo: {@code filetype.TEXT.isDirectory=false}  par
   * chave-valor que informa que o tipo arquivo texto  aplicavel a arquivos.
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * 
   * @return {@code true} se for aplicavel a diretrios ou {@code false}
   * se for aplicavel a arquivos.
   * Se a chave no existir, assume que  aplicavel a arquivos, por questes
   * de compatibilidade com o legado.
   */
  private boolean isDirectory(Properties properties, String code) {
    String text = getText(properties, code, IS_DIRECTORY_KEY);
    if (text.isEmpty()) {
      String message = String.format("No propriedade %s para o tipo %s, "
          + "logo assumindo que o tipo  aplicavel a arquivos.",
          IS_DIRECTORY_KEY, code);
      Server.logWarningMessage(message);
      return false;
    }

    if (text.equals(Boolean.TRUE.toString())) {
      return true;
    }

    if (text.equals(Boolean.FALSE.toString())) {
      return false;
    }

    String message = String.format("O valor '%s' da propriedade %s para o "
        + "tipo %s no  um valor booleano vlido, "
       + "assumindo falso, logo assumindo que o tipo  aplicavel a arquivos. "
       + "Valores vlidos: %s ou %s.",
       text, IS_DIRECTORY_KEY, code, Boolean.TRUE, Boolean.FALSE);
    Server.logWarningMessage(message);
    return false;
  }

  /**
   * Obtm a descrio de um tipo de arquivo de projeto especfico.
   * 
   * O formato da chave da propriedade 
   * {@code filetype.CODE.description_LOCALE}, onde {@code CODE} deve ser
   * substitudo pelo valor do parmetro {@code code} e {@code LOCALE} deve ser
   * substituido por {@link Locale#toString()}.
   * 
   * Exemplo: {@code filetype.TEXT.description_pt_BR=Arquivo Texto}  par
   * chave-valor que informa que a descrio em Portugus do Brasil para o tipo
   * de arquivo texto  "Arquivo Texto".
   * 
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * @param locale
   *          O idioma (no pode ser {@code null}).
   * 
   * @return A descrio.
   * Retornar uma string vazia se no for informado o valor da propriedade ou 
   * ele estiver em branco.
   */
  private String getDescription(Properties properties, String code,
      Locale locale) {
    return getText(properties, code, locale, DESCRIPTION_KEY);
  }

  /**
   * Obtm o mime-type de um tipo de arquivo de projeto especfico.
   * 
   * O formato da chave da propriedade  {@code filetype.CODE.mimeType}, onde
   * {@code CODE} deve ser substitudo pelo valor do parmetro {@code code}.
   * 
   * Exemplo: {@code filetype.HTML.mimeType=text/html}  par chave-valor que
   * informa que o mime-type para o tipo de arquivo HTML  "text/html".
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * 
   * @return O mime-type.
   * Retornar uma string vazia se no for informado o valor da propriedade ou 
   * ele estiver em branco.
   */
  private String getMimeType(Properties properties, String code) {
    return getText(properties, code, MIME_TYPE_KEY);
  }

  /**
   * L o cone base ({@link ProjectFileTypeInfo#getBaseIcon()}) de um tipo de
   * arquivo de projeto especfico.
   * 
   * O formato da chave da propriedade  {@code filetype.CODE.baseIcon}, onde
   * {@code CODE} deve ser substitudo pelo valor do parmetro {@code code}. O
   * valor dessa chave  o caminho na classpath para o arquivo do cone.
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * 
   * @return O cone.
   * Retornar um vetor vazio se qualquer uma das condies a seguir ocorrer:
   * <ul>
   * <li>No foi informado o valor da propriedade.</li>
   * <li>O valor da propriedade est em branco.</li>
   * <li>O arquivo do cone no existe.</li>
   * <li>O arquivo do cone est vazio.</li>
   * <li>Erro de E/S ao ler o arquivo do cone.</li>
   * </ul>
   */
  private byte[] readBaseIcon(Properties properties, String code) {
    return readIcon(properties, code, BASE_ICON_KEY);
  }

  /**
   * L o cone "sob construo" (
   * {@link ProjectFileTypeInfo#getUnderConstructionIcon()}) de um tipo de
   * arquivo de projeto especfico.
   * 
   * O formato da chave da propriedade  {@code filetype.CODE.ucIcon}, onde
   * {@code CODE} deve ser substitudo pelo valor do parmetro {@code code}. O
   * valor dessa chave  o caminho na classpath para o arquivo do cone.
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * 
   * @return O cone.
   * Retornar um vetor vazio se qualquer uma das condies a seguir ocorrer:
   * <ul>
   * <li>No foi informado o valor da propriedade.</li>
   * <li>O valor da propriedade est em branco.</li>
   * <li>O arquivo do cone no existe.</li>
   * <li>O arquivo do cone est vazio.</li>
   * <li>Erro de E/S ao ler o arquivo do cone.</li>
   * </ul>
   */
  private byte[] readUnderConstructionIcon(Properties properties, String code) {
    return readIcon(properties, code, UNDER_CONSTRUCTION_ICON_KEY);
  }

  /**
   * L o cone "cortar" ({@link ProjectFileTypeInfo#getCutIcon()}) de um tipo
   * de arquivo de projeto especfico.
   * 
   * O formato da chave da propriedade  {@code filetype.CODE.cutIcon}, onde
   * {@code CODE} deve ser substitudo pelo valor do parmetro {@code code}. O
   * valor dessa chave  o caminho na classpath para o arquivo do cone.
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * 
   * @return O cone.
   * Retornar um vetor vazio se qualquer uma das condies a seguir ocorrer:
   * <ul>
   * <li>No foi informado o valor da propriedade.</li>
   * <li>O valor da propriedade est em branco.</li>
   * <li>O arquivo do cone no existe.</li>
   * <li>O arquivo do cone est vazio.</li>
   * <li>Erro de E/S ao ler o arquivo do cone.</li>
   * </ul>
   */
  private byte[] readCutIcon(Properties properties, String code) {
    return readIcon(properties, code, CUT_ICON_KEY);
  }

  /**
   * L o cone padro.
   * 
   * O cone padro  o cone base do tipo desconhecido
   * {@link ProjectFileType#UNKNOWN}.
   * 
   * Se o cone base do tipo desconhecido no estiver presente, o cone padro
   * ser um vetor com 0 bytes.
   */
  private void readDefaultIcon(Properties properties) {
    byte [] icon = readBaseIcon(properties, ProjectFileType.UNKNOWN);
    if (icon.length == 0) {
      String msg = String.format("No foi possvel encontrar o cone base %s "
          + "sobre o tipo %s no arquivo %s presente no classpath do sistema. "
          + "Se algum tipo no tiver um dos cones definidos, o cone no "
          + "aparecer no sistema. Corrija o arquivo.\n",
      BASE_ICON_KEY, ProjectFileType.UNKNOWN, path);
      Server.logWarningMessage(msg);
    }
    defaultIcon = icon;
  }

  /**
   * Obtm as extenses de um tipo de arquivo de projeto especfico.
   * 
   * O formato da chave da propriedade  {@code filetype.CODE.extensions}, onde
   * {@code CODE} deve ser substitudo pelo valor do parmetro {@code code}.
   * 
   * As extenses devem estar separadas por vrgula.
   * 
   * Exemplo: {@code filetype.HTML.extensions=html,htm}  par chave-valor que
   * informa que as extenses para o tipo de arquivo HTML so html e htm.
   * 
   * @param properties
   *          As propriedades contendo todas as informaes dos tipos de
   *          arquivos de projeto (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo (no pode ser {@code null}).
   * 
   * @return As extenses. O conjunto retornado  imutvel. Essa propriedade 
   *         opcional, logo se no estiver presente retornar um conjunto vazio.
   */
  private Set<String> getExtensions(Properties properties, String code) {
    String text = getText(properties, code, EXTENSIONS_KEY);

    StringTokenizer tokenizer = new StringTokenizer(text, ",");
    Set<String> extensions = new HashSet<String>();
    while (tokenizer.hasMoreTokens()) {
      String token = tokenizer.nextToken();
      token = token.trim();
      extensions.add(token);
    }

    return Collections.unmodifiableSet(extensions);
  }

  /**
   * Obtm o valor de uma propriedade especfica para um tipo de arquivo
   * especfico e um idioma especfico.
   * 
   * A chave da propriedade  formada usando o formato, o cdigo e o idioma.
   * 
   * @param properties
   *          As propriedades (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo especfico (no pode ser {@code null}
   *          ).
   * @param locale
   *          O idioma (no pode ser {@code null}).
   * @param format
   *          O formato da chave da propriedade (no pode ser {@code null}).
   * 
   * @return O valor da propriedade. Se o valor no existir ele retornar 
   * uma string vazia.
   */
  private String getText(Properties properties, String code, Locale locale,
      String format) {
    String key = String.format(format, code, locale);
    return getText(properties, key);
  }

  /**
   * Obtm o valor de uma propriedade especfica para um tipo de arquivo
   * especfico.
   * 
   * A chave da propriedade  formada usando o formato e o cdigo.
   * 
   * @param properties
   *          As propriedades (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo especfico (no pode ser {@code null}
   *          ).
   * @param format
   *          O formato da chave da propriedade (no pode ser {@code null}).
   * 
   * @return O valor da propriedade. Se o valor no existir ele retornar 
   *         uma string vazia.
   */
  private String getText(Properties properties, String code, String format) {
    String key = String.format(format, code);
    return getText(properties, key);
  }

  /**
   * Obtm o valor de uma propriedade.
   * 
   * A chave da propriedade  formada usando o formato e o cdigo.
   * 
   * @param properties
   *          As propriedades (no pode ser {@code null}).
   * @param key
   *          A chave da propriedade (no pode ser {@code null}).
   * 
   * @return O valor da propriedade. Se o valor no existir e a propriedade for
   *         opcional ento ele retornar uma string vazia.
   */
  private String getText(Properties properties, String key) {
    String value = properties.getProperty(key);
    if (value == null) {
      String msg =
        String.format("No foi possvel encontrar a chave %s no arquivo %s.", key, path);
      Server.logWarningMessage(msg);
      return "";
    }
    value = value.trim();
    if (value.isEmpty()) {
      String msg =
          String.format("O valor da chave %s no arquivo %s est vazio.", key, path);
        Server.logWarningMessage(msg);
    }
    return value;
  }

  /**
   * L o cone referenciado por uma propriedade.
   * 
   * A chave da propriedade  formada usando o formato e o cdigo.
   * 
   * O valor da propriedade  o caminho para o arquivo do cone e deve estar 
   * acessvel utilizando o classpath .
   * 
   * @param properties
   *          As propriedades (no pode ser {@code null}).
   * @param code
   *          O cdigo do tipo de arquivo especfico (no pode ser {@code null}
   *          ).
   * @param format
   *          O formato da chave da propriedade (no pode ser {@code null}).
   * @param isOptional
   *          {@code true} se a propriedade for opcional ou {@code false}
   *          obrigatria.
   * 
   * @return O cone.
   * Retornar um vetor vazio se qualquer uma das condies a seguir ocorrer:
   * <ul>
   * <li>No foi informado o valor da propriedade.</li>
   * <li>O valor da propriedade est em branco.</li>
   * <li>O arquivo do cone no existe.</li>
   * <li>O arquivo do cone est vazio.</li>
   * <li>Erro de E/S ao ler o arquivo do cone.</li>
   * </ul>
   */
  private byte[] readIcon(Properties properties, String code, String format) {
    String key = String.format(format, code);
    String value = getText(properties, key);
    if (value.isEmpty()) {
      return new byte[]{};
    }
    InputStream inputStream = getClass().getResourceAsStream(value);
    if (inputStream == null) {
      String msg = String.format(
        "O arquivo %s que  mencionando como valor da chave %s do arquivo %s "
              + "no est no classpath.", value, key, path);
      Server.logWarningMessage(msg);
      return new byte[]{};
    }
    try {
      try {
        return IOUtils.toByteArray(inputStream);
      } catch (IOException e) {
        String msg = String.format(
            "Erro de E/S ao ler arquivo %s que  mencionando como valor da "
                + "chave %s do arquivo %s do classpath.", value, key, path);
        Server.logWarningMessage(msg);
        return new byte[]{};
      }
    } finally {
      IOUtils.closeQuietly(inputStream);
    }
  }
}
