package csbase.servlet;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.URL;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import csbase.logic.url.URLParameters;

/**
 * Servlet de inicializao.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class InitHtmlServlet extends AbstractServlet {

  /**
   * Parmetro para check de verso.
   */
  private static final String VERSION_CHECK_PARAMETER = "version_check";

  /**
   * Parmetro da pgina.
   */
  private static final String PAGE_PARAMETER = "page";

  /**
   * Verso da JRE
   */
  private static final String JRE_VERSION_PARAMETER = "jre_version";

  /**
   * Pgina de erro.
   */
  private static final String ERROR_PAGE_PARAMETER = "error_page";

  /**
   * Destino (pgina)
   */
  private static final String DESTINATION = "destination_page";

  /**
   * Constri uma pgina HTML de entrada do sistema. A pgina construda  uma
   * cpia da pgina original, incluindo aprmetros adicionais para o applet. A
   * pgina original deve existir na mesma URL base deste servlet. O nome da
   * pgina pode ser fornecido atravs do parmetro page. Se o nome no for
   * fornecido,  assumido index.html. A pgina construida por este servlet
   * inclui um parmetro para o applet indicando o endereo do cliente que fez a
   * requisio. Alm disso, todos os parmetros recebidos pelo servlet tambm
   * so repassados para o applet,  exceo do parmetro page.
   * 
   * @param request requisio.
   * @param response resposta
   * 
   * @throws IOException em caso de falha de I/O.
   * @throws ServletException em caso de falha no servlet.
   */
  @Override
  protected void doGet(HttpServletRequest request, HttpServletResponse response)
    throws IOException, ServletException {
    /* TODO Substituir Hashtable por Map */
    Hashtable<String, String> parameters = createParameterMap(request);
    if (parameters.containsKey(VERSION_CHECK_PARAMETER)) {
      parameters.remove(VERSION_CHECK_PARAMETER);
      String version =
        getServletContext().getInitParameter(JRE_VERSION_PARAMETER);
      String pageURL = getServletURL(request);
      String errorPageParameter = parameters.remove(ERROR_PAGE_PARAMETER);
      if (errorPageParameter == null) {
        errorPageParameter = "error.jsp";
      }
      int lastSlash = pageURL.lastIndexOf("/");
      String errorPage =
        pageURL.substring(0, lastSlash + 1) + errorPageParameter + "?"
          + JRE_VERSION_PARAMETER + "=" + version;
      Enumeration<String> e = parameters.keys();
      String destinationPage = pageURL;
      String separator = "?";
      while (e.hasMoreElements()) {
        String parameterName = e.nextElement();
        String parameterValue = parameters.get(parameterName);
        destinationPage =
          destinationPage + separator + parameterName + "=" + parameterValue;
        separator = "&";
      }
      Hashtable<String, String> appletParameters =
        new Hashtable<String, String>();
      appletParameters.put(DESTINATION, destinationPage);
      appletParameters.put(ERROR_PAGE_PARAMETER, errorPage);
      appletParameters.put(JRE_VERSION_PARAMETER, version);
      String pageContent =
        getPage(request, pageURL + "?page=versioncheck.html", appletParameters);
      pageContent = insertScriptParameter(pageContent, errorPage);
      printPage(response, pageContent);
    }
    else {
      final String clientAddrParamName = URLParameters.CLIENT_ADDRESS_PARAMETER;
      parameters.put(clientAddrParamName, request.getRemoteAddr());
      parameters.remove(PAGE_PARAMETER);
      String page = getPageParameter(request);
      String pageURL = getServletURL(request);

      // Constri a mesma URL usada para acessar este servlet, trocando o nome
      // do arquivo (init.html) pelo nome da pgina original.
      int lastSlash = pageURL.lastIndexOf("/");
      pageURL = pageURL.substring(0, lastSlash + 1) + page;
      String pageContent = getPage(request, pageURL, parameters);
      printPage(response, pageContent);
    }
  }

  /**
   * Impresso da pgina
   * 
   * @param response a resposta.
   * @param pageContent contedo.
   * @throws IOException em caso de falha de I/O.
   * @throws ServletException em caso de falha no servlet.
   */
  private void printPage(HttpServletResponse response, String pageContent)
    throws IOException, ServletException {
    response.setContentType("text/html");
    PrintWriter out = response.getWriter();
    try {
      out.println(pageContent);
      out.close();
    }
    catch (Exception e) {
      out.println("<html><head>Erro</head><body>");
      out.println("Erro!<br>");
      out.println("Por favor, entre em contato com o suporte do sistema,<br>");
      out.println("e fornea a seguinte mensagem:<br>");
      out.println(e.toString());
      out.println("</body></html>");
      out.close();
      throw new ServletException(e);
    }
  }

  /**
   * Cria hash de parmetros.
   * 
   * @param request requisio
   * @return a hash de parametrizao.
   */
  private Hashtable<String, String> createParameterMap(
    HttpServletRequest request) {
    Hashtable<String, String> parameters = new Hashtable<String, String>();
    Enumeration<?> e = request.getParameterNames();
    while (e.hasMoreElements()) {
      String name = (String) e.nextElement();
      parameters.put(name, request.getParameter(name));
      System.out.println(name + ": " + parameters.get(name));
    }
    return parameters;
  }

  /**
   * Obtm a pgina original.
   * 
   * @param request requisio
   * @param page pgina
   * @param parameters mapa de parmetros.
   * 
   * @return a pgina
   * 
   * @throws IOException em caso de falha de I/O.
   */
  private String getPage(HttpServletRequest request, String page,
    Hashtable<String, String> parameters) throws IOException {
    URL url = new URL(page);
    InputStream is = url.openStream();
    InputStreamReader isr = new InputStreamReader(is);
    BufferedReader br = new BufferedReader(isr);
    StringBuffer buffer = new StringBuffer();
    String line = null;
    while ((line = br.readLine()) != null) {
      buffer.append(line).append("\n");
    }
    br.close();
    isr.close();
    is.close();
    String result = buffer.toString();
    return insertParameters(result, parameters);
  }

  /**
   * Obtm uma URL para acesso ao servlet, sem parmetros.
   * 
   * @param request requisio.
   * 
   * @return a URL (texto).
   */
  private String getServletURL(HttpServletRequest request) {
    String serverName = request.getServerName();
    String serverPort = Integer.toString(request.getServerPort());
    String requestURI = request.getRequestURI();
    return "http://" + serverName + ":" + serverPort + requestURI;
  }

  /**
   * Obtm o valor do parmetro definido em {@link #PAGE_PARAMETER}.
   * 
   * @param request requisio.
   * 
   * @return O valor do parmetro ou um valor default, caso no seja encontrado.
   */
  private String getPageParameter(HttpServletRequest request) {
    String page = request.getParameter(PAGE_PARAMETER);
    if (page == null) {
      page = "index.html";
    }
    return page;
  }

  /**
   * Insere os parmetros fornecidos na pgina a ser construda. Este mtodo
   * procura o final do primeiro tag applet da pgina fornecida e repassa esses
   * parmetros para o applet.
   * 
   * @param page pgina
   * @param parameters parmetros
   * 
   * @return texto.
   */
  private String insertParameters(String page,
    Hashtable<String, String> parameters) {
    int pos = page.toLowerCase().indexOf("</applet>");
    if (pos == -1) {
      return page;
    }
    StringBuffer buffer = new StringBuffer(page.substring(0, pos));
    for (Enumeration<String> e = parameters.keys(); e.hasMoreElements();) {
      String name = e.nextElement();
      String value = parameters.get(name);
      buffer.append("<param name=").append(name);
      buffer.append(" value=").append(value).append(">\n");
    }
    buffer.append(page.substring(pos));
    return buffer.toString();
  }

  /**
   * Insere o parmetro para o JavaScript que verifica a existncia da JVM no
   * cliente.
   * 
   * @param page pgina
   * @param parameter parmetro
   * 
   * @return texto de insero.
   */
  private String insertScriptParameter(String page, String parameter) {
    int pos = page.toLowerCase().indexOf("</script>");
    if (pos == -1) {
      return page;
    }
    StringBuffer buffer = new StringBuffer(page.substring(0, pos));
    buffer.append("\nparent.location=\"").append(parameter + "\"\n");
    buffer.append(page.substring(pos));
    return buffer.toString();
  }
}
