/**
 * $Id: PBSDriver.java 170939 2016-01-04 13:27:08Z ericaflr $
 */

package csbase.sga.ssh.pbs.lncc;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import sgaidl.COMMAND_CPU_TIME_SEC;
import sgaidl.COMMAND_EXEC_HOST;
import sgaidl.COMMAND_MEMORY_RAM_SIZE_MB;
import sgaidl.COMMAND_MEMORY_SWAP_SIZE_MB;
import sgaidl.COMMAND_PID;
import sgaidl.COMMAND_STATE;
import sgaidl.COMMAND_WALL_TIME_SEC;
import sgaidl.ProcessState;
import csbase.sga.executor.JobData;
import csbase.sga.executor.JobInfo;
import csbase.sga.monitor.SGAInfo;
import csbase.sga.ssh.SGADriver;

/**
 * @author Tecgraf
 */
public class PBSDriver implements SGADriver {
	private static final String QUEUE_KEY = "sga_pbs_queue";
	//PBS resource list, eg: nodes=2:ppn=8
	private static final String RESOURCE_LIST = "sga_pbs_resource_list";	
	private static final String ENVIRONMENT_CMD = "sga_pbs_environment_cmd";

	private static final String SUBMIT_JOB = "qsub {0} {1}";
	private static final String CHECK_JOB = "qstat -x {0};echo \"{1}\";tracejob -v {2}";
	private static final String KILL_JOB = "qdel {0}";
	private static final String CHECK_ALL_NODES_XML = "pbsnodes -x";
	private static final String CHECK_ALL_JOBS_XML = "qstat -x";

	private static final Pattern JOB_ID_PATTERN = Pattern.compile("([^\\.]+)\\.(.*)\n");
	private static final Pattern TRACEJOB_JOB_PROPERTIES_PATTERN = Pattern.compile("\\s{1}([^\\s]+)=([^\\s]+)\\s{1}");
	private static final Pattern TRACEJOB_JOB_ID_PATTERN = Pattern.compile("pjob: (\\d+)");
	private static final String CHECK_JOB_CMD_SEPARATOR = "%%";

	private Properties properties;

	@Override
	public void init(Properties properties) {
		this.properties = properties;
	}

	private String configureEnvironment() {
		if (properties.containsKey(ENVIRONMENT_CMD)) {
			return properties.getProperty(ENVIRONMENT_CMD) + ";";
		}
		return "";
	}

	@Override
	public String buildSubmitJobCommand(String script, Map<String, String> extraParam) {
		String cmd = configureEnvironment() + SUBMIT_JOB;
		String extra = "";
		if (properties.containsKey(QUEUE_KEY)) {
			extra += " -q " + properties.get(QUEUE_KEY);
		}
		if(properties.contains(RESOURCE_LIST)) {
			extra += " -l " + properties.get(RESOURCE_LIST);
		}
		String[] splitedScript = script.split("\\s");

		return MessageFormat.format(cmd, extra, splitedScript[1]);
	}

	@Override
	public String buildKillJobCommand(JobData jobData) {
		String cmd = configureEnvironment() + KILL_JOB;
		PBSJobData data = (PBSJobData) jobData;
		return MessageFormat.format(cmd, data.getJobId());
	}

	@Override
	public String buildCheckJobCommand(JobData jobData) {
		String cmd = configureEnvironment() + CHECK_JOB;
		PBSJobData data = (PBSJobData) jobData;
		return MessageFormat.format(cmd, data.getJobId(), CHECK_JOB_CMD_SEPARATOR, data.getJobId());
	}

	@Override
	public String buildCheckAllJobsCommand() {
		return null;
	}

	@Override
	public JobData parseJobSubmissionOutput(String output) {
		Matcher matcher = JOB_ID_PATTERN.matcher(output);
		if (matcher.matches()) {
			return new PBSJobData(matcher.group(1));
		} else {
			return null;
		}
	}

	@Override
	public Map<JobData, JobInfo> parseCheckJobOutput(String output) {
		return parseXMLJobOutput(output);
	}

	private Map<JobData, JobInfo> parseXMLJobOutput(String output) {
		Map<JobData, JobInfo> jobsInfo = new HashMap<JobData, JobInfo>();
		if (output.isEmpty()) // n�o h� jobs
			return jobsInfo;
		// retrieve the output of qstat
		String[] data = output.split(CHECK_JOB_CMD_SEPARATOR);
		String qstat = data[0];
		String tracejob = data[1];
		{
			byte[] xmlbytes = qstat.getBytes();
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			try {
				DocumentBuilder builder = factory.newDocumentBuilder();
				ByteArrayInputStream input = new ByteArrayInputStream(xmlbytes);
				Document doc = builder.parse(input);
				doc.getDocumentElement().normalize();
				NodeList nList = doc.getElementsByTagName("Job");

				for (int temp = 0; temp < nList.getLength(); temp++) {
					Map<String, String> attrsMap = new HashMap<String, String>();
					Node nNode = nList.item(temp);
					if (nNode.getNodeType() == Node.ELEMENT_NODE) {
						Element jobElem = (Element) nNode;
						splitJobId(jobElem.getTextContent(), attrsMap);
						fillJobProperties(jobElem, attrsMap);
						JobInfo job = convertToJobInfo(attrsMap);
						jobsInfo.put(new PBSJobData(attrsMap.get("pid")), job);
					}
				}

			} catch (ParserConfigurationException e) {
				// do nothing
			} catch (SAXException e) {
				// do nothing
			} catch (IOException e) {
				// do nothing
			}
		}
		{
			String jobId = "";
			Matcher jobIdMatcher = TRACEJOB_JOB_ID_PATTERN.matcher(tracejob);
			if (jobIdMatcher.find()) {
				jobId = jobIdMatcher.group(1);
			}
			Matcher matcher = TRACEJOB_JOB_PROPERTIES_PATTERN.matcher(tracejob);
			int start = 0;
			Map<String, String> properties = new HashMap<String, String>();
			while (matcher.find(start)) {
				String key = matcher.group(1);
				String value = matcher.group(2);
				properties.put(key, value);
				start = matcher.start() + 1;
			}

			PBSJobData jobData = new PBSJobData(jobId);
			JobInfo jobInfo = jobsInfo.get(jobData);
			if (jobInfo == null) {
				jobInfo = new JobInfo();
				jobInfo.jobParam.put(COMMAND_PID.value, jobId);
				jobInfo.jobParam.put(COMMAND_STATE.value, String.valueOf(ProcessState.FINISHED));
				String host = properties.get("exec_host");
				jobInfo.jobParam.put(COMMAND_EXEC_HOST.value, (host == null) ? "" : host);
				String ctimeS = properties.get("resources_used.cput");
				String ramS = properties.get("resources_used.mem");
				String swapS = properties.get("resources_used.vmem");
				String walltimeS = properties.get("resources_used.walltime");

				long ram = (ramS != null) ? getInfoInMb(ramS) : -1;
				long swap = (swapS != null) ? getInfoInMb(swapS) : -1;
				long ctime = Long.parseLong(ctimeS);
				long walltime = Long.parseLong(walltimeS);

				jobInfo.jobParam.put(COMMAND_MEMORY_RAM_SIZE_MB.value, Long.toString(ram));
				jobInfo.jobParam.put(COMMAND_MEMORY_SWAP_SIZE_MB.value, Long.toString(swap));
				jobInfo.jobParam.put(COMMAND_CPU_TIME_SEC.value, Long.toString(ctime));
				jobInfo.jobParam.put(COMMAND_WALL_TIME_SEC.value, Long.toString(walltime));
				jobsInfo.put(jobData, jobInfo);
			}
		}
		return jobsInfo;
	}

	private void fillJobProperties(Element jobElem, Map<String, String> attrsMap) {
		int resources_usedCounter = 0;
		NodeList atts = jobElem.getElementsByTagName("*");
		for (int i = 0; i < atts.getLength(); i++) {
			// No caso do Torque temos a chave "resources_used"
			// sobrescrita; temos que reconhecer os recursos
			// utilizados uma a um.
			if (atts.item(i).getNodeName().equals("resources_used")) {
				resources_usedCounter++;
				attrsMap.put(atts.item(i).getNodeName() + resources_usedCounter, atts.item(i).getTextContent());
			} else
				attrsMap.put(atts.item(i).getNodeName(), atts.item(i).getTextContent());
		}
	}

	private void splitJobId(String jobId, Map<String, String> attrsMap) {
		Pattern JOBID_PATTERN = Pattern.compile("([^.]+).(\\S+)");
		Matcher attrsMatcher = JOBID_PATTERN.matcher(jobId);
		attrsMatcher.find();
		attrsMap.put("pid", attrsMatcher.group(1));
	}

	private JobInfo convertToJobInfo(Map<String, String> jobMap) {
		JobInfo jobInfo = new JobInfo();
		jobInfo.jobParam.put(COMMAND_PID.value, jobMap.get("pid"));
		jobInfo.jobParam.put(COMMAND_STATE.value, convertJobState(jobMap.get("job_state")).toString());
		String host = jobMap.get("exec_host");
		jobInfo.jobParam.put(COMMAND_EXEC_HOST.value, (host == null) ? "" : host);
		retrieveResourcesUsed(jobInfo, jobMap);
		return jobInfo;
	}

	private ProcessState convertJobState(String state) {
		switch (state) {

		case "C": // Job is completed after having run
			return ProcessState.FINISHED;
		case "E": // Job is exiting after having run
			return ProcessState.FINISHED;
		case "H": // Job is held
			return ProcessState.SLEEPING;
		case "Q": // job is queued, eligible to run or routed
			return ProcessState.WAITING;
		case "R": // job is running
			return ProcessState.RUNNING;
		case "T": // job is being moved to new location
			return ProcessState.RUNNING;
		case "W": // job is waiting for its execution time (-a option) to be
			// reached.
			return ProcessState.SLEEPING;
		case "S": // (Unicos only) job is suspend
			return ProcessState.FINISHED;
		default:
			return ProcessState.WAITING;
		}
	}

	private void retrieveResourcesUsed(JobInfo jobInfo, Map<String, String> jobMap) {
		// 1 -> cpu time
		// 2 -> RAM used
		// 3 -> Swap used
		// 4 -> walltime
		String ctimeS = jobMap.get("cput");
		String ramS = jobMap.get("mem");
		String swapS = jobMap.get("vmem");
		String walltimeS = jobMap.get("walltime");

		long ram = (ramS != null) ? getInfoInMb(ramS) : -1;
		long swap = (swapS != null) ? getInfoInMb(swapS) : -1;
		long ctime = (ctimeS != null) ? getInfoInSec(ctimeS) : -1;
		long walltime = (walltimeS != null) ? getInfoInSec(walltimeS) : -1;

		jobInfo.jobParam.put(COMMAND_MEMORY_RAM_SIZE_MB.value, Long.toString(ram));
		jobInfo.jobParam.put(COMMAND_MEMORY_SWAP_SIZE_MB.value, Long.toString(swap));
		jobInfo.jobParam.put(COMMAND_CPU_TIME_SEC.value, Long.toString(ctime));
		jobInfo.jobParam.put(COMMAND_WALL_TIME_SEC.value, Long.toString(walltime));
	}

	@Override
	public String buildCheckEnvironmentCommand() {
		String cmd = configureEnvironment() + CHECK_ALL_NODES_XML;
		if (properties.containsKey(QUEUE_KEY)) {
			cmd += " :" + properties.get(QUEUE_KEY);
		}
		return cmd;
	}

	@Override
	public SGAInfo parseCheckEnvironmentOutput(String output) {
		SGAInfo sga = new SGAInfo(properties);
		byte[] xmlbytes = output.getBytes();
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();

			ByteArrayInputStream input = new ByteArrayInputStream(xmlbytes);
			Document doc = builder.parse(input);
			doc.getDocumentElement().normalize();
			NodeList nList = doc.getElementsByTagName("Node");

			for (int temp = 0; temp < nList.getLength(); temp++) {
				Node nNode = nList.item(temp);
				PBSNodeInfo pbsnode = new PBSNodeInfo();
				if (nNode.getNodeType() == Node.ELEMENT_NODE) {
					Element nodeElem = (Element) nNode;
					NodeList atts = nodeElem.getElementsByTagName("*");
					for (int i = 0; i < atts.getLength(); i++)
						pbsnode.attMap.put(atts.item(i).getNodeName(), atts.item(i).getTextContent());
					splitInfo(pbsnode);
				}

				long ramMem = pbsnode.physmemkb;
				long swapMem = pbsnode.totmemkb - pbsnode.physmemkb;
				long availableRam = pbsnode.availmemkb >= (swapMem) ? pbsnode.availmemkb - swapMem : 0;
				long availableSwap = pbsnode.availmemkb - availableRam;
				double freeRamPerc = availableRam * 100 / ramMem;
				double freeSwapPerc = availableSwap * 100 / swapMem;

				sga.addNode(pbsnode.attMap.get("name"), null, pbsnode.attMap.get("ncpus"), "0", Long.toString(ramMem), Long.toString(swapMem), Double.toString(freeRamPerc), Double.toString(freeSwapPerc), pbsnode.attMap.get("loadave"), pbsnode.attMap.get("loadave"), pbsnode.attMap.get("loadave"), Integer.toString(pbsnode.njobs));
			}
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return sga;
	}

	private void splitInfo(PBSNodeInfo pbsnode) {
		Pattern NODES_ATTRIBUTES_PATTERN = Pattern.compile("([^,=]+)=([^,]+)");

		// n�mero de jobs tem que ser contado antes de separar os atributos de
		// status, se n�o o valor ser� sobrescrito.
		pbsnode.njobs = countJobs(pbsnode);

		Matcher attrsMatcher = NODES_ATTRIBUTES_PATTERN.matcher(pbsnode.attMap.get("status"));
		while (attrsMatcher.find()) {
			pbsnode.attMap.put(attrsMatcher.group(1), attrsMatcher.group(2));
		}

		pbsnode.totmemkb = getInfoInMb(pbsnode.attMap.get("totmem"));
		pbsnode.availmemkb = getInfoInMb(pbsnode.attMap.get("availmem"));
		pbsnode.physmemkb = getInfoInMb(pbsnode.attMap.get("physmem"));
	}

	private int countJobs(PBSNodeInfo pbsnode) {
		Pattern JOB_IDS_PATTERN = Pattern.compile("([^,]+)");
		if (pbsnode.attMap.containsKey("jobs")) {
			Matcher jobsMatcher = JOB_IDS_PATTERN.matcher(pbsnode.attMap.get("jobs"));
			int jobCounter = 0;
			while (jobsMatcher.find())
				jobCounter++;
			return jobCounter;
		}
		return 0;
	}

	private long getInfoInMb(String string) {
		Pattern BYTE_INFO_PATTERN = Pattern.compile("(\\d+)(\\S+)");
		Matcher attrsMatcher = BYTE_INFO_PATTERN.matcher(string);
		attrsMatcher.find();
		long value = Long.parseLong(attrsMatcher.group(1));
		String measure = attrsMatcher.group(2);
		switch (measure) {
		case "b":
			value /= 1000000;
			break;
		case "kb":
			value /= 1000;
			break;
		case "mb":
			break;
		case "gb":
			value *= 1000;
			break;
		case "tb":
			value *= 1000000;
			break;
		default:
			value = 0;
			break;

		}

		return value;
	}

	private long getInfoInSec(String string) {
		// TODO: verificar como o Torque exibe jobs que rodam h� pelo menos 1
		// dia
		Pattern BYTE_INFO_PATTERN = Pattern.compile("(\\d+):(\\d+):(\\d+)");
		Matcher attrsMatcher = BYTE_INFO_PATTERN.matcher(string);
		attrsMatcher.find();
		return Long.parseLong(attrsMatcher.group(1)) * 3600 + Long.parseLong(attrsMatcher.group(2)) * 60 + Long.parseLong(attrsMatcher.group(3));
	}

	private class PBSNodeInfo {
		public int njobs;
		public long totmemkb;
		public long availmemkb;
		public long physmemkb;
		public Map<String, String> attMap = new HashMap<String, String>();

	}

	@Override
	public Map<JobData, JobInfo> parseCheckAllJobsOutput(String output) {
		return null;
	}

}
