/**
 * $Id: SSHExecutor.java 172780 2016-04-26 18:31:42Z fpina $
 */
package csbase.sga.ssh;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.sga.executor.JobData;
import csbase.sga.executor.JobExecutor;
import csbase.sga.executor.JobInfo;
import csbase.sga.executor.JobObserver;
import csbase.sshclient.CommandResult;
import csbase.sshclient.SSHClient;
import csbase.sshclient.SSHClientException;
import sgaidl.ActionNotSupportedException;
import sgaidl.COMMAND_STATE;
import sgaidl.InvalidActionException;
import sgaidl.JobControlAction;
import sgaidl.ProcessState;

/**
 * Implements the interface {@link JobExecutor} using SSH as the execution layer
 * to talk to the execution environment.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SSHExecutor implements JobExecutor {
  /**
   * SGA properties
   */
  private Properties pluginProperties;
  /**
   * Execution environment driver
   */
  private SGADriver driver;
  /**
   * SSH client pool
   */
  private SSHClientPool sshClientPool;
  /**
   * Jobs storage
   */
  private JobStorage jobStorage;
  /**
   * Logger
   */
  private Logger logger;
  /**
   * Finished job's notifier executor
   */
  private final ExecutorService notifier;

  /**
   * Initialize the executor with a properties set and a {@link SGADriver} tied
   * to an execution environment.
   *
   * @param pluginProperties the properties
   * @param driver           the execution environment driver
   */
  public SSHExecutor(Properties pluginProperties, SGADriver driver,
                     SSHClientPool sshClientPool) {
    this.pluginProperties = pluginProperties;

    String sgaName = pluginProperties.getProperty(sgaidl.SGA_NAME.value);
    logger = Logger.getLogger(SGASSH.class.getName() + "." + sgaName);

    long jobInfoMaxAge = 60 * 1000;
    if (pluginProperties.containsKey(SGASSH.SGA_PROCESS_TIME_KEY)) {
      jobInfoMaxAge =
          Long.parseLong(
              pluginProperties.getProperty(SGASSH.SGA_PROCESS_TIME_KEY)) * 1000;
    }

    this.jobStorage = new JobStorage(jobInfoMaxAge);
    this.driver = driver;
    this.sshClientPool = sshClientPool;
    this.notifier = Executors.newCachedThreadPool();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobData executeJob(final String jobCommand, Map<String, String> extraParams, JobObserver observer) {
    JobData data = null;
    SSHClient client = null;
    try {
      client = sshClientPool.retrieveSSHClient();
      String command = driver.buildSubmitJobCommand(jobCommand, extraParams);
      logger.fine("Job's command line: " + command);
      CommandResult result = client.execute(command);
      if (result.getStatus() > 0) {
        logger.log(
            Level.WARNING, "Job execution return code: " + result.getStatus()
                + "\nOutput: " + result.getOutput() + "\nError: "
                + result.getError());

        observer.onJobLost();
        return null;
      }
      logger.fine(
          "Job execution return code: " + result.getStatus() + "\nOutput: "
              + result.getOutput() + "\nError: " + result.getError());
      data = driver.parseJobSubmissionOutput(result.getOutput());

      if (data == null) {
        logger.log(Level.SEVERE, "Cannot parse job submission output");
        observer.onJobLost();
        return null;
      } else {
        logger.fine("Job's JobData: " + data);
        jobStorage.addJob(data, observer);
      }
    } catch (IOException | SSHClientException e) {
      logger.log(Level.SEVERE, "Erro submitting job", e);
      if (client != null) {
        client.disconnect();
      }
    } finally {
      if (client != null) {
        sshClientPool.returnSSHClient(client);
      }
    }

    return data;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void recoverJob(JobData data, JobObserver observer) {
    try {
      updateSingleJobInfo(data);
      if (!jobStorage.recoverJob(data, observer)) {
        observer.onJobLost();
      }
    } catch (SSHClientException | IOException e) {
      logger.log(Level.WARNING, "Error recovering job", e);
      observer.onJobLost();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void controlJob(JobData data, String child, JobControlAction action)
      throws InvalidActionException, ActionNotSupportedException {
    SSHClient client = null;
    try {
      client = sshClientPool.retrieveSSHClient();
      String controlCommand = driver.buildKillJobCommand(data);
      logger.fine("Job's control command line: " + controlCommand);
      CommandResult result = client.execute(controlCommand);
      if (result.getStatus() > 0) {
        logger.log(
            Level.WARNING, "Job control return code: " + result.getStatus()
                + "\nOutput: " + result.getOutput() + "\nError: "
                + result.getError());
        return;
      }
      logger.fine(
          "Job control return code: " + result.getStatus() + "\nOutput: "
              + result.getOutput() + "\nError: " + result.getError());
      JobObserver observer = jobStorage.getObserver(data);
      jobStorage.removeJob(data);
      observer.onJobKilled();
    } catch (IOException | SSHClientException e) {
      logger.log(Level.SEVERE, "Erro controlling job", e);
      if (client != null) {
        client.disconnect();
      }
    } finally {
      if (client != null) {
        sshClientPool.returnSSHClient(client);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobInfo getJobInfo(JobData data) {
    try {
      updateJobsInfo();
    } catch (SSHClientException | IOException e) {
      logger.log(Level.SEVERE, "Error updating job info", e);
    }

    return jobStorage.getJobInfo(data);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void searchAndNotifyFinishedJobs() {
    try {
      updateJobsInfo();
    } catch (Exception e) {
      logger.log(Level.SEVERE, "Error while searching for finished jobs", e);
    }

    for (final JobData data : jobStorage.getJobs()) {
      final JobInfo info = jobStorage.getJobInfo(data);
      if (info != null && isFinished(info)) {
        final JobObserver observer = jobStorage.getObserver(data);
        if (observer != null) {
          notifier.execute(new Runnable() {
            @Override
            public void run() {
              observer.onJobCompleted(info);
            }
          });
          jobStorage.removeJob(data);
        }
      }
    }
  }

  private boolean isFinished(JobInfo info) {
    return info.jobParam.get(COMMAND_STATE.value).equals(ProcessState.FINISHED.toString());
  }

  private void updateSingleJobInfo(JobData data) throws SSHClientException, IOException {
    SSHClient client = null;
    try {
      client = sshClientPool.retrieveSSHClient();
      String updateCommand = driver.buildCheckJobCommand(data);
      logger.fine("Job's update command line: " + updateCommand);
      CommandResult result = client.execute(updateCommand);
      if (result.getStatus() > 0) {
        logger.log(
            Level.WARNING, "Update job info return code: "
                + result.getStatus() + "\nOutput: " + result.getOutput()
                + "\nError: " + result.getError());
        return;
      }
      logger.fine(
          "Update job info return code: " + result.getStatus() + "\nOutput: "
              + result.getOutput() + "\nError: " + result.getError());
      Map<JobData, JobInfo> jobs = new HashMap<>();
      jobs.put(data, driver.parseCheckJobOutput(data, result.getOutput()));

      jobStorage.updateJobs(jobs);
    } finally {
      if (client != null) {
        sshClientPool.returnSSHClient(client);
      }
    }
  }

  private void updateAllJobsInfo() throws SSHClientException, IOException {
    JobData[] jobsData = jobStorage.getJobs().toArray(new JobData[0]);
    SSHClient client = null;
    try {
      client = sshClientPool.retrieveSSHClient();
      String updateCommand = driver.buildCheckJobListCommand(jobsData);
      logger.fine("Job's update command line: " + updateCommand);
      CommandResult result = client.execute(updateCommand);
      if (result.getStatus() > 0) {
        logger.log(
            Level.WARNING, "Update all jobs info return code: "
                + result.getStatus() + "\nOutput: " + result.getOutput()
                + "\nError: " + result.getError());
        return;
      }
      logger.fine(
          "Update all jobs info return code: " + result.getStatus()
              + "\nOutput: " + result.getOutput() + "\nError: "
              + result.getError());
      jobStorage.updateJobs(driver.parseCheckJobListOutput(jobsData, result.getOutput()));
    } finally {
      if (client != null) {
        sshClientPool.returnSSHClient(client);
      }
    }
  }

  /**
   * Updates de jobs information. If the {@link SGADriver} in use implements
   * {@link SGADriver#buildCheckJobListCommand} all jobs are updated in one
   * command. Otherwise, the update is done one by one.
   *
   * @throws SSHClientException failure in the SSH client
   * @throws IOException        failure in the SSH client
   */
  private void updateJobsInfo() throws SSHClientException, IOException {
    if (!jobStorage.needsUpdate()) {
      return;
    }

    long startTime = System.currentTimeMillis();

    JobData[] jobsData = jobStorage.getJobs().toArray(new JobData[0]);

    try {
      if (driver.buildCheckJobListCommand(jobsData) != null) {
        updateAllJobsInfo();
      } else {
        for (JobData d : jobsData) {
          updateSingleJobInfo(d);
        }
      }
    } catch (Exception e) {
      logger.log(Level.SEVERE, "Error while updating jobs info", e);
    }

    logger.fine("Time took to update job info: " + (System.currentTimeMillis() - startTime + " ms"));
  }
}
