/**
 * $Id: SSHMonitor.java 171822 2016-03-03 18:41:09Z fpina $
 */
package csbase.sga.ssh;

import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.sga.monitor.EnvironmentMonitor;
import csbase.sga.monitor.SGAInfo;
import csbase.sshclient.CommandResult;
import csbase.sshclient.SSHClient;

/**
 * @author Tecgraf/PUC-Rio
 */
public class SSHMonitor implements EnvironmentMonitor {
  private final Properties pluginProperties;
  private final SGADriver driver;
  private SSHClientPool sshClientPool;
  private SGAInfo sgaInfo;
  /** Logger */
  private Logger logger;

  private long lastJobInfoUpdate = 0;
  private long checkEnvironmentInterval = 20 * 1000;

  public SSHMonitor(Properties pluginProperties, SGADriver driver,
    SSHClientPool sshClientPool) {
    this.pluginProperties = pluginProperties;

    if (pluginProperties.containsKey(SGASSH.SGA_MACHINE_TIME_KEY)) {
      checkEnvironmentInterval =
        Long.parseLong(
          pluginProperties.getProperty(SGASSH.SGA_MACHINE_TIME_KEY)) * 1000;
    }

    this.driver = driver;
    this.sshClientPool = sshClientPool;
    this.sgaInfo = new SGAInfo(this.pluginProperties);

    String sgaName = pluginProperties.getProperty(sgaidl.SGA_NAME.value);
    logger = Logger.getLogger(SGASSH.class.getName() + "." + sgaName);
  }

  @Override
  public boolean isAlive() {
    boolean alive = true;
    SSHClient client = null;
    try {
      client = sshClientPool.retrieveSSHClient();
      client.execute("ls /");
    } catch (Exception e) {
      alive = false;
      logger.log(Level.SEVERE, "Erro connection to remote host", e);
      if (client != null) {
        client.disconnect();
      }
    } finally {
      if (client != null) {
        sshClientPool.returnSSHClient(client);
      }
    }

    return alive;
  }

  @Override
  public synchronized SGAInfo getSGAInfo() {
    updateSGAInfo();
    return sgaInfo;
  }

  private void updateSGAInfo() {
    if (System.currentTimeMillis()
      - lastJobInfoUpdate <= checkEnvironmentInterval) {
      return;
    }
    long startTime = System.currentTimeMillis();

    SSHClient client = null;
    try {
      client = sshClientPool.retrieveSSHClient();
      logger.fine("Time took to create a SSH Client: " + (System.currentTimeMillis() - startTime + " ms"));
      String command = driver.buildCheckEnvironmentCommand();
      logger.fine("Check environment command line: " + command);
      CommandResult result = client.execute(command);
      if (result.getStatus() > 0) {
        logger.log(
          Level.WARNING, "Check environment return code: " + result.getStatus()
          + "\nOutput: " + result.getOutput() + "\nError: "
          + result.getError());
        return;
      }
      logger.fine(
        "Check environment return code: " + result.getStatus() + "\nOutput: "
          + result.getOutput() + "\nError: " + result.getError());
      this.sgaInfo = driver.parseCheckEnvironmentOutput(result.getOutput());
      lastJobInfoUpdate = System.currentTimeMillis();
      //TODO Record in the log file the info
    }
    catch (Exception e) {
      logger.log(Level.SEVERE, "Erro checking the environment", e);
      if (client != null) {
        client.disconnect();
      }
    }
    finally {
      if (client != null) {
        sshClientPool.returnSSHClient(client);
      }
    }

    logger.fine("Time took to update environment info: " + (System.currentTimeMillis() - startTime + " ms"));
  }
}
