
package csbase.tools;

import javax.swing.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.io.*;
import java.util.*;

/**
 * Modelo de armazenamento de propriedades de um arquivo.
 * @author Andr Luiz Clinio.
 */
class PropModel extends AbstractTableModel {

  /** Tabbed de propriedade correspondente */
  private final PropTab propTab;

  /**
   * Consulta ao nmero de colunas.
   * @return duas colunas (amostra/valor).
   */
  public int getColumnCount() {
    return 3;
  }

  /**
   * Consulta ao nmero de elementos do array.
   * @return a quantindade de amostras.
   */
  public int getRowCount() {
    return propTab.getCardinality();
  }

  /**
   * Consulta ao ttulo da coluna.
   * @param col o ndice da coluna.
   * @return o texto
   */
  public String getColumnName(int col) {
    if (col == 0) return "Propriedade";
    else if (col == 1) return "Valor";
    else if (col == 2) return "Descrio";
    else return "ERROR";
  }

  /**
   * Consulta a um valor.
   * @param row a linha editada.
   * @param col a coluna editada.
   * @return o valor 
   */
  public Object getValueAt(int row, int col) {
    if (col == 0) return propTab.getKey(row);
    else if (col == 1) return propTab.getValue(row);
    else if (col == 2) return propTab.getDescription(row);
    else return "ERROR";
  }

  /**
   * Consulta  classe de uma coluna
   * @param col a coluna
   * @return a classe que representa um double.
   */
  public Class getColumnClass(int col) {
    return String.class;
  }

  /**
   * Consulta de edio de clula.
   * @param row a linha editada.
   * @param col a coluna editada.
   * @return verdadeiro somente para a coluna de valor.
   */
  public boolean isCellEditable(int row, int col) {
    return false;
  }

  /**
   * Construtor do modelo.
   * @param values o array.
   */
  public PropModel(final PropTab propTab) {
    this.propTab = propTab;
  }
}

/**
 * Representa uma aba de visualizao de um arquivo de
 * propriedades.
 * @author Andr Clinio.
 */
class PropTab {

   /** Editor */
   final private PropEditor propEditor;

   /** Painel de visualizao do arquivo. */
   private JPanel panel = new JPanel();

   /** Arquivo Java representante da propriedade. */
   private final File file;

   /** Widget (tabela) de visualizao do arquivo de propriedades. */
   private final JTable table = new JTable();

   /** Proprieades armazenadas no arquivo. */
   private final Properties properties = new Properties();

   /**
    * Consulta ao painel de visualizao do arquivo.
    * @return o painel.
    */
   public JPanel getPanel() {
      return panel;
   }

   /**
    * Consulta a i-sima descrio de propriedade de um arquivo.
    * @param key a chave original
    * @return a string 
    */
   public String getDescription(final int index) {
      final String key = (String)getKey(index);
      final String name = getName();
      final String prefix = name.replaceAll("\\.properties$", "");
      final String newKey = prefix+"@"+key+"@description";
      final String desc = (String)properties.get(newKey);
      if (desc != null) return desc;
      return propEditor.getEditorValue(newKey);
   }

   /**
    * Consulta a i-simo valor de propriedade de um arquivo.
    * @param index o ndice
    * @return a string
    */
   public String getValue(final int index) {
     final Enumeration elements = properties.elements();
     for (int i = 0; i < index; i++) elements.nextElement();
     return (String)elements.nextElement();
   }

   /**
    * Consulta a i-sima chave de propriedade de um arquivo.
    * @param index o ndice
    * @return a string da chave.
    */
   public String getKey(final int index) {
     final Enumeration keys = properties.keys();
     for (int i = 0; i < index; i++) keys.nextElement();
     return (String)keys.nextElement();
   }

   /**
    * Retorna a quantidade de elementos (chaves/valores) do arquivo.
    * @return um nmero inteiro.
    */
   public int getCardinality() {
     return properties.size();
   }

   /**
    * Consulta ao nome do arquivo representado.
    * @return o nome do arquivo.
    */
   public String getName() {
      return file.getName();
   }

   /**
    * Construtor padro com bas em um arquivo de propriedades.
    * @param file o arquivo Java.
    * @throws Exception se houver falha na carga do arquivo.
    */
   public PropTab(final PropEditor propEditor, final File file) 
   throws Exception {
     this.file = file;
     this.propEditor = propEditor;

     properties.load(new FileInputStream(file));
     PropEditor.onSpecialSyntax(properties, true, false);
     table.setModel(new PropModel(this));
     JScrollPane scrollPane = new JScrollPane(table);
     scrollPane.setBorder(BorderFactory.createLineBorder(Color.darkGray));

     panel.setLayout(new BorderLayout());
     panel.add(new JPanel(), BorderLayout.EAST);
     panel.add(new JPanel(), BorderLayout.WEST);
     panel.add(new JPanel(), BorderLayout.SOUTH);
     panel.add(new JPanel(), BorderLayout.NORTH);
     panel.add(scrollPane, BorderLayout.CENTER);
   }

}

/**
 * Classe que representa o visualizador de propriedades.
 * @author Andr Luiz Clinio.
 */
public class PropEditor {

    /** Lista de diretrios a serem varridos. */
    private final Vector directories = new Vector();

    /** Propriedades de descrio. */
    private final Properties metaProperties = new Properties();

    /** Janela principal */
    private final JFrame mainFrame = new JFrame();

    /** Tab do dilogo principal */
    private final JTabbedPane tabbedPane = new JTabbedPane(JTabbedPane.LEFT);  

    /**
     * Montagem e exibio do dilogo.
     */
    private void startDialog() {
       final JButton closeButton = new JButton();
       closeButton.setText("Fechar visualizao");
       closeButton.addActionListener(new ActionListener() {
           public void actionPerformed(ActionEvent ae) {
              System.exit(0);
           }
       });

       final JPanel buttonPanel = new JPanel();
       buttonPanel.add(new JPanel());
       buttonPanel.add(closeButton);
       buttonPanel.add(new JPanel());

       final JPanel mainPanel = new JPanel();
       mainPanel.setLayout(new BorderLayout());
       mainPanel.add(new JPanel(), BorderLayout.EAST);
       mainPanel.add(new JPanel(), BorderLayout.WEST);
       mainPanel.add(buttonPanel, BorderLayout.SOUTH);
       mainPanel.add(new JPanel(), BorderLayout.NORTH);
       mainPanel.add(tabbedPane, BorderLayout.CENTER);
       mainFrame.setContentPane(mainPanel);
       mainFrame.setTitle("Visualizador de Propriedades");
       mainFrame.addWindowListener(new WindowAdapter() {
           public void windowClosing(WindowEvent e) {
              System.exit(0);
           };
       });

       final Dimension dim = new Dimension(700,600);
       mainFrame.setSize(dim);
       mainFrame.setVisible(true);
    }

    /**
     * Tentativa de carga de arquivo como propriedade.
     * @param dir o diretrio selecionado.
     */
    private void loadFiles(final File dir) {
       File[] files = dir.listFiles();
       final int len = files.length;
       if (len <= 0) {
          System.out.println("No files found!");
          return;
       }
       for (int i = 0; i < len; i++) {
           final File child = files[i];
           final String name = child.getName();
           try {
             if (child.isDirectory()) continue;
             if (name.matches(".+\\.properties$")) {
                final PropTab pTab = new PropTab(this, child);
                final String title = name.replaceAll("\\.properties$", "");
                tabbedPane.add(title, pTab.getPanel());
                metaProperties.load(new FileInputStream(child));
             }
             else if (name.matches(".+\\.meta-properties$")) {
                metaProperties.load(new FileInputStream(child));
             }
           }
           catch (Exception e) {
             System.out.println(
                 "File exception:\n"+
                 "["+e.getMessage()+"]\n"+
                 "File: "+name+"\n"+
                 "Causa: "+e.getCause()+"\n\n");
             e.printStackTrace();
           }
       }
       PropEditor.onSpecialSyntax(metaProperties, false, true);
    }

    /**
     * Busca e anlise de um diretrio escolhido.
     * @param dirName o nome (path) do diretrio.
     */
    private void loadDirectory(final String dirName) {
       System.out.println("Checking ["+dirName+"]...");
       File dir = new File(dirName);
       if (!dir.isDirectory()) {
          System.out.println("  + Not a directory ["+dirName+"]...");
          return;
       }
       loadFiles(dir);
    }

    /**
     * Busca dos diretrios selecionados.
     */
    private boolean searchForDirectories() {
       final int n = directories.size();
       if (n <= 0) return false;
       System.out.println("Checking "+n+" directories...");
       for (int i = 0; i < n; i++) {
           final String dir = (String)directories.get(i);
           if (dir != null) loadDirectory(dir);
           else System.out.println("Could not load a directory.");
       }
       return true;
    }

    /**
     * Parser dos argumentos fornecidos ao programa.
     * @param args o argumentos.
     */
    private void parseArgs(final String[] args) {
       final int len = args.length;
       for (int i = 0; i < len; i++) {
           final String s = args[i];
           if (s.startsWith("--directory")) {
              System.out.println("Seeking directory...");
              final String[] split = s.split(":");
              if (split == null || split.length < 2) {
                 System.out.println("  + Error in directory definition.");
                 continue;
              }
              else {
                final String dir = split[1];
                if (dir != null) directories.add(dir);
                else {
                   System.out.println("  + Could not seek directory argument.");
                   continue;
                }
              }
           }
           else {
              System.out.println("  + Unrecognized parameter: "+s);
           }
       }
    }

    /**
     * Busca de uma descrio de chave junto do editor.
     * @param key a chave.
     */
    public String getEditorValue(final String key) {
       final String val = (String)metaProperties.getProperty(key);
       if (val == null) return "???                     {"+key+"}";
       return val;
    }

    /**
     * Mtodo para retirar (conforme o caso) as propriedades de
     * smbolos especiais.
     * @param prop as propriedades.
     * @param unloadSpecial descartar se especial.
     * @param unloadCommon descartar se comum.
     */
    public static void onSpecialSyntax(final Properties prop, 
    final boolean unloadSpecial, final boolean unloadCommon) {
      for (Enumeration e = prop.propertyNames(); e.hasMoreElements() ;) {
        final String key = (String)e.nextElement();
        if (key.matches("^.*\\@.*\\@description$")) {
           if (unloadSpecial) prop.remove(key);
        }
        else {
           if (unloadCommon) prop.remove(key);
        }
      }
    }

    /**
     * Construtor padro que recebe os argumentos do programa.
     * @param args o argumentos.
     */
    public PropEditor(final String[] args) {
       parseArgs(args);
       if (!searchForDirectories()) return;
       startDialog();
    }

    /**
     * Ponto de entrada do programa.
     * @param args argumentos.
     */
    public static void main(final String[] args) {
       if (args == null || args.length <= 0) {
          System.out.println("No arguments found!");
          return;
       }
       PropEditor propEditor = new PropEditor(args);
    }
}

