/**
 * $Id: PropertiesDumper.java 170738 2015-12-16 17:39:22Z fpina $
 */
package csbase.tools;

import java.io.IOException;
import java.util.Properties;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.properties.PropertiesUtils;
import tecgraf.javautils.core.properties.PropertyException;

/**
 * Ferramenta que processa um arquivo de propriedades resolvendo backreferences
 * e exibe em <code>stdout</code> todas as propriedades no formato "chave=valor"
 * (ou apenas uma ou mais propriedades definidas pelo usurio).
 * 
 * @author Tecgraf
 */
public class PropertiesDumper {
  /**
   * Cria um coletor, l e exibe as propriedades. Se o usurio pediu por uma
   * propriedade especfica, exibe apenas seu valor (ou <code>null</code> caso
   * no exista). Se o usurio pediu por mais de uma propriedade, exibe seus
   * valores no formato <code>chave=valor</code>. Se o usurio no pediu por
   * nenhuma propriedade especfica, exibe <b>todas</b> no formato
   * <code>chave=valor</code>.
   * 
   * @param args parmetros fornecidos pelo usurio. O primeiro parmetro deve
   *        obrigatoriamente ser o path para o arquivo de propriedades; os
   *        demais so opcionais e correspondem a nomes das propriedades.
   * 
   * @throws PropertyException em caso de ciclos ou se uma propriedade
   *         referenciada no existe
   */
  private PropertiesDumper(String[] args) throws PropertyException {
    try {
      Properties props = PropertiesUtils.loadProperties(args[0]);
      if (args.length == 1) {
        props.list(System.out);
      }
      else if (args.length == 2) {
        System.out.println(props.getProperty(args[1]));
      }
      else {
        for (int i = 1; i < args.length; i++) {
          System.out.println(String.format("%s=%s", args[i],
            props.getProperty(args[i])));
        }
      }
    }
    catch (IOException e) {
      abort(LNG.get("PropertiesDumper.inexistent", new String[]{args[0]}), 1);
    }
  }

  /**
   * Aborta a execuo exibindo uma mensagem de erro.
   * 
   * @param msg mensagem de erro
   * @param code cdigo de sada
   */
  private static void abort(String msg, int code) {
    System.err.println(msg);
    System.exit(code);
  }

  /**
   * Executa o coletor.
   * 
   * @param args parmetros fornecidos pelo usurio. 0 = path, 1..n = chaves
   *        (opcionais). Se no for fornecida nenhuma chave, todas sero
   *        exibidas no formato "chave=valor".
   */
  public static void main(String[] args) {
    if (args.length < 1) {
      abort(LNG.get("PropertiesDumper.usage"), 1);
    }
    try {
      new PropertiesDumper(args);
    }
    catch (PropertyException e) {
      abort(e.getMessage(), 1);
    }
  }
}
