package csbase.tools;

import java.lang.reflect.Method;

/**
 * Gerador de instncia "indisponvel" de um servio do csbase.
 *
 * @author Jorge Alexandre e Ana Moura
 */
public class UnavailableServiceGenerator extends Generator {
  /**
   * Mtodo principal que ser executado quando o gerador
   * for chamado da linha de comando.
   * @param args args[0]: nome da interface remota do servio (com pacote)
   *             args[1]: nome da classe que implementa o servio (com pacote)
   * @throws Exception em caso de erro.
   */
  public static void main(String[] args)
  throws Exception {
    if (args.length != 2 || 
        getPackageName(args[0]) == null ||
        getPackageName(args[1]) == null) {
      usage();
      System.exit(1);
    }
    UnavailableServiceGenerator unavailablegen = 
      new UnavailableServiceGenerator(args[0], args[1]);
    unavailablegen.generate();
    unavailablegen.flushToFile(unavailablegen.getBuffer());
  }

  /**
   * Contri uma instncia do gerador de proxies.
   * @param interfaceName nome completo da interface remota do servio
   * @param serviceClassName nome completo da classe que implementa o servio
   */
  protected UnavailableServiceGenerator(String interfaceName, String serviceClassName) {
    super(interfaceName, serviceClassName, "Unavailable");
  }

  /**
   * Indica se o construtor da classe gerada invoca o construtor da
   * super classe.
   * @return true se o construtor da super classe  invocado
   *         false caso contrrio
   */
  protected boolean invokesSuperClassConstructor() {
    return false;
  }

  /**
   * Acrescenta excees especficas a um construtor.
   * Deve ser redefinida, se o construtor da classe tem excees
   * especficas.
   *
   * @return array com os nomes das excees lanadas
   */
  protected String[] getConstructorExceptions() {
    return new String[] { "ServerException" };
  }

  /**
   * Acrescenta as linhas de cdigo para o construtor da instncia
   * "indisponvel" do servio.
   *
   * @param code buffer que armazena o cdigo
   * @param paremeterNames nomes dos parmetros passados pelo construtor.
   */
  protected void getConstructorLines(StringBuffer code,
                                     String[] paremeterNames) {
  }

  /**
   * Indica se lana as excees relacionadas na interface.
   * @return true se as excees so lanadas
   *         false caso contrrio
   */
  protected boolean throwsInterfaceExceptions() {
    return false;
  }

  /**
   * Acrescenta ao cdigo a implementao de um mtodo.
   *
   * @param code buffer que armazena o cdigo
   * @param method descrio do mtodo a implementar
   * @param parameterNames nomes dos parmetros do mtodo
   */
  protected void getMethodLines(StringBuffer code,
                                Method method,
                                String[] parameterNames) {
    String returnLine =
      getNoPackagesClassName(serviceClassName)+".getInstance()." +
      method.getName()+"(";
    for (int i = 0; i < parameterNames.length; i++) {
      if ( i > 0 )
        returnLine += ", ";
      returnLine += parameterNames[i];
    }
    returnLine += ");\n";
    if (!"void".equals(method.getReturnType().getName()))
      returnLine = "return "+returnLine;

    code.append("  throw new UnavailableServiceException();\n");
  }

  /**
   * Obtm a super classe da instncia "indisponvel" do servio.
   * @return a prpria classe do servio
   */
  protected Class getSuperClass() throws ClassNotFoundException {
    return Class.forName(serviceClassName);
  }

  /**
   * Obtm o nome da super classe da classe gerada.
   * @return o nome da super classe que o classe gerada ir estender.
   */
  protected String getSuperClassName() {
    return  getNoPackagesClassName(serviceClassName);
  }


  /**
   * Obtm os pacotes a serem importados pela classe gerada.
   * @return um array com strings que representam os pacotes a serem 
   *         importados.
   */
  protected String[] getImports() {
    String[] imports =
      {
        "csbase.server.*", 
        "csbase.exception.*", 
      };
    return imports;
  }
}
