package csbase.util.messages;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;

/**
 * <p>
 * Representa uma mensagem serializvel.
 * </p>
 * <p>
 * Uma mensagem  composta de cabealho e corpo.<br>
 * O cabealho carrega informaes de identificao e tempo devida da mensagem;
 * enquanto o corpo carrega o dado a ser entregue.
 * </p>
 * 
 * @author Tecgraf
 * @version 1.0
 */
public class Message implements Serializable {

  /**
   * Verso da classe.
   */
  private static final long serialVersionUID = 1L;

  /**
   * @serial Identificador da mensagem.
   * @since 1.0
   */
  private UUID id;
  /**
   * @serial Data de criao da mensagem.
   * @since 1.0
   */
  private long creationDate;
  /**
   * @serial TimeZone padro do host em que a data de criao foi atribuda.
   * @since 1.0
   */
  private TimeZone timeZone;
  /**
   * @serial Possibilita aos usurios adicionar informaes extra a mensagem.
   * @since 1.0
   */
  private Map<String, Serializable> headers;

  /**
   * @serial Corpo da mensagem.
   * @since 1.0
   */
  private Serializable body;

  /**
   * Construtor.
   * 
   * @param body Corpo da mensagem.
   */
  public Message(Serializable body) {
    setBody(body);
    creationDate = System.currentTimeMillis();
    timeZone = TimeZone.getDefault();
  }

  /**
   * Obtm o identificador nico da mensagem.
   * 
   * @return Identificador nico da mensagem.
   */
  public UUID getId() {
    return id;
  }

  /**
   * <p>
   * Atribui o identificador nico da mensagem.
   * </p>
   * <p>
   * Esse mtodo ser utilizado pelo servio de entrega de mensagens para
   * identificar a mensagem dentro dele. Consequentemente, ir sobrescrever
   * qualquer valor que tenha sido atribudo ao id anteriormente.
   * </p>
   * 
   * @param id o identificador nico da mensagem.
   */
  public void setId(UUID id) {
    if (null == id) {
      throw new IllegalArgumentException("id == null");
    }

    this.id = id;
  }

  /**
   * Obtm o valor de um cabealho da mensagem.
   * 
   * @param name Nome do cabealho.
   * 
   * @return o valor de um cabealho da mensagem, ou {@code null} caso ele no
   *         exista. Obs.: {@code null} tambm  um valor vlido um cabealho,
   *         para saber se ele no existe, use o mtodo
   *         {@link #containsHeader(String)}.
   *        
   * @see #containsHeader(String)
   * @see #setHeader(String, Serializable)
   */
  public synchronized Serializable getHeader(String name) {
    if (headers == null) {
      return null;
    }
    return headers.get(name);
  }

  /**
   * Indica se existe um cabealho dada a seu nome.
   * 
   * @param name Nome do cabealho.
   * 
   * @return {@code true} caso o cabealho exista.
   * 
   * @see #getHeader(String)
   * @see #setHeader(String, Serializable)
   */
  public synchronized boolean containsHeader(String name) {
    return headers != null && headers.containsKey(name);
  }

  /**
   * Atribui o valor de um cabealho da mensagem. Se este cabealho ainda no
   * existir, ele ser criado.
   * 
   * @param name Nome do cabealho.
   * @param value Valor a ser atribudo.
   * 
   * @see #getHeader(String)
   * @see #containsHeader(String)
   */
  public synchronized void setHeader(String name, Serializable value) {
    if (headers == null) {
      headers = new HashMap<String, Serializable>();
    }
    headers.put(name, value);
  }

  /**
   * Obtm o corpo da mensagem.
   * 
   * @return O corpo da mensagem.
   */
  public Serializable getBody() {
    return body;
  }

  /**
   * Atribui um novo corpo a mensagem.
   * 
   * @param body O novo corpo da mensagem.
   */
  public void setBody(Serializable body) {
    if (null == body) {
      throw new IllegalArgumentException("body == null");
    }

    this.body = body;
  }

  /**
   * Obtm a data de criao da mensagem.
   * 
   * @return a data de criao da mensagem.
   */
  public long getCreationDate() {
    return creationDate;
  }

  /**
   * Obtm o {@link TimeZone} padro do host em que a data de criao foi
   * atribuda.
   * 
   * @return o {@link TimeZone} padro do host em que a data de criao foi
   *         atribuda.
   * 
   * @see #getCreationDate()
   */
  public TimeZone getTimeZone() {
    return timeZone;
  }
}
