package csbase.util.messages;

import java.io.Serializable;
import java.util.UUID;

/**
 * <p>
 * Identificador nico do consumidor de mensagens.
 * </p>
 * <p>
 * Sendo criado pelo {@link MessageBroker} ele tira a obrigao do cliente da
 * apiu de mensagens de criar seus prprios indentificadores e garantir sua
 * unicidade.
 * </p>
 * 
 * @author Tecgraf
 * @version 1.0
 */
class ConsumerId implements Serializable {

  /**
   * Verso da classe.
   */
  private static final long serialVersionUID = 1L;

  /**
   * @serial O identificador do broker que criou essa instncia.
   * @since 1.0
   */
  private final UUID brokerId;
  /**
   * @serial Valor utilizado para diferenciar cada instncia dessa classe.
   * @since 1.0
   */
  private final UUID value;

  /**
   * Construtor.
   * 
   * @param brokerId o identificador do broker criando essa instncia.
   */
  ConsumerId(UUID brokerId) {
    super();
    this.brokerId = brokerId;
    this.value = UUID.randomUUID();
  }

  /**
   * Obtm o identificador do broker que criou essa instncia.
   * 
   * @return o identificador do broker que criou essa instncia.
   */
  UUID getBrokerId() {
    return brokerId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((brokerId == null) ? 0 : brokerId.hashCode());
    result = prime * result + ((value == null) ? 0 : value.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    ConsumerId other = (ConsumerId) obj;
    if (brokerId == null) {
      if (other.brokerId != null)
        return false;
    }
    else if (!brokerId.equals(other.brokerId))
      return false;
    if (value == null) {
      if (other.value != null)
        return false;
    }
    else if (!value.equals(other.value))
      return false;
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return "ConsumerId [brokerId=" + brokerId + ", id=" + value + "]";
  }
}
