package csbase.util.proxy;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

/**
 * Responsvel por permitir a invocao remota de mtodos.
 * 
 * @author Tecgraf
 */
public class RMIInvocationHandlerImpl implements RMIInvocationHandler {

  /**
   * Objeto no qual se deseja fazer as chamadas remotas.
   */
  private Object obj;
  /**
   * Responsvel por tratar as chamadas ao objeto, obj, localmente.
   */
  private InvocationHandler handler;

  /**
   * Construtor.
   * 
   * @param obj Objeto no qual se deseja fazer as chamadas remotas.
   * @param handler Responsvel por tratar as chamadas ao objeto, obj,
   *        localmente.
   */
  public RMIInvocationHandlerImpl(Object obj, InvocationHandler handler) {
    this.obj = obj;
    this.handler = handler;
  }

  /**
   * <p>
   * Recebe as invocaes remotamente e as delega ao {@link InvocationHandler
   * handler} passado no construtor.
   * </p>
   * <p>
   * IMPORTANTE:<br>
   * A documentao do mtodo
   * {@link InvocationHandler#invoke(Object, Method, Object[])} diz que o proxy
   * no qual a chamada foi executada, ser passado como primeiro argumento.
   * Entretanto, para evitar a constante serializao do proxy, ser passado o
   * objeto para o qual o proxy foi criado para o {@link InvocationHandler}
   * responssvel pela execuo dos mtodos. No consideramos isso um problema,
   * pois a funo do parmetro proxy  apenas a de diferenciar onde a chamada
   * foi executada, j que um mesmo {@link InvocationHandler} pode ser utilizado
   * em vrios proxies e um proxy pode ser criado apenas com interfaces.
   * </p>
   * {@inheritDoc}
   */
  public Object invoke(String methodName, Class<?>[] parameterTypes,
    Object[] params) throws Throwable {
    Method method = obj.getClass().getMethod(methodName, parameterTypes);
    return handler.invoke(obj, method, params);
  }
}