/**
 * $Id: AbstractValidation.java 149543 2014-02-11 13:02:45Z oikawa $
 */
package validations;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import tecgraf.javautils.core.io.FileUtils;
import validations.util.ValidationsLogger;
import validations.util.ValidatorUtils;
import csbase.util.FileSystemUtils;

/**
 * Validao abstrata. Define o comportamento geral de uma validao, mas delega
 * para as subclasses concretas a implementao de etapas especficas.
 * 
 * @see Validator
 * @see ValidatorUtils
 */
public abstract class AbstractValidation {
  /**
   * Diretrio de recursos dos validadores.
   */
  protected static final String RESOURCE_DIR_NAME = "/validations/resources";

  /**
   * Logger para registro de mensagens.
   */
  protected ValidationsLogger logger;

  /**
   * Estados da validao/correo.
   */
  public enum Status {
    /**
     * Inicializao falhou.
     */
    INIT_FAILED(false, false),
    /**
     * Validao foi concluda com sucesso.
     */
    VALIDATION_OK(true, false),
    /**
     * Correo foi aplicada com sucesso (pode ser revertida).
     */
    PATCH_OK(true, true),
    /**
     * Validao falhou
     */
    VALIDATION_FAILED(false, false),
    /**
     * Correo falhou (pode ser revertida).
     */
    PATCH_FAILED(false, true),
    /**
     * Backup falhou.
     */
    BACKUP_FAILED(false, false),
    /**
     * J existe um backup de uma execuo anterior.
     */
    BACKUP_EXISTS(false, false),
    /**
     * Restaurao do backup foi bem sucedida.
     */
    ROLLBACK_OK(false, false),
    /**
     * Restaurao do backup falhou.
     */
    ROLLBACK_FAILED(false, false),
    /**
     * Ocorreu um erro durante a validao. Este caso  diferente de
     * {@link #VALIDATION_FAILED} porque o erro significa que a validao no
     * foi concluda, ao passo que a falha significa que ela foi concluda e
     * detectou que a converso  necessria.
     */
    VALIDATION_ERROR(false, false);

    /**
     * Operao bem-sucedida.
     */
    private final boolean isSuccess;
    /**
     * Operao pode ser revertida (rollback).
     */
    private final boolean isReversibleOperation;

    /**
     * Construtor.
     * 
     * @param isSuccess indica se esta constante est associada a um trmino
     *        bem-sucedido
     * @param isReversibleOperation indica se o status corresponde a uma
     *        operao que pode ser revertida (rollback)
     */
    private Status(boolean isSuccess, boolean isReversibleOperation) {
      this.isSuccess = isSuccess;
      this.isReversibleOperation = isReversibleOperation;
    }
  }

  /**
   * Estado em que se encontra a validao/correo.
   * 
   * @see #getStatus()
   */
  private Status status;

  /**
   * Referncia para o validador, que possui parmetros globais de configurao.
   */
  private Validator runValidator;

  /**
   * Path para o diretrio de backups desta validao. Aponta para um diretrio
   * com o nome da classe desta validao dentro do diretrio temporrio.
   */
  private String backupDirPath;

  /**
   * Inicializao. Este mtodo deve fazer as inicializaes necessrias 
   * execuo da validao, mas no deve implementar nenhum processamento.
   * 
   * @see #run(Validator)
   * 
   * @return flag indicando se a inicializao foi bem sucedida. Caso no seja,
   *         interrompe o processo.
   */
  protected abstract boolean init();

  /**
   * Obtm o texto exibido no incio do processamento da validao.
   * 
   * @return texto exibido no incio do processamento da validao. Pode ser
   *         <code>null</code>.
   */
  protected abstract String getStartMessage();

  /**
   * Indica se a validao s precisa ser executada com sucesso uma nica vez.
   * 
   * @return <code>true</code> se a validao s precisa ser executada com
   *         sucesso uma nica vez
   */
  protected abstract boolean runsOnlyOnce();

  /**
   * Obtm mensagens exibidas quando ocorre algum erro que no  genrico.
   * 
   * @see #getFailureMessage()
   * 
   * @param inputStatus status da validao
   * @param errors lista a ser preenchida com as mensagens de erro
   */
  protected abstract void getSpecificFailureMessage(Status inputStatus,
    List<String> errors);

  /**
   * Obtm a mensagem exibida quando a converso  bem sucedida.
   * 
   * @param inputStatus status da validao
   * 
   * @return mensagem exibida quando a validao bem sucedida
   */
  protected abstract String getSuccessMessage(Status inputStatus);

  /**
   * Valida o sistema, sem efetuar nenhuma alterao.
   * 
   * @see Validator
   * @see #applyPatch()
   * 
   * @return <code>true</code> se a validao foi bem sucedida, i.e. se no 
   *         necessrio aplicar uma converso. Este mtodo deve retornar
   *         <code>false</code> apenas se a execuo transcorreu sem problemas,
   *         e foi detectado que  necessrio aplicar uma converso. Excees e
   *         outros erros que impeam a execuo da validao devem ser
   *         sinalizados via {@link ValidationException}
   * @throws ValidationException se houve um erro no previsto durante a
   *         validao
   */
  protected abstract boolean validate() throws ValidationException;

  /**
   * Faz backup dos dados que sero alterados.  executado pelo validador antes
   * da aplicao do patch.
   * 
   * @see Validator
   * @see #applyPatch()
   * 
   * @return <code>true</code> se o backup foi realizado com sucesso
   */
  protected abstract boolean backupData();

  /**
   * Desfaz as alteraes, restaurando os backups feitos por
   * {@link #backupData()}.  executado pelo validador quando
   * {@link #applyPatch()} retorna <code>false</code>.
   * <p>
   * IMPORTANTE: este mtodo pode ser executado mesmo que o patch tenha sido
   * bem-sucedido, caso um patch subsequente falhe.
   * 
   * @return <code>true</code> se a restaurao foi bem-sucedida
   */
  protected abstract boolean restoreBackup();

  /**
   * Aplica as correes. Este mtodo s  executado se tanto
   * {@link #validate()} quanto {@link #isValidatingOnly()} retornaram
   * <code>false</code>.
   * 
   * @return <code>true</code> se as correes foram aplicadas com sucesso,
   *         false em caso contrrio.
   */
  protected abstract boolean applyPatch();

  /**
   * Finalizao do processo. Permite a exibio de mensagens customizadas etc.
   * Este mtodo  executado mesmo em caso de falha, portanto cabe a ele
   * verificar o status da validao.
   * 
   * @see #status
   */
  protected abstract void finish();

  /**
   * Obtm o status da validao.
   * 
   * @see Status
   * 
   * @return status da validao
   */
  public Status getStatus() {
    return status;
  }

  /**
   * Obtm a mensagem de erro em caso de falha. Caso no seja um erro genrico,
   * recorre a {@link #getSpecificFailureMessage(Status, List)}.
   * 
   * @return mensagem de erro em caso de falha
   */
  private List<String> getFailureMessage() {
    List<String> errors = new ArrayList<String>();
    switch (status) {
      case BACKUP_EXISTS:
        errors.add("*** ATUALIZAO FALHOU");
        errors
          .add("    No  possvel sobrescrever backup de atualizao anterior");
        errors.add("    Diretrio: " + backupDirPath);
        break;

      case BACKUP_FAILED:
        errors.add("*** ATUALIZAO FALHOU");
        errors.add("    Backup dos dados falhou");
        break;

      case ROLLBACK_FAILED:
        errors.add("*** ATUALIZAO FALHOU");
        errors.add("    Erro restaurando backup dos dados");
        break;

      case ROLLBACK_OK:
        errors.add("*** ATUALIZAO FALHOU");
        errors.add("    Os dados foram restaurados com sucesso");
        break;

      case VALIDATION_ERROR:
        errors.add("*** ERRO VALIDANDO O SISTEMA");
        errors.add("    Os dados no foram alterados");
        break;

      default:
        getSpecificFailureMessage(status, errors);
    }
    return errors;
  }

  /**
   * Executa a validao (e, opcionalmente, as correes).
   * <p>
   * Este mtodo  responsvel por:
   * <ul>
   * <li>inicializar o validador
   * <li>fazer a validao (<i>modo validao</i>)
   * <li>aplicar correes (<i>modo patch</i>)
   * </ul>
   * 
   * @param validator referncia para o validador, para obteno de informaes
   *        globais tais como o logger, se estamos apenas em modo validao etc.
   * 
   * @return true se o processo foi bem-sucedido <b>na ntegra</b>
   */
  public final boolean run(Validator validator) {
    this.runValidator = validator;
    this.logger = validator.getLogger();
    this.backupDirPath =
      getTempDirPath() + File.separatorChar + getClass().getSimpleName();

    /* inicializao da validao */
    if (!init()) {
      status = Status.INIT_FAILED;
      return false;
    }

    logger.sectionSeparator(Level.FINE);
    String startMessage = getStartMessage();
    if (startMessage != null) {
      logger.fine(startMessage);
    }

    boolean validationOK = false;
    try {
      validationOK = validate();
    }
    catch (Exception e) {
      /* no conseguimos nem mesmo validar o sistema... */
      logger.exception(e);
      status = Status.VALIDATION_ERROR;
    }

    if (status != Status.VALIDATION_ERROR) {
      if (validationOK) {
        status = Status.VALIDATION_OK;
      }
      else {
        /* A validao falhou! */
        /* Se no estamos apenas validando tentamos converter */
        if (isValidatingOnly()) {
          status = Status.VALIDATION_FAILED;
        }
        else {
          /* Primeiro, fazemos o backup dos dados */
          boolean backupOK;
          try {
            if (!prepareBackupDir()) {
              return false;
            }
            logger.fine("Backups armazenados em: " + backupDirPath);
            backupOK = backupData();
          }
          catch (Exception e) {
            backupOK = false;
          }
          if (backupOK) {
            /* Aplicao da converses. */
            try {
              status = applyPatch() ? Status.PATCH_OK : Status.PATCH_FAILED;
            }
            catch (Exception e) {
              status = Status.PATCH_FAILED;
            }
          }
          else {
            status = Status.BACKUP_FAILED;
          }
        }
      }
    }

    finish();
    return status.isSuccess;
  }

  /**
   * Exibio de "relatrio" de acordo com o cdigo de trmino.
   */
  void reportStatus() {
    if (status.isSuccess) {
      logger.separator(Level.FINE);
      logger.fine(getSuccessMessage(status));
    }
    else {
      logger.separator(Level.SEVERE);
      for (String error : getFailureMessage()) {
        logger.severe(error);
      }
    }
  }

  /**
   * Cria o diretrio para backups.
   * 
   * @return <code>true</code> se o diretrio foi criado com sucesso
   */
  private boolean prepareBackupDir() {
    if (FileSystemUtils.dirExists(backupDirPath)) {
      status = Status.BACKUP_EXISTS;
      return false;
    }
    if (!ValidatorUtils.mkDir(backupDirPath)) {
      status = Status.BACKUP_FAILED;
      return false;
    }
    return true;
  }

  /**
   * Desfaz as alteraes efetuadas por esta validao (apenas se a validao
   * est em um estado reversvel).
   * 
   * @see #restoreBackup()
   * @see Status#isReversibleOperation
   * 
   * @return <code>true</code> se a restaurao foi bem-sucedida
   */
  public final boolean rollback() {
    if (!status.isReversibleOperation) {
      final String fmt = "%s : status = %s";
      final String className = getClass().getSimpleName();
      final String msg = String.format(fmt, className, status.toString());
      logger.fine(msg);
      return false;
    }
    return restoreBackup();
  }

  /**
   * Retorna o valor associado a uma propriedade do sistema, ou
   * <code>null</code> caso esta no exista.
   * 
   * @param propName nome da propriedade
   * 
   * @return valor associado  propriedade do sistema, ou <code>null</code> caso
   *         esta no exista
   * 
   * @see Validator#getSystemProperty(String)
   * @see #getMandatorySystemProperty(String)
   */
  protected final String getSystemProperty(String propName) {
    return runValidator.getSystemProperty(propName);
  }

  /**
   * Retorna o valor associado a uma propriedade do sistema. Caso esta no
   * exista, sinaliza no log e retorna <code>null</code>.
   * 
   * @param propName nome da propriedade
   * @return valor associado  propriedade do sistema, ou <code>null</code> caso
   *         esta no exista
   * 
   * @see Validator#getMandatorySystemProperty(String)
   * @see #getSystemProperty(String)
   */
  protected final String getMandatorySystemProperty(String propName) {
    return runValidator.getMandatorySystemProperty(propName);
  }

  /**
   * Retorna o repositrio de projetos.
   * 
   * @return repositrio de projetos
   */
  public File getProjectDir() {
    return runValidator.getProjectDir();
  }

  /**
   * Retorna o repositrio de algoritmos.
   * 
   * @return repositrio de algoritmos
   */
  public File getAlgorithmDir() {
    return runValidator.getAlgorithmDir();
  }

  /**
   * Obtm o path para o diretrio temporrio da instalao.
   * 
   * @return path para o diretrio temporrio da instalao
   */
  protected final String getTempDirPath() {
    return runValidator.getTempDirPath();
  }

  /**
   * Obtm o path para o diretrio onde sero armazenados os backups.
   * 
   * @return path para o diretrio onde sero armazenados os backups
   */
  protected final String getBackupDirPath() {
    return backupDirPath;
  }

  /**
   * Verifica se estamos apenas validando (i.e. no devemos nem tentar
   * corrigir).
   * 
   * @return <code>true</code> se estamos apenas validando
   */
  protected final boolean isValidatingOnly() {
    return runValidator.isValidatingOnly();
  }

  /**
   * Verifica se estamos em modo verbose.
   * 
   * @return <code>true</code> se estamos em modo verbose
   */
  protected final boolean isVerbose() {
    return runValidator.isVerbose();
  }

  /**
   * Remove o diretrio de backups desta validao.
   * 
   * @return <code>true</code> se a operao foi bem sucedida
   */
  public boolean removeBackupDir() {
    if (isValidatingOnly() || status == Status.VALIDATION_OK) {
      /* Se estvamos apenas validando.. */
      /* Ou se no foi aplicada nenhuma converso, no temos backups a remover */
      return true;
    }
    return FileUtils.delete(new File(backupDirPath));
  }
}
