/**
 * $Id$
 */
package validations;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import tecgraf.javautils.core.io.FileUtils;
import validations.util.ValidatorUtils;
import csbase.util.FileSystemUtils;

/**
 * Corrige a propriedade {@link #WALL_TIME_PROPERTY} dos comandos, removendo-a
 * dos arquivos quando esta for menor que 0.
 * 
 * @author Tecgraf
 */
public class WallTimePropertyValidation extends AbstractValidation {

  /**
   * Nome do diretrio de comandos.
   */
  private static final String COMMANDS_DIRECTORY = ".cmds";
  /**
   * Nome do arquivo de propriedades de um comando.
   */
  private static final String COMMANDS_PROPERTIES = "cmd.properties";
  /**
   * Nome da propriedade a ser corrigida.
   */
  private static final String WALL_TIME_PROPERTY = "wallTimeSec";

  /**
   * Lgica para a validao da propriedade {@link #WALL_TIME_PROPERTY}.
   */
  private final ValidateWallTimeProperty VALIDATE_WALL_TIME_PROPERTY =
    new ValidateWallTimeProperty();
  /**
   * Lgica para a correo da propriedade {@link #WALL_TIME_PROPERTY}.
   */
  private final FixWallTimeProperty FIX_WALL_TIME_PROPERTY =
    new FixWallTimeProperty();
  /**
   * Lgica para fazer o backup das propriedades dos comandos.
   */
  private final BackupProperties BACKUP_PROPERTIES = new BackupProperties();
  /**
   * Lgica para restaurar a propriedades dos comandos usando o backup.
   */
  private final RestoreProperties RESTORE_PROPERTIES = new RestoreProperties();

  /**
   * Referncia para o diretrio <code>project</code>.
   */
  private File projectDir;

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean init() {
    this.projectDir = getProjectDir();
    if (projectDir == null) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validate() {
    return visitCommandsDirs(projectDir, VALIDATE_WALL_TIME_PROPERTY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    return visitCommandsDirs(projectDir, BACKUP_PROPERTIES);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    File backupDir = new File(getBackupDirPath());
    if (visitCommandsDirs(backupDir, RESTORE_PROPERTIES)) {
      /*
       * Uma vez que os arquivos de propriedade foram reataurados com sucesso,
       * j podemos apagar o backup.
       */
      FileUtils.delete(backupDir);
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    return visitCommandsDirs(projectDir, FIX_WALL_TIME_PROPERTY);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
    // No faz nada.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    return "Corrigindo propriedade '" + WALL_TIME_PROPERTY
      + "' com valor negativo nos comandos.";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(Status status) {
    switch (status) {
      case VALIDATION_OK:
        return "*** TODOS OS COMANDOS TEM UM VALOR VLIDO PARA A PROPRIEDADE '"
          + WALL_TIME_PROPERTY + "'.";

      case PATCH_OK:
        return "*** A PROPRIEDADE '" + WALL_TIME_PROPERTY
          + "' FOI CORRIGIDA EM TODOS OS COMANDOS.";

      default:
        return "ESTADO INVLIDO: " + status.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(Status status, List<String> errors) {
    switch (status) {
      case VALIDATION_FAILED:
        errors.add("*** EXISTEM COMANDOS COM VALOR INVLIDO NA PROPRIEDADE '"
          + WALL_TIME_PROPERTY + "'.");
        break;

      case PATCH_FAILED:
        errors.add("*** OCORRE UM ERRO AO TENTAR CORRIGIR A PROPRIEDADE '"
          + WALL_TIME_PROPERTY + "' DE UM OU MAIS COMANDOS.");
        break;

      case INIT_FAILED:
        errors.add("*** FALHA NA INICIALIZAO");
        break;

      default:
        errors.add("ESTADO INVLIDO: " + status.toString());
    }
  }

  /**
   * Corrige arquivos de propriedade de comandos removendo a propriedade
   * {@link WallTimePropertyValidation#WALL_TIME_PROPERTY} caso essa exista e
   * seja menor que 0.
   * 
   * @author Tecgraf
   */
  class FixWallTimeProperty implements FileVisitor {

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean visit(File properties) {
      if (properties.exists()) {
        File cmd = properties.getParentFile();
        File commands = cmd.getParentFile();
        File project = commands.getParentFile();
        File user = project.getParentFile();

        String bkpPropertiesPath =
          user.getName() + File.separatorChar + project.getName()
            + File.separatorChar + commands.getName() + File.separator
            + cmd.getName() + File.separator + properties.getName();
        File bkpProperties = new File(getBackupDirPath(), bkpPropertiesPath);

        BufferedReader reader = null;
        BufferedWriter writer = null;

        try {
          /*
           * Apago o arquivo de propriedades para criar um novo com a
           * propriedade corrigida.
           */
          properties.delete();

          reader = new BufferedReader(new FileReader(bkpProperties));
          writer = new BufferedWriter(new FileWriter(properties));
          String line;
          while ((line = reader.readLine()) != null) {
            if (line.matches("\\s*" + WALL_TIME_PROPERTY
              + "\\s*[=\\s]\\s*-\\d+\\s*")) {
              writer.write("# " + WALL_TIME_PROPERTY + " = ");
            }
            else {
              writer.write(line);
            }
            writer.newLine();
          }
          writer.flush();
        }
        catch (Exception e) {
          logger.exception(
            "Erro tentando ler o arquivo '" + bkpProperties.getAbsolutePath()
              + "' e escrever no arquivo '" + properties.getAbsolutePath()
              + "'", e);
          return false;
        }
        finally {
          try {
            if (writer != null) {
              writer.close();
            }
          }
          catch (IOException e) {
            logger.exception(String.format(
              "Erro fechando stream para o arquivo '%s'",
              properties.getAbsolutePath()), e);
          }
          try {
            if (reader != null) {
              reader.close();
            }
          }
          catch (IOException e) {
            logger.exception(String.format(
              "Erro fechando stream para o arquivo '%s'",
              bkpProperties.getAbsolutePath()), e);
          }
        }
      }
      else {
        logger.warning("Arquivo '" + properties.getAbsolutePath()
          + "' no encontrado");
      }
      // No estou interessado se o arquivo no existe.
      return true;
    }
  }

  /**
   * Faz uma cpia de cada arquivo de propriedades.
   * 
   * @author Tecgraf
   */
  class RestoreProperties implements FileVisitor {
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean visit(File bkpProperties) {
      File bkpCmd = bkpProperties.getParentFile();
      File bkpCommands = bkpCmd.getParentFile();
      File bkpProject = bkpCommands.getParentFile();
      File bkpUser = bkpProject.getParentFile();

      String cmdPath =
        bkpUser.getName() + File.separatorChar + bkpProject.getName()
          + File.separatorChar + bkpCommands.getName() + File.separator
          + bkpCmd.getName();
      File cmd = new File(projectDir, cmdPath);
      if (!cmd.exists()) {
        logger.severe("Diretrio do comando '" + cmd.getAbsolutePath()
          + "' no existe");
        return false;
      }

      File properties = new File(cmd, bkpProperties.getName());
      if (!ValidatorUtils.copyFile(bkpProperties, properties, logger, false)) {
        logger.severe(String.format(
          "Erro restaurando arquivo '%s' do usurio %s",
          bkpProperties.getAbsolutePath(), bkpUser.getName()));
        return false;
      }

      /*
       * Tendo o arquivo de propriedades do comando restaurado com sucesso,
       * removemos o diretrio de backup daquele comando.
       */
      FileUtils.delete(bkpCmd);
      return true;
    }
  }

  /**
   * Faz uma cpia de cada arquivo de propriedades.
   * 
   * @author Tecgraf
   */
  class BackupProperties implements FileVisitor {
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean visit(File properties) {
      File cmd = properties.getParentFile();
      File commands = cmd.getParentFile();
      File project = commands.getParentFile();
      File user = project.getParentFile();

      String bkpCmdPath =
        user.getName() + File.separatorChar + project.getName()
          + File.separatorChar + commands.getName() + File.separator
          + cmd.getName();
      File bkpCmd = new File(getBackupDirPath(), bkpCmdPath);
      if (!bkpCmd.exists() && !bkpCmd.mkdirs()) {
        logger.severe("Erro criando diretrio de backup '"
          + bkpCmd.getAbsolutePath() + "'");
        return false;
      }

      File bkpProperties = new File(bkpCmd, properties.getName());
      if (!ValidatorUtils.copyFile(properties, bkpProperties, logger, true)) {
        logger.severe(String.format(
          "erro fazendo backup do arquivo '%s' do usurio %s",
          properties.getAbsolutePath(), user.getName()));
        return false;
      }
      return true;
    }
  }

  /**
   * Verifica se a propriedade
   * {@link WallTimePropertyValidation#WALL_TIME_PROPERTY} no est presente ou
   *  maior ou igual a 0 para cada comando.
   * 
   * @author Tecgraf
   */
  class ValidateWallTimeProperty implements FileVisitor {
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean visit(File propertiesFile) {
      if (propertiesFile.exists()) {
        FileInputStream inStream = null;
        try {
          inStream = new FileInputStream(propertiesFile);
          Properties properties = new Properties();
          properties.load(inStream);
          if (properties.containsKey(WALL_TIME_PROPERTY)) {
            int wallTime =
              Integer
                .valueOf(properties.getProperty(WALL_TIME_PROPERTY).trim());
            return wallTime >= 0;
          }
        }
        catch (Exception e) {
          logger.exception(
            "Exceo tentando ler o arquivo '"
              + propertiesFile.getAbsolutePath() + "'", e);
        }
        finally {
          try {
            if (inStream != null) {
              inStream.close();
            }
          }
          catch (IOException e) {
            logger.exception(String.format(
              "Erro fechando stream para o arquivo '%s'",
              propertiesFile.getAbsolutePath()), e);
          }
        }
      }
      else {
        logger.warning("Arquivo '" + propertiesFile.getAbsolutePath()
          + "' no encontrado");
      }
      /*
       * O objetivo deste mtodo  verificar se o valor da propriedade wallTime
       *  maior ou igual a 0 quando esta estiver presente. No temos interesse
       * em verificar se os arquivos de propriedade existem e esto abrindo
       * corretamente. Nestes casos, iremos apenas logar o fato.
       */
      return true;
    }
  }

  /**
   * Forma abstrata de repassar a lgica de processamento de arquivos para
   * mtodos/objetos que iteram sobre eles.
   * 
   * @author Tecgraf
   */
  interface FileVisitor {
    /**
     * Visita um arquivo.
     * 
     * @param file Arquivo a ser visitado.
     * @return Se deve visitar os arquivos restantes.
     */
    boolean visit(File file);
  }

  /**
   * Itera sobre os arquivos de propriedades de cada comando j executado e
   * repassando-os ao visitante para que este possa process-los de acordo.
   * 
   * @param prjDir Diretrio dos projetos.
   * @param visitor Processa os diretrios de comandos.
   * 
   * @return <tt>true</tt> indicando se percorreu todos os arquivos de
   *         propriedades de comandos ou false indicando que parou antes do fim
   *         a pedido do visitante.
   */
  private boolean visitCommandsDirs(File prjDir, FileVisitor visitor) {
    final File[] users = FileSystemUtils.getSubDirs(prjDir);
    if (users == null || users.length == 0) {
      final String path = prjDir.getAbsolutePath();
      logger.fine("No h diretrios de usurios em: " + path);
      return true;
    }

    for (File aUser : users) {
      final File[] aProjects = FileSystemUtils.getSubDirs(aUser);
      if (aProjects == null || aProjects.length == 0) {
        final String path = aUser.getAbsolutePath();
        logger.fine("No h diretrios de projetos em: " + path);
        return true;
      }

      for (File aProject : aProjects) {
        File cmds = new File(aProject, COMMANDS_DIRECTORY);
        if (!cmds.exists()) {
          continue;
        }

        final File[] aCmds = FileSystemUtils.getSubDirs(cmds);
        if (aCmds == null || aCmds.length == 0) {
          final String path = aUser.getAbsolutePath();
          logger.fine("No h diretrios de comandos em: " + path);
          return true;
        }
        for (File aCmd : aCmds) {
          File propertiesFile = new File(aCmd, COMMANDS_PROPERTIES);
          if (!visitor.visit(propertiesFile)) {
            return false;
          }
        }
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return true;
  }

}
