/**
 * $Id: PrjPermsValidation.java 149543 2014-02-11 13:02:45Z oikawa $
 */
package validations;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import tecgraf.javautils.core.io.FileUtils;
import validations.util.ValidatorUtils;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectPermissions;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.util.FileSystemUtils;

/**
 * Patch para adaptar o modelo de compartilhamento de projetos para o novo
 * esquema de permisses (duas listas simultaneas de compartilhamento -- RO e
 * RW).
 * <p>
 * Todos os arquivos <code>*.csbase_project_info</code> nos diretrios
 * <code>project/[user]</code> so alterados para ficarem de acordo com o novo
 * modelo.
 * 
 * @see AbstractValidation
 * @see Validator
 * 
 * @author Tecgraf
 */
public class PrjPermsValidation extends AbstractValidation {

  /**
   * Sufixo dos arquivos de controle dos projetos (arquivos que contm objetos
   * {@link CommonProjectInfo} serializados).
   */
  private static final String CONTROL_FILE_SUFFIX = ".csbase_project_info";

  /**
   * Primeiro nvel de identao das mensagens de log.
   */
  private static final String IDENT1 = "    ";

  /**
   * Segundo nvel de identao das mensagens de log.
   */
  private static final String IDENT2 = "        ";

  /**
   * Antiga chave usada para determinar se o projeto era pblico.
   */
  private static final String OLD_IS_PUBLIC_KEY = "IS_PUBLIC";

  /**
   * Antiga chave usada para determinar se o projeto era compartilhado em modo
   * RO.
   */
  private static final String OLD_IS_READ_ONLY_KEY = "IS_READ_ONLY";

  /**
   * Antiga chave usada para armazenar a lista de usurios que tinham acesso ao
   * projeto.
   */
  private static final String OLD_USERS_KEY = "USERS";

  /**
   * Referncia para o diretrio <code>project</code>.
   */
  private File baseProjectDir;

  /**
   * Valida o diretrio de projetos de um usurio.
   * 
   * @param userDir diretrio de projetos do usurio
   * @return true se todos os projetos esto no formato correto
   * @throws ValidationException em caso de erro de validao em um arquivo.
   */
  private boolean validateUserDir(File userDir) throws ValidationException {
    File[] cfgFiles = getProjectControlFilesForUser(userDir);
    if (cfgFiles == null || cfgFiles.length == 0) {
      String path = userDir.getAbsolutePath();
      logger.fine("Nada encontrado para validar em:" + path);
      return true;
    }

    boolean result = true;
    for (File cfgFile : cfgFiles) {
      String prjName = getPrjNameFromCfgFile(cfgFile);
      try {
        if (validate(cfgFile)) {
          /*
           * projeto j convertido
           */
          logger.fine(String.format(IDENT1 + "%s ... OK", prjName));
        }
        else {
          /*
           * projeto no formato antigo, registramos e abortamos a verificao
           * indicando que projetos precisam ser convertidos
           */
          logger.fine(String.format(IDENT1 + "%s ... CONVERTER", prjName));
          result = false;
        }
      }
      catch (IOException e) {
        String err = String.format("Erro lendo configurao: %s", prjName);
        throw new ValidationException(err, e);
      }
    }
    return result;
  }

  /**
   * Processa todos os projetos de um usurio especfico. Todos os projetos so
   * verificados, independente de um ou mais estarem no formato antigo. Se
   * estamos em modo patch (i.e. se estamos convertendo), qualquer erro na
   * tentativa de converter um projeto interrompe o processo e retorna
   * <code>false</code>.
   * 
   * @param userDir - diretrio <code>project/[user]</code>, onde ficam todos os
   *        projetos
   * @return <code>true</code> se <code>validateOnly == true</code> e o projeto
   *         j havia sido convertido ou <code>validateOnly == false</code> e o
   *         projeto foi convertido com sucesso
   */
  private boolean processUserDir(File userDir) {
    File[] cfgFiles = getProjectControlFilesForUser(userDir);
    if (cfgFiles == null || cfgFiles.length == 0) {
      String path = userDir.getAbsolutePath();
      logger.fine("Nada encontrado para processar em:" + path);
      return true;
    }

    int converted = 0;
    int alreadyConverted = 0;
    boolean succeeded = true;
    for (File cfgFile : cfgFiles) {
      String prjName = getPrjNameFromCfgFile(cfgFile);
      try {
        if (validate(cfgFile)) {
          /*
           * projeto j convertido
           */
          alreadyConverted++;
        }
        else {
          /*
           * projeto no formato antigo, tentamos converter
           */
          converted++;
          logger.fine(String.format(IDENT1 + "projeto '%s'", prjName));
          if (!fixPerms(cfgFile)) {
            succeeded = false;
            break;
          }
        }
      }
      catch (ValidationException e) {
        logger
          .exception("Erro validando configurao do projeto " + prjName, e);
        succeeded = false;
        break;
      }
      catch (IOException e) {
        logger.exception("Erro convertendo configurao do projeto" + prjName,
          e);
        succeeded = false;
        break;
      }
    }
    String msgPrefix = String.format("Usurio %10s :", userDir.getName());
    int count;
    count = succeeded ? alreadyConverted + converted : converted;
    logger.fine(String.format("%s %d de %d projetos convertidos", msgPrefix,
      count, cfgFiles.length));
    return succeeded;
  }

  /**
   * Obtm os arquivo de controle para todos os projetos de um determinado
   * usurio.
   * 
   * @param userDir diretrio do usurio
   * @return array com os arquivos de controle para todos os projetos do usurio
   */
  private File[] getProjectControlFilesForUser(File userDir) {
    File[] cfgFiles = userDir.listFiles(new FileFilter() {
      @Override
      public boolean accept(File pathname) {
        final boolean isFile = pathname.isFile();
        final String name = pathname.getName();
        return isFile && name.endsWith(CONTROL_FILE_SUFFIX);
      }
    });
    return cfgFiles;
  }

  /**
   * Obtm o nome do projeto a partir do nome do seu arquivo de configurao.
   * 
   * @param cfgFile - arquivo de configurao
   * @return nome do projeto correspondente
   */
  private String getPrjNameFromCfgFile(File cfgFile) {
    String cfgFileName = cfgFile.getName();
    return cfgFileName.substring(0, cfgFileName.lastIndexOf('.'));
  }

  /**
   * Verifica se um determinado projeto j foi convertido.
   * 
   * @param cfgFile - arquivo de configurao do projeto
   * @return true se o projeto j foi convertido (possui um tipo
   *         {@link SharingType} != null).
   * @throws IOException se houve algum erro lendo os dados do arquivo de
   *         configurao
   * @throws ValidationException se houve algum erro na validao (p.ex.
   *         de-serializao)
   */
  private boolean validate(File cfgFile) throws IOException,
    ValidationException {
    CommonProjectInfo info = readProjectInfo(cfgFile);
    if (info == null) {
      throw new ValidationException(String.format(
        "erro processando configurao do projeto '%s'",
        getPrjNameFromCfgFile(cfgFile)));
    }
    try {
      ProjectPermissions.getSharingType(info);
    }
    catch (IllegalStateException e) {
      return false;
    }
    return true;
  }

  /**
   * Ajusta as permisses para um determinado projeto.
   * 
   * @param cfgFile - arquivo de configurao do projeto
   *        <code>[prj].csbase_project_info</code>
   * @return <code>true</code> se a operao foi bem-sucedida
   * @throws FileNotFoundException se o arquivo de configurao no existe
   * @throws IOException se no foi possvel ler ou gravar o arquivo de
   *         configurao
   */
  private boolean fixPerms(File cfgFile) throws IOException {
    CommonProjectInfo info = readProjectInfo(cfgFile);
    if (info == null) {
      logger.severe("erro obtendo informacoes do projeto");
      return false;
    }
    if (fixPerms(info)) {
      return writeProjectInfo(cfgFile, info);
    }
    return false;
  }

  /**
   * Grava as informaes do projeto (convertidas para o novo modelo).
   * 
   * @param cfgFile - arquivo de configurao
   * @param info - informaes do projeto
   * @return <code>true</code> se a gravao foi bem-sucedida
   * @throws IOException em caso de erro de I/O.
   */
  private boolean writeProjectInfo(File cfgFile, CommonProjectInfo info)
    throws IOException {
    logger.fine(IDENT1 + "gravando informacoes do projeto");
    ObjectOutputStream out = null;
    boolean success = true;
    try {
      out =
        new ObjectOutputStream(new DataOutputStream(new BufferedOutputStream(
          new FileOutputStream(cfgFile))));
      out.writeObject(info);
    }
    catch (Exception e) {
      logger.severe(e.toString());
      success = false;
    }
    finally {
      if (out != null) {
        out.close();
      }
    }
    return success;
  }

  /**
   * L as informaes do projeto do seu arquivo de configurao.
   * 
   * @param cfgFile - arquivo de configurao
   * @return informaes do projeto
   * @throws IOException em caso de erro de I/O.
   * @see CommonProjectInfo
   */
  private CommonProjectInfo readProjectInfo(File cfgFile) throws IOException {
    ObjectInputStream in = null;
    CommonProjectInfo info;
    // Testando caso especfico detectado em instalao de produo
    // do WebSintesi onde projetos (criados em verses antigas?)
    // ficam com tamanho zerado. Neste caso, o administrador
    // precisar rodar a ferramenta de recuperao.
    if (cfgFile.length() <= 0) {
      final String path = cfgFile.getAbsolutePath();
      final String err = "Detectado arquivo de configurao zerado: " + path;
      throw new IOException(err);
    }
    try {
      in =
        new ObjectInputStream(new DataInputStream(new BufferedInputStream(
          new FileInputStream(cfgFile))));
      info = (CommonProjectInfo) in.readObject();
    }
    catch (ClassNotFoundException e) {
      logger.exception(e);
      return null;
    }
    finally {
      if (in != null) {
        in.close();
      }
    }
    return info;
  }

  /**
   * Adapta as permisses para o novo modelo.
   * 
   * @param info - atributos do projeto
   * @return true se o projeto foi convertido com sucesso
   */
  private boolean fixPerms(CommonProjectInfo info) {
    try {
      if (ProjectPermissions.getSharingType(info) != null) {
        logger.fine("projeto j convertido");
        return false;
      }
    }
    catch (IllegalStateException e) {
      /*
       * projeto ainda no foi convertido -- no precisamos fazer nada aqui,
       * pois vamos convert-lo em seguida
       */
    }

    Boolean isPublic = (Boolean) getAttribute(info, OLD_IS_PUBLIC_KEY);
    isPublic = isPublic == null ? Boolean.FALSE : isPublic;
    logger.fine(IDENT2 + "IS_PUBLIC = " + isPublic);

    Boolean isRO = (Boolean) getAttribute(info, OLD_IS_READ_ONLY_KEY);
    isRO = isRO == null ? Boolean.FALSE : isRO;
    logger.fine(IDENT2 + "IS_READ_ONLY = " + isRO);

    Object[] users = (Object[]) getAttribute(info, OLD_USERS_KEY);

    SharingType sharing;
    /*
     * primeiro verificamos se o projeto era pblico
     */
    if (isPublic) {
      sharing = isRO ? SharingType.ALL_RO : SharingType.ALL_RW;
    }
    else {
      if (users == null) {
        /*
         * projeto era privado
         */
        sharing = ProjectPermissions.SharingType.PRIVATE;
      }
      else {
        /*
         * projeto era compartilhado seletivamente, precisamos associar a lista
         * original  lista correta (RO ou RW)
         */
        logger.fine(IDENT2 + "USERS = " + Arrays.toString(users));
        sharing = SharingType.PARTIAL;
        List<Object> usersList = Arrays.asList(users);
        if (isRO) {
          /*
           * usurios tinham acesso RO, a lista de usurios RW ser vazia
           */
          setUsers(info, usersList, true);
          setUsers(info, new ArrayList<Object>(), false);
        }
        else {
          /*
           * usurios tinham acesso RW, a lista de usurios RO ser vazia
           */
          setUsers(info, usersList, false);
          setUsers(info, new ArrayList<Object>(), true);
        }
        /*
         * removemos a lista original de usurios
         */
        removeAttribute(info, OLD_USERS_KEY);
      }
    }
    /*
     * removemos o flag antigo de projeto pblico
     */
    removeAttribute(info, OLD_IS_PUBLIC_KEY);
    /*
     * agora removemos o flag antigo de RO
     */
    removeAttribute(info, OLD_IS_READ_ONLY_KEY);
    /*
     * finalmente, definimos o novo atributo SHARING
     */
    setSharingType(info, sharing);
    return true;
  }

  /**
   * Obtm um atributo.
   * 
   * @param info - atributos do projeto
   * @param attr - atributo desejado
   * @return o valor associado ao atributo, ou null caso no exista
   */
  private Object getAttribute(CommonProjectInfo info, String attr) {
    return info.getAttribute(attr);
  }

  /**
   * Remove um atributo.
   * 
   * @param info - atributos do projeto
   * @param attr - atributo a ser removido
   */
  private void removeAttribute(CommonProjectInfo info, String attr) {
    logger.fine(IDENT2 + "- removendo atributo " + attr);
    info.getAttributes().remove(attr);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean init() {
    baseProjectDir = getProjectDir();
    if (baseProjectDir == null) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validate() throws ValidationException {
    File[] usersProjects = FileSystemUtils.getSubDirs(baseProjectDir);
    if (usersProjects == null || usersProjects.length == 0) {
      final String path = baseProjectDir.getAbsolutePath();
      logger.fine("Nada a validar em: " + path);
      return true;
    }

    for (File userPrj : usersProjects) {
      logger.fine("Verificando projetos de " + userPrj.getName());
      if (!validateUserDir(userPrj)) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    File[] usersProjects = FileSystemUtils.getSubDirs(baseProjectDir);
    if (usersProjects == null || usersProjects.length == 0) {
      final String path = baseProjectDir.getAbsolutePath();
      logger.fine("Nada a converter em: " + path);
      return true;
    }

    for (File userPrj : usersProjects) {
      logger.fine("Convertendo projetos de " + userPrj.getName());
      if (!processUserDir(userPrj)) {
        /*
         * no devemos tentar prosseguir se houve algum erro
         */
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
    // vazio
  }

  /**
   * Wrapper para definio do tipo de compartilhamento.
   * 
   * @param prjInfo - informaes do projeto
   * @param type - tipo de compartilhamento
   */
  private void setSharingType(CommonProjectInfo prjInfo, SharingType type) {
    logger.fine(IDENT2 + "+ compartilhamento = " + type.toString());
    ProjectPermissions.setSharingType(prjInfo, type);
  }

  /**
   * Wrapper para definio das listas de usurios com permisso de acesso ao
   * projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param users - lista de usurios com acesso ao projeto
   * @param readOnly - flag que indica se os usurios acima tm acesso apenas de
   *        leitura (<code>true</code>) ou de leitura e escrita (
   *        <code>false</code>)
   */
  private void setUsers(CommonProjectInfo prjInfo, List<Object> users,
    boolean readOnly) {
    logger.fine(String.format("%s+ usuarios %s = %s", IDENT2, readOnly ? "RO"
      : "RW", users));
    ProjectPermissions.setUsers(prjInfo, new HashSet<Object>(users), readOnly);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    return "Convertendo permisses de compartilhamento dos projetos";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(Status status, List<String> errors) {
    switch (status) {
      case VALIDATION_FAILED:
        errors.add("*** EXISTEM PROJETOS NO FORMATO ANTIGO");
        break;

      case PATCH_FAILED:
        errors.add("*** A CONVERSO DE UM OU MAIS PROJETOS RESULTOU EM ERRO");
        break;

      case INIT_FAILED:
        errors.add("*** FALHA NA INICIALIZAO");
        break;

      default:
        errors.add("ESTADO INVLIDO: " + status.toString());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(Status status) {
    switch (status) {
      case VALIDATION_OK:
        return "*** TODOS OS PROJETOS ENCONTRAM-SE NO NOVO FORMATO";

      case PATCH_OK:
        return "*** TODOS OS PROJETOS FORAM CONVERTIDOS COM SUCESSO";

      default:
        return "ESTADO INVLIDO: " + status.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    /*
     * fazemos o backup dos arquivos de controle que ainda no foram convertidos
     * Cada usurio possui um diretrio de backup prprio, onde so gravados os
     * arquivos de controle dos seus projetos
     */
    File[] usersDirs = FileSystemUtils.getSubDirs(baseProjectDir);
    if (usersDirs == null || usersDirs.length == 0) {
      final String path = baseProjectDir.getAbsolutePath();
      logger.fine("Nada a fazer backup em: " + path);
      return true;
    }

    for (File userDir : usersDirs) {
      String userBackupDirPath =
        getBackupDirPath() + File.separatorChar + userDir.getName();
      File userBackupDir = new File(userBackupDirPath);
      if (!userBackupDir.exists()) {
        if (!userBackupDir.mkdir()) {
          // TODO deveramos fazer algo mais?
          logger.severe("erro criando diretrio de backup para o usurio "
            + userDir.getName());
          return false;
        }
      }
      File[] controlFiles = getProjectControlFilesForUser(userDir);
      if (controlFiles == null || controlFiles.length == 0) {
        /* Caso que no deve ocorrer pois usersDirs est vazio */
        throw new RuntimeException("Erro interno: control-files! (backup)");
      }

      try {
        for (File ctrlFile : controlFiles) {
          if (validate(ctrlFile)) {
            /*
             * no precisamos fazer backup de projetos que j foram convertidos
             */
            continue;
          }
          if (!ValidatorUtils.copyFile(ctrlFile, userBackupDir, logger, true)) {
            logger.severe(String.format(
              "erro fazendo backup do arquivo %s do usurio %s",
              ctrlFile.getName(), userDir.getName()));
            return false;
          }
        }
      }
      catch (Exception e) {
        logger.exception(e);
        return false;
      }
    }
    /*
     * se chegamos aqui todos os arquivos de controles de projetos que precisam
     * ser convertidos j foram preservados
     */
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    final File backupDir = new File(getBackupDirPath());
    File[] usersBkpDirs = FileSystemUtils.getSubDirs(backupDir);
    if (usersBkpDirs == null || usersBkpDirs.length == 0) {
      final String path = backupDir.getAbsolutePath();
      logger.fine("Nada a restaurar backup em: " + path);
      return true;
    }

    /*
     * mapa (nome --> File) dos diretrios de cada usurio na rea de projetos
     */
    Map<String, File> usersDirsMap =
      ValidatorUtils.getSubdirsMap(baseProjectDir);
    /*
     * iteramos sobre todos os diretrios de backups dos usurios
     */
    for (File usersBkpDir : usersBkpDirs) {
      File[] bkpControlFiles = getProjectControlFilesForUser(usersBkpDir);
      if (bkpControlFiles == null || bkpControlFiles.length == 0) {
        /* Caso que no deve ocorrer pois {@link #usersDirs} est vazio */
        throw new RuntimeException("Erro interno: control-files (restore)!");
      }

      /*
       * agora iteramos sobre todos os arquivos de controle de cada diretrio
       */
      final String dirName = usersBkpDir.getName();
      File userDir = usersDirsMap.get(dirName);
      if (userDir == null) {
        logger.severe("Diretrio do usurio: " + dirName + " no existe");
        return false;
      }
      if (!ValidatorUtils.copyFiles(bkpControlFiles, userDir, logger, false)) {
        return false;
      }
      logger.info(String.format(
        "%sUsurio %10s : arquivos restaurados com sucesso", IDENT1, dirName));
      /*
       * todos os arquivos foram restaurados com sucesso, removemos o diretrio
       * de backup
       */
      FileUtils.delete(usersBkpDir);
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return true;
  }
}
