package csdk.v2.api.core;

import java.awt.Window;
import java.nio.charset.Charset;
import java.util.Locale;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.application.IMessage;

/**
 * Interface com o ambiente CSDK.
 */
public interface ICSDKEnvironment {

  /**
   * Inicia a execuo de outra aplicao.
   *
   * @param appId identificador da aplicao.
   * @return o identificador da nova instncia da aplicao ou <code>null</code>
   *         caso no exista aplicao com o identificador definido.
   *
   * @throws ApplicationException caso ocorra falha ao iniciar a nova aplicao.
   */
  public String runApplication(String appId) throws ApplicationException;

  /**
   * Envia uma mensagem sncrona a uma instncia de aplicao em execuo.
   *
   * @param instanceId o identificador da instncia da aplicao.
   * @param message a mensagem a ser enviada.
   * @return a resposta da mensagem.
   * @throws ApplicationException em caso de erro ao tentar enviar a mensagem.
   */
  public Object sendSyncMessage(String instanceId, IMessage message)
    throws ApplicationException;

  /**
   * Envia uma mensagem assncrona a todas instncias de aplicao em execuo.
   *
   * @param message a mensagem a ser enviada.
   * @throws ApplicationException em caso de erro ao tentar enviar a mensagem.
   */
  public void broadcastAsyncMessage(IMessage message)
    throws ApplicationException;

  /**
   * Envia uma mensagem assncrona a uma instncia de aplicao em execuo.
   *
   * @param instanceId o identificador da instncia da aplicao.
   * @param message a mensagem a ser enviada.
   * @throws ApplicationException em caso de erro ao tentar enviar a mensagem.
   */
  public void sendAsyncMessage(String instanceId, IMessage message)
    throws ApplicationException;

  /**
   * Requisita o encerramento da execuo da aplicao. Este mtodo deve ser
   * chamado caso a aplicao deseje ser terminada a qualquer hora antes da
   * finalizao pelo prprio ambiente. A aplicao precisa estar ativa para que
   * possa ser finalizada.
   */
  public void finishApplication();

  /**
   * Delega para o ambiente o tratamento de uma exceo.
   *
   * @param error a exceo.
   * @param parent janela que pode ser usada como base para mostrar mensagens de
   *        erro, se necessrio. Aceita <code>null</code>.
   */
  public void handleException(Exception error, Window parent);

  /**
   * Obtm o locale do ambiente.
   *
   * @return o locale.
   */
  public Locale getLocale();

  /**
   * Obtm o charset do servidor.
   *
   * @return o charset.
   */
  public Charset getServerCharset();

  /**
   * Obtm o charset do ambiente.
   *
   * @return o charset.
   */
  public Charset getCharset();

  /**
   * Obtm um contexto do ambiente a partir da sua interface. O contexto pode
   * no estar disponvel no ambiente e, nesse caso, o mtodo retorna
   * <code>null</code>.
   *
   * @param <T> interface do contexto.
   *
   * @param contextClass a interface do contexto desejado.
   * @return uma instncia do contexto ou <code>null</code> caso este contexto
   *         no esteja disponvel no ambiente.
   */
  public <T extends IContext> T getContext(Class<T> contextClass);

}
