package csdk.v2.api.filesystem;

import java.awt.Window;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;

/**
 * Interface para representao de arquivos locais e de projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public interface IFile {
  /**
   * Retorna a lista de arquivos deste diretrio.
   * 
   * @return A lista de arquivos deste diretrio.
   * @throws Exception Erro na busca de arquivos do diretrio.
   */
  public IFile[] getChildren() throws Exception;

  /**
   * Obtm um fluxo de entrada para leitura do arquivo.
   * 
   * @return Fluxo de entrada para leitura do arquivo.
   * @throws IOException em caso de falha de I/O no procedimento.
   */
  public InputStream getInputStream() throws IOException;

  /**
   * Retorna se existe o arquivo fsico.
   * 
   * @return {@code true} se o arquivo existe ou {@code false} caso
   *         contrrio.
   * @throws IOException em caso de falha de I/O no procedimento.
   */
  boolean exists() throws IOException;

  /**
   * Consulta a data/hora da ltima modificao do arquivo.
   * 
   * @return a data no formato {@link Date#getTime()}
   */
  public long getModificationDate();

  /**
   * Retorna o nome do arquivo.
   * 
   * @return Nome do arquivo.
   */
  public String getName();

  /**
   * Obtm um fluxo de sada para escrita no arquivo.
   * 
   * @return Fluxo de sada para escrita no arquivo.
   * @throws IOException em caso de falha de I/O no procedimento.
   */
  public OutputStream getOutputStream() throws IOException;

  /**
   * Retorna o diretrio-pai
   * 
   * @return o diretrio que contem este arquivo.
   */
  public IFile getParent();

  /**
   * Retorna o caminho para o arquivo na forma de um array.
   * 
   * @return Caminho para o arquivo na forma de um array.
   */
  public String[] getPath();

  /**
   * Retorna o caminho para o arquivo na forma de uma string.
   * 
   * @return Caminho para o arquivo na forma de uma string.
   */
  public String getStringPath();

  /**
   * Informa o tipo do arquivo.
   * 
   * @return O tipo do arquivo.
   */
  public String getType();

  /**
   * Indica se  um diretrio.
   * 
   * @return {@code true} caso seja diretrio ou {@code false} caso
   *         contrrio.
   */
  public boolean isDirectory();

  /**
   * Indica se o arquivo tem permisso de leitura.
   * 
   * @return {@code true} se o arquivo tem permisso de leitura,
   *         {@code false} caso contrrio.
   */
  public boolean canRead();

  /**
   * Indica se o arquivo tem permisso de escrita.
   * 
   * @return {@code true} se o arquivo tem permisso de escrita,
   *         {@code false} caso contrrio.
   */
  public boolean canWrite();

  /**
   * Indica se o arquivo tem permisso de execuo.
   * 
   * @return {@code true} se o arquivo tem permisso de execuo,
   *         {@code false} caso contrrio.
   */
  public boolean canExecute();

  /**
   * Abre o arquivo para leitura ou para leitura e escrita.
   * 
   * @param readOnly determinad se o arquivo deve ser aberto para leitura ou
   *        para leitura e escrita.
   * @throws Exception em caso de falha na abertura do arquivo.
   */
  public void open(boolean readOnly) throws Exception;

  /**
   * L uma seqncia de bytes deste arquivo a partir de uma determinada
   * posio.
   * 
   * @param dst O buffer no qual os bytes sero escritos.
   * @param position A posio do arquivo a partir da qual os bytes sero lidos.
   * 
   * @return O nmero de bytes lidos.
   * @throws Exception em caso de falha na leitura do arquivo.
   */
  public int read(byte[] dst, long position) throws Exception;

  /**
   * L uma seqncia de bytes de um determinado tamanho deste arquivo a partir
   * de uma determinada posio.
   * 
   * @param dst O buffer no qual os bytes sero escritos.
   * @param off O ndice do buffer a partir do qual os bytes sero escritos.
   * @param len O nmero de bytes a ler do arquivo e escrever no buffer.
   * @param position A posio do arquivo a partir do qual os bytes sero lidos.
   * 
   * @return O nmero de bytes lidos.
   * @throws Exception em caso de falha na leitura do arquivo.
   */
  public int read(byte[] dst, int off, int len, long position) throws Exception;

  /**
   * Fecha este arquivo. Aps chamado este mtodo, o objeto que representa o
   * arquivo no poder mais ser utilizado para leitura e/ou escrita.
   * 
   * @param force determina se o arquivo deve ser considerado fechado mesmo em
   *        caso de erro.
   * 
   * @throws IOException em caso de falha ao fechar o arquivo.
   */
  public void close(boolean force) throws IOException;

  /**
   * Informa o tamanho corrente deste arquivo. Reflete o nmero corrente de
   * bytes do arquivo.
   * 
   * @return Tamanho do arquivo em bytes.
   */
  public long size();

  /**
   * Identifica o tipo do arquivo quanto ao local onde o arquivo est
   * armazenado, podendo ser {@link FileLocationType#LOCAL} ou
   * {@link FileLocationType#REMOTE}
   * 
   * @return Identifica o tipo do arquivo quanto ao local onde o arquivo est
   *         armazenado.
   */
  public FileLocationType getFileLocationType();

  /**
   * Escreve uma sequncia de bytes no arquivo. Esse mtodo tenta fazer escritas
   * consecutivas at conseguir enviar a quantidade de bytes solicitada. Caso
   * no consiga, o mtodo lana uma exceo.
   * 
   * @param src O buffer contendo os bytes a serem escritos no arquivo.
   * @param off O deslocamento no buffer a partir do qual os bytes sero lidos.
   * @param len A quantidade de bytes a serem escritos do arquivo.
   * @param position A posio do arquivo a partir da qual a escrita deve ser
   *        iniciada.
   * 
   * @throws IOException em caso de falha ao escrever o arquivo.
   * @throws FileLockedException caso o arquivo esteja bloqueado para escrita
   *         por outra sesso de usurio.
   */
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException;

  /**
   * Escreve uma sequncia de bytes do arquivo. Esse mtodo tenta fazer escritas
   * consecutivas at conseguir enviar a quantidade de bytes solicitada. Caso
   * no consiga, o mtodo lana uma exceo.
   * 
   * @param src O buffer contendo os bytes a serem escritos no arquivo.
   * @param position A posio do arquivo a partir da qual a escrita deve ser
   *        iniciada.
   * 
   * @throws IOException em caso de falha ao escrever o arquivo.
   * @throws FileLockedException caso o arquivo esteja bloqueado para escrita
   *         por outra sesso de usurio
   */
  public void write(byte[] src, long position) throws IOException,
    FileLockedException;

  /**
   * Informa a posio corrente deste arquivo. Reflete o nmero de bytes a
   * partir do incio do arquivo at a posio corrente.
   * 
   * @return Posio corrente do arquivo.
   * 
   * @throws IOException em caso de falha de I/O no procedimento.
   */
  public long position() throws IOException;

  /**
   * Altera a posio corrente deste arquivo. Reflete o nmero de bytes a partir
   * do incio do arquivo at a posio corrente.
   * 
   * @param newPosition Nova posio corrente do arquivo.
   * 
   * @throws IOException em caso de falha de I/O no procedimento.
   */
  public void position(long newPosition) throws IOException;

  /**
   * Tenta obter um lock exclusivo para o arquivo. Locks exclusivo pode ser
   * obtido apenas um por vez, por arquivo.
   * 
   * Caso a RemoteTask seja cancelada, chama o mtodo para remover o lock no
   * servidor
   * 
   * @param window janela-me para a ao.
   * @return IFileLock que possui o status do lock
   * @throws Exception em caso de erro ao obter o lock
   */
  public IFileLock acquireExclusiveLock(Window window) throws Exception;

  /**
   * Tenta obter um lock compartilhado para o arquivo. Locks compartilhados
   * podem ser obtidos mltiplas vezes.
   * 
   * Este mtodo cadastra uma solicitao de lock compartilhado em uma fila
   * priorizada pela ordem de chegada dos pedidos de lock e espera pela
   * notificao de que o lock foi obtido ou seu timeout foi atingido.
   * 
   * @param window janela-me para a ao.
   * @return IFileLock que possui o status do lock
   * @throws Exception em caso de erro ao obter o lock
   */
  public IFileLock acquireSharedLock(Window window) throws Exception;

}
