package csdk.v2.api.filesystem.local;

import java.awt.Window;

import csdk.v2.api.core.IContext;
import csdk.v2.api.filesystem.FileSelectionType;
import csdk.v2.api.filesystem.IFile;

/**
 * Define os mtodos disponveis para acesso da aplicao ao sistema de arquivos
 * local do cliente. A vantagem de se usar este contexto ao invs de acessar
 * diretamente os arquivos pelas classes de I/O padro  a forma transparente de
 * tratar arquivos locais e de projeto na aplicao atravs da interface
 * {@link IFile}.
 */
public interface ILocalFileSystemContext extends IContext {

  /**
   * Mostra um navegador para seleo de um arquivo local em modo de leitura.
   * 
   * @param fileTypes os tipos de arquivos selecionveis no navegador. Exemplo:
   *        String[] fileTypes = {"XML", "TEXT}.
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param currentLocalDirectory diretrio inicial do navegador de arquivos.
   * @param selectionType modo de seleo do navegador (somente arquivos
   *        {@link FileSelectionType#FILES_ONLY} ou somente diretrios
   *        {@link FileSelectionType#DIRECTORIES_ONLY}).
   * @param window a janela-me do navegador de arquivos.
   * 
   * @return o arquivo selecionado ou {@code null} caso a operao tenha
   *         sido cancelada.
   * 
   * @throws LocalFileSystemException em caso de erro ao acessar o sistema de
   *         arquivos local.
   */
  public IFile browseLocalFileInOpenMode(String[] fileTypes, boolean allowAll,
    IFile currentLocalDirectory, FileSelectionType selectionType, Window window)
    throws LocalFileSystemException;

  /**
   * Mostra um navegador para seleo de mltiplos arquivos locais em modo de
   * leitura.
   * 
   * @param fileTypes os tipos de arquivos selecionveis no navegador. Exemplo:
   *        String[] fileTypes = {"XML", "TEXT}.
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param currentLocalDirectory diretrio inicial do navegador de arquivos.
   * @param selectionType modo de seleo do navegador (somente arquivos
   *        {@link FileSelectionType#FILES_ONLY} ou somente diretrios
   *        {@link FileSelectionType#DIRECTORIES_ONLY}).
   * @param window a janela-me do navegador de arquivos.
   * 
   * @return o arquivo selecionado ou {@code null} caso a operao tenha
   *         sido cancelada.
   * 
   * @throws LocalFileSystemException em caso de erro ao acessar o sistema de
   *         arquivos local.
   */
  public IFile[] browseMultipleLocalFilesInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentLocalDirectory,
    FileSelectionType selectionType, Window window)
    throws LocalFileSystemException;

  /**
   * Mostra um navegador para seleo de um arquivo local em modo de escrita.
   * 
   * @param fileTypes os tipos de arquivos selecionveis no navegador. Exemplo:
   *        String[] fileTypes = {"XML", "TEXT}.
   * @param allowAll determina a incluso da opo de selecionar qualquer tipo
   *        de arquivo, alm dos listados no parmetro fileTypes.
   * @param currentLocalDirectory diretrio inicial do navegador de arquivos.
   * @param window a janela-me do navegador de arquivos.
   * 
   * @return o arquivo selecionado ou {@code null} caso a operao tenha
   *         sido cancelada.
   * 
   * @throws LocalFileSystemException em caso de erro ao acessar o sistema de
   *         arquivos local.
   */
  public IFile browseLocalFileInSaveMode(String[] fileTypes, boolean allowAll,
    IFile currentLocalDirectory, Window window)
    throws LocalFileSystemException;

  /**
   * Obtm o arquivo local com o caminho especificado. O caminho deve ser
   * informado em forma de array, para evitar problemas de separadores
   * dependentes de plataforma.
   * 
   * Exemplos:
   * 
   * (Diretrio corrente) "./abc.txt" {@code ->} {"abc.txt"}
   * 
   * (Caminho relativo) "./dir/abc/" {@code ->} {"dir", "abc"}
   * 
   * (Caminho absoluto) "/tmp/dir/abc.txt" {@code ->} { "", "tmp", "dir", "abc.txt"}.
   * 
   * @param path o caminho sem separador.
   * 
   * @return o arquivo ou {@code null} caso no exista.
   * 
   * @throws LocalFileSystemException em caso de erro ao obter o arquivo.
   */
  public IFile getLocalFile(String[] path) throws LocalFileSystemException;

  /**
   * Cria um arquivo local no diretrio especificado. O caminho do diretrio pai
   * deve ser informado em forma de array, para evitar problemas de separadores
   * dependentes de plataforma.
   * 
   * Exemplos:
   * 
   * (Diretrio corrente) "./" {@code ->} {@code null} ou array vazio
   * 
   * (Caminho relativo) "./dir/abc/" {@code ->} {"dir", "abc"}
   * 
   * (Caminho absoluto) "/tmp/dir/abc/" {@code ->} { "", "tmp", "dir", "abc"}.
   * 
   * @param path o caminho do diretrio pai sem separador.
   * @param name o nome do arquivo a ser criado.
   * @param isDir determina se o arquivo a ser criado  um diretrio.
   * 
   * @return o arquivo ou {@code null} caso no seja possvel criar o
   *         arquivo.
   * 
   * @throws LocalFileSystemException em caso de erro ao criar o arquivo.
   */
  public IFile createLocalFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException;

  /**
   * Remove um arquivo ou diretrio local no caminho especificado. O caminho
   * deve ser informado em forma de array, para evitar problemas de separadores
   * dependentes de plataforma. Para remover um diretrio, este precisa estar
   * vazio.
   * 
   * Exemplos:
   * 
   * (Diretrio corrente) "./abc.txt" {@code ->} {"abc.txt"}
   * 
   * (Caminho relativo) "./dir/abc/" {@code ->} {"dir", "abc"}
   * 
   * (Caminho absoluto) "/tmp/dir/abc.txt" {@code ->} { "", "tmp", "dir", "abc.txt"}.
   * 
   * @param path o caminho para o arquivo sem separador.
   * 
   * @return {@code true} caso a remoo seja feita com sucesso ou
   *         {@code false} caso contrrio.
   * 
   * @throws LocalFileSystemException em caso de erro ao remover o arquivo.
   */
  public boolean removeLocalFile(String[] path) throws LocalFileSystemException;
}