package csdk.v1_0.api.application;

import java.io.Serializable;

import csdk.v1_0.api.core.ICSDKEnvironment;

/**
 * Interface que define os mtodos a serem implementados por todas as aplicaes
 * CSDK.
 * 
 * A classe principal da aplicao ser construda via reflexo, que deve
 * implementar esta interface e possuir um construtor pblico que receba como
 * parmetro uma instncia de {@link ICSDKEnvironment}:
 * 
 * <code>
 * public class MyApplication implements IApplication {
 * 
 * public MyApplication(ICSDKInterface csdkInterface) {
 *   // cdigo do construtor
 * }
 * 
 * // cdigo do restante da classe
 * 
 * }
 * </code>
 * 
 * [ATENO] Os mtodos dessa interface *no* devem ser chamados explicitamente
 * pela aplicao, eles so de uso exclusivo do CSDK, para controle do ciclo de
 * vida da aplicao.
 */
public interface IApplication extends Serializable {

  /**
   * Inicializa a aplicao.
   * 
   * [ATENO] Este mtodo *no* deve ser chamado explicitamente pela aplicao,
   * ele  de uso exclusivo do CSDK.
   * 
   * Este mtodo  chamado uma nica vez pelo CSDK logo aps o construtor da
   * classe e logo antes da aplicao ser adicionada  lista de aplicaes
   * ativas no sistema.
   * 
   * @throws ApplicationException Em caso de falha na inicializao da
   *         aplicao.
   */
  public void onApplicationStart() throws ApplicationException;

  /**
   * Encerra a execuo da aplicao.
   * 
   * [ATENO] Este mtodo *no* deve ser chamado explicitamente pela aplicao,
   * ele  de uso exclusivo do CSDK.
   * 
   * Este mtodo  chamado uma nica vez pelo CSDK durante a finalizao da
   * aplicao, logo antes da aplicao ser removida da lista de aplicaes
   * ativas no sistema.
   * 
   * @throws ApplicationException Em caso de falha na finalizao da aplicao.
   */
  public void onApplicationEnd() throws ApplicationException;

  /**
   * Trata uma mensagem recebida pela aplicao.
   * 
   * [ATENO] Este mtodo *no* deve ser chamado explicitamente pela aplicao,
   * ele  de uso exclusivo do CSDK.
   * 
   * A aplicao poder receber uma mensagem a qualquer momento de sua execuo.
   * O remetente da mensagem pode ser outra aplicao, representada pelo
   * parmetro {@link IMessageSender sender} ou o prprio ambiente, nesse caso o
   * parmetro {@link IMessageSender sender} vem nulo. Uma mesma aplicao pode
   * receber diferentes tipos de mensagem, definidos por
   * {@link IMessage#getType()}. O valor da mensagem  definido em
   * {@link IMessage#getValue()}.
   * 
   * @param message a mensagem recebida.
   * @param sender a aplicao que enviou a mensagem. (pode ser
   *        <code>null</code> - caso a mensagem tenha vindo do sistema, por
   *        exemplo)
   */
  public void onMessageReceived(IMessage message, IMessageSender sender);

  /**
   * Determina se a aplicao pode ser encerrada.
   * 
   * [ATENO] Este mtodo *no* deve ser chamado explicitamente pela aplicao,
   * ele  de uso exclusivo do CSDK.
   * 
   * Este mtodo  chamado pelo CSDK quando o fechamento da aplicao 
   * requisitado, dando chance da aplicao adiar o fechamento. No entanto, nem
   * sempre  possvel respeitar a recusa da aplicao em se fechar ou at
   * sequer h chance deste mtodo ser chamado antes da finalizao. A aplicao
   * deve estar pronta para lidar com todos esses cenrios.
   * 
   * @return <code>true</code> se a aplicao pode ser finalizada no momento ou
   *         <code>false</code> caso contrrio.
   */
  public boolean canEndApplication();

}