package csdk.v1_0.helper.application;

import java.awt.Window;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.Locale;

import javax.swing.ImageIcon;

import csdk.v1_0.api.application.ApplicationException;
import csdk.v1_0.api.application.IApplication;
import csdk.v1_0.api.application.IApplicationContext;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.core.IContext;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.helper.ContextNotAvailableException;

/**
 * Classe abstrata para facilitar o desenvolvimento de aplicaes com CSDK.
 */
public abstract class AbstractApplication implements IApplication,
  ICSDKEnvironment, IApplicationContext {

  /**
   * Contexto da aplicao.
   */
  private IApplicationContext context;

  /**
   * Interface padro para o ambiente CSDK.
   */
  private ICSDKEnvironment csdkInterface;

  /**
   * Construtor.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  protected AbstractApplication(ICSDKEnvironment csdkInterface) {
    this.csdkInterface = csdkInterface;
    this.context = csdkInterface.getContext(IApplicationContext.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessageReceived(IMessage message, IMessageSender sender) {
    //No faz nada
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasString(String key) {
    checkApplicationContext();
    return this.context.hasString(key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getString(String key, Object... objs) {
    checkApplicationContext();
    return this.context.getString(key, objs);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getApplicationId() {
    checkApplicationContext();
    return this.context.getApplicationId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getInstanceId() {
    checkApplicationContext();
    return this.context.getInstanceId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isPropertyNull(String propName) {
    checkApplicationContext();
    return this.context.isPropertyNull(propName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isBundleRequired() {
    checkApplicationContext();
    return this.context.isBundleRequired();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getProperty(String propName) {
    checkApplicationContext();
    return this.context.getProperty(propName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getApplicationName() {
    checkApplicationContext();
    return this.context.getApplicationName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getImageIcon(String[] path) {
    checkApplicationContext();
    return this.context.getImageIcon(path);
  }

  /**
   * Obtm o pacote da classe especificada.
   * 
   * @param clazz a classe.
   * 
   * @return o pacote.
   */
  private String getPackageName(Class<?> clazz) {
    String packageName;
    String className = clazz.getName();
    int index = className.lastIndexOf(".");
    if (index != -1) {
      packageName = className.substring(0, index);
    }
    else {
      packageName = "";
    }
    return packageName;
  }

  /**
   * Obtm o caminho para busca dos recursos da classe.
   * 
   * @param path o caminho relativo do recurso.
   * 
   * @return o caminho absoluto para busca do recurso.
   */
  private String getResourcePath(String[] path) {
    String sep = "/";
    Class<?> baseClass = this.getClass();
    String packageName = getPackageName(baseClass);
    StringBuilder resPath = new StringBuilder();
    resPath.append(packageName.replace(".", "/"));
    resPath.append(sep);
    resPath.append("resources");
    resPath.append(sep);
    for (String pathElement : path) {
      resPath.append(pathElement);
      resPath.append(sep);
    }
    resPath.deleteCharAt(resPath.length() - 1);
    return resPath.toString();
  }

  /**
   * Obtm a URL de um recurso no caminho especificado (relativo ao diretrio
   * padro de recursos da aplicao).
   * 
   * @param path o caminho para o recurso, relativo ao diretrio de recursos
   *        padro da aplicao.
   * @return a url ou <code>null</code> caso a recurso no seja encontrado no
   *         caminho especificado.
   */
  public URL getResource(String[] path) {
    try {
      ClassLoader classLoader = this.getClass().getClassLoader();
      String resourcePath = getResourcePath(path);
      return classLoader.getResource(resourcePath);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Obtm o Stream de um recurso no caminho especificado (relativo ao diretrio
   * padro de recursos da aplicao).
   * 
   * @param path o caminho para o recurso, relativo ao diretrio de recursos
   *        padro da aplicao.
   * @return o stream ou <code>null</code> caso a recurso no seja encontrado no
   *         caminho especificado.
   */
  public InputStream getResourceAsStream(String[] path) {
    try {
      ClassLoader classLoader = this.getClass().getClassLoader();
      String resourcePath = getResourcePath(path);
      return classLoader.getResourceAsStream(resourcePath);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Obtm o charset padro para leitura do arquivo. Se o arquivo for local,
   * retorna o charset do ambiente local. Se o arquivo for remoto, retorna o
   * charset do servidor.
   * 
   * @param file o arquivo.
   * 
   * @return o charset.
   */
  public Charset getCharset(IFile file) {
    if (file == null) {
      throw new IllegalArgumentException("Null file");
    }
    switch (file.getFileLocationType()) {
      case LOCAL:
        return getCharset();
      case REMOTE:
        return getServerCharset();
      default:
        throw new IllegalStateException("Cannot determine charset");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getSmallApplicationIcon() {
    checkApplicationContext();
    return this.context.getSmallApplicationIcon();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getApplicationIcon() {
    checkApplicationContext();
    return this.context.getApplicationIcon();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getVersion() {
    checkApplicationContext();
    return this.context.getVersion();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getApplicationDescription() {
    checkApplicationContext();
    return this.context.getApplicationDescription();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getAuthor() {
    checkApplicationContext();
    return this.context.getAuthor();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getAuthorEmail() {
    checkApplicationContext();
    return this.context.getAuthorEmail();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSingleton() {
    checkApplicationContext();
    return this.context.isSingleton();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean requiresProject() {
    checkApplicationContext();
    return this.context.requiresProject();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getApplicationFileTypes() {
    checkApplicationContext();
    return this.context.getApplicationFileTypes();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String runApplication(String id) throws ApplicationException {
    return this.csdkInterface.runApplication(id);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean sendMessage(String instanceId, IMessage message) {
    return this.csdkInterface.sendMessage(instanceId, message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void finishApplication() {
    this.csdkInterface.finishApplication();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T extends IContext> T getContext(Class<T> contextClass) {
    return this.csdkInterface.getContext(contextClass);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleException(Exception error, Window parent) {
    this.csdkInterface.handleException(error, parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Locale getLocale() {
    return this.csdkInterface.getLocale();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getCharset() {
    return this.csdkInterface.getCharset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getServerCharset() {
    return this.csdkInterface.getServerCharset();
  }

  /**
   * Indica se o contexto de aplicao j est disponvel.
   * 
   * @return <code>true</code> se o contexto estiver disponvel ou
   *         <code>false</code> caso contrrio.
   */
  public boolean isApplicationContextAvailable() {
    return this.context != null && this.context.isActive();
  }

  /**
   * Verifica se o contexto de aplicao est disponvel, lanando uma exceo
   * {@link ContextNotAvailableException} caso negativo.
   */
  private void checkApplicationContext() {
    if (!isApplicationContextAvailable()) {
      throw new ContextNotAvailableException(IApplicationContext.class);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onContextActivate() {
    // No faz nada.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onContextDeactivate() {
    // No faz nada.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isActive() {
    return false;
  }

  /**
   * Indica se um texto de internacionalizao foi definido, usando uma chave
   * prefixada pelo nome da classe definida.
   * 
   * @param clazz a classe que servir que prefixo pra chave de
   *        internacionalizao.
   * @param key_suffix o sufixo da chave. A chave ser: \" {@code clazz}.
   *        {@code key_suffix}\"
   * 
   * @return <code>true</code> se o texto foi definido ou <code>false</code>
   *         caso contrrio.
   */
  public boolean hasClassString(Class<?> clazz, final String key_suffix) {
    final String classPrefix = clazz.getSimpleName();
    return hasString(classPrefix + "." + key_suffix);
  }

  /**
   * Retorna um texto de internacionalizao, usando uma chave prefixada pelo
   * nome da classe definida.
   * 
   * @param clazz a classe que servir que prefixo pra chave de
   *        internacionalizao.
   * @param key_suffix o sufixo da chave. A chave ser: \" {@code clazz}.
   *        {@code key_suffix}\"
   * 
   * @param args argumentos do texto. O texto ser formatado utilizando-se da
   *        classe {@link MessageFormat}.
   * 
   * @return o texto internacionalizado.
   */
  public String getClassString(Class<?> clazz, String key_suffix,
    Object... args) {
    final String classPrefix = clazz.getSimpleName();
    return getString(classPrefix + "." + key_suffix, args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    disposeContexts();
  }

  /**
   * Limpa as referncias para os contextos.
   */
  protected void disposeContexts() {
    this.csdkInterface = null;
    this.context = null;
  }

}
