package csdk.v2.runner.filesystem;

import java.awt.Window;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.Channel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import csdk.v2.api.filesystem.FileLocationType;
import csdk.v2.api.filesystem.FileLockedException;
import csdk.v2.api.filesystem.IFile;
import csdk.v2.api.filesystem.IFileLock;
import csdk.v2.runner.Runner;

/**
 * Simula um arquivo na sandbox do CSDK.
 *
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class RunnerFile implements IFile {

  /**
   * Arquivo interno associado
   */
  private File file;

  /**
   * Arquivo interno para acesso randmico (I/O).
   */
  private RandomAccessFile randomFile;

  /**
   * Tipo do arquivo.
   */
  private String type;

  /**
   * Extrai a extenso do arquivo especificado.
   *
   * @param path caminho (ou o prprio nome) para o arquivo do qual pretende-se
   *        extrair a extenso.
   *
   * @return extenso do arquivo ou {@code null} se o arquivo no possuir
   *         extenso.
   */
  private static String getFileExtension(String path) {
    if (path == null) {
      String msg = "Internal error: null path not allowed.";
      throw new IllegalArgumentException(msg);
    }
    if (path.trim().equals("")) {
      String msg = "Internal error: empty path not allowed.";
      throw new IllegalArgumentException(msg);
    }
    int periodIndex = path.lastIndexOf(".");
    if (periodIndex == -1 || periodIndex == path.length() - 1) {
      return null;
    }
    return path.substring(periodIndex + 1).toLowerCase();
  }

  /**
   * Obtm o arquivo interno.
   *
   * @return o arquivo.
   */
  private File getFile() {
    return file;
  }

  /**
   * Retorna o caminho para o arquivo. Esse caminho no  relativo, ou seja, o
   * caminho absoluto de onde o arquivo se encontra.
   *
   * @return caminho absoluto at o arquivo, inclusive.
   */
  public String getAbsolutePath() {
    return file.getAbsolutePath();
  }

  /**
   * Define o tipo do arquivo.
   *
   * @param type o tipo.
   */
  public void setType(String type) {
    this.type = type;
  }

  /**
   * Informa o tipo do arquivo.
   *
   * @return o tipo.
   */
  @Override
  public String getType() {
    if (this.type == null) {
      String path = getAbsolutePath();
      String ext = getFileExtension(path);
      if (ext == null) {
        return FileTypes.UNKNOWN;
      }
      return getTypeFromExtension(ext);
    }
    else {
      return this.type;
    }
  }

  /**
   * Busca um tipo de arquivo aplicvel a extenso sugerida.
   *
   * @param extension extenso
   * @return o tipo
   */
  private static String getTypeFromExtension(String extension) {
    FileType fileType = FileTypes.getFileTypeFromExtension(extension);
    if (fileType == null) {
      return FileTypes.UNKNOWN;
    }
    return fileType.getName();
  }

  /**
   * Indica se este arquivo  um diretrio.
   *
   * @return verdadeiro caso seja um diretrio e falso caso contrrio.
   */
  @Override
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /**
   * Informa a posio corrente deste arquivo. Reflete o nmero de bytes a
   * partir do incio do arquivo at a posio corrente.
   *
   * @return a posio corrente do arquivo.
   *
   * @throws IOException se houver falha no acesso ao arquivo.
   */
  @Override
  public long position() throws IOException {
    return randomFile.getFilePointer();
  }

  /**
   * Altera a posio corrente deste arquivo. Reflete o nmero de bytes a partir
   * do incio do arquivo at a posio corrente.
   *
   * @param newPosition Nova posio corrente do arquivo.
   *
   * @throws IOException se houver falha no acesso ao arquivo.
   */
  @Override
  public void position(long newPosition) throws IOException {
    randomFile.seek(newPosition);
  }

  /**
   * Fecha este arquivo. Aps chamado este mtodo, o objeto que representa o
   * arquivo no poder mais ser utilizado para leitura e/ou escrita.
   *
   * @throws IOException se houver falha no acesso ao arquivo.
   *
   * @see Channel#close
   */
  @Override
  public void close(boolean forceQuit) throws IOException {
    if (this.randomFile != null) {
      RandomAccessFile raf = randomFile;
      if (forceQuit) {
        randomFile = null;
      }
      raf.close();
      randomFile = null;
    }
  }

  /**
   * Fecha o arquivo.
   *
   * @throws IOException em caso de erro de I/O.
   */
  public void close() throws IOException {
    close(true);
  }

  /**
   * Consulta o tamanho do arquivo.
   *
   * @return o tamanho.
   */
  @Override
  public long size() {
    return file.length();
  }

  /**
   * Construtor.
   *
   * @param file o arquivo local associado.
   */
  public RunnerFile(File file) {
    this.file = file;
    this.randomFile = null;
  }

  /**
   * Construtor.
   *
   * @param file o arquivo local associado.
   * @param fileType o tipo do arquivo.
   */
  public RunnerFile(File file, String fileType) {
    this.file = file;
    this.randomFile = null;
    this.type = fileType;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] getChildren() {
    if (!file.isDirectory()) {
      return null;
    }

    List<IFile> list = new ArrayList<>();
    for (File f : file.listFiles()) {
      list.add(new RunnerFile(f));
    }
    return list.toArray(new IFile[list.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    return new FileInputStream(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return file.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    return new FileOutputStream(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    return FileUtils.splitPath(file.getPath());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    return file.getAbsolutePath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(boolean readOnly) throws Exception {
    if (file.isDirectory()) {
      throw new Exception("Directory cannot be opened!");
    }
    String mode = readOnly ? "r" : "rws";
    randomFile = new RandomAccessFile(file, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate() {
    return file.lastModified();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len, long position) throws Exception {
    randomFile.seek(position);
    return randomFile.read(dst, off, len);
  }

  /**
   * L uma seqncia de bytes deste arquivo a partir da posio fornecida. Veja
   * a interface ReadableByteChannel para uma descrio completa da semntica
   * deste mtodo.
   *
   * @param dst O buffer no qual os bytes sero escritos.
   * @param position A posio do arquivo a partir da qual os bytes sero lidos.
   *
   * @return O nmero de bytes lidos ou -1 no caso de final de arquivo.
   *
   * @throws Exception se houver falha no acesso ao arquivo.
   */
  @Override
  public int read(byte[] dst, long position) throws Exception {
    randomFile.seek(position);
    return randomFile.read(dst);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException {
    randomFile.seek(position);
    randomFile.write(src, off, len);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position) throws IOException,
    FileLockedException {
    randomFile.seek(position);
    randomFile.write(src);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileLocationType getFileLocationType() {
    return FileLocationType.LOCAL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getParent() {
    return new RunnerFile(file.getParentFile());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean exists() throws IOException {
    return file.exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return file.canRead();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return file.canWrite();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return file.canExecute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFileLock acquireExclusiveLock(Window window) throws Exception {
    return new CSDKLocalFileLock(file, false, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFileLock acquireSharedLock(Window window) throws Exception {
    return new CSDKLocalFileLock(file, true, window);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getChild(final String name, Window window) throws Exception {
    if (name == null || name.trim().isEmpty()) {
      throw new Exception("Child name cannot be null");
    }

    if (isDirectory()) {
      File[] files = file.listFiles(
        (dir, fileName) -> dir.equals(file) && name.equals(fileName));
      if (files != null && files.length == 1) {
        return new RunnerFile(files[0]);
      }
      return null;
    }
    else {
      throw new Exception("File is not a directory: " + file.getAbsolutePath());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isHidden() {
    return file.isHidden();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean rename(String newName, Window window) throws Exception {
    return moveTo(getParent(), newName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean move(IFile newParent, Window window) throws Exception {
    return moveTo(newParent, getName());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile copy(IFile newParent, Window window) throws Exception {
    if (newParent == null) {
      throw new Exception("Parent cannot be null");
    }

    if (isDirectory()) {
      throw new Exception("Cannot copy directory");
    }

    if (!newParent.getFileLocationType().equals(getFileLocationType())) {
      throw new Exception("Parent on different file system");
    }

    if (!newParent.isDirectory()) {
      throw new Exception(
        "Parent is not a directory: " + newParent.getStringPath());
    }

    if (!file.exists()) {
      throw new Exception("File does not exist: " + file.getPath());
    }

    Path targetDir = ((RunnerFile) newParent).getFile().toPath();
    Path targetPath = targetDir.resolve(file.getName());
    Path copy = Files.copy(file.toPath(), targetPath);
    return new RunnerFile(copy.toFile());
  }

  /**
   * Move o arquivo.
   *
   * @param newParent o novo diretrio pai.
   * @param newName o novo nome do arquivo.
   * @return {@code true} se arquivo foi movido com sucesso ou {@code false},
   * caso contrrio.
   * @throws Exception em caso de falha na operao.
   */
  private synchronized boolean moveTo(IFile newParent,
    String newName) throws Exception {
    if (randomFile != null) {
      throw new Exception("Cannot move open file");
    }

    if (!file.exists()) {
      throw new Exception("File does not exist: " + getStringPath());
    }

    if (newParent == null) {
      throw new Exception("Parent cannot be null");
    }

    if (newName == null) {
      throw new Exception("Name cannot be null");
    }

    if (isDirectory()) {
      throw new Exception("Cannot move directory!");
    }

    if (!newParent.getFileLocationType().equals(getFileLocationType())) {
      throw new Exception("Parent on different file system");
    }

    if (!newParent.isDirectory()) {
      throw new Exception(
        "Parent is not a directory: " + newParent.getStringPath());
    }

    Path path = file.toPath();
    File dir = new File(newParent.getStringPath());
    Path target = dir.toPath().resolve(newName);
    try {
      Path newPath = Files.move(path, target);
      if (newPath != null) {
        this.file = newPath.toFile();
        return true;
      }
      return false;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean delete(Window window) throws Exception {
    if (randomFile != null) {
      throw new Exception("Cannot delete open file");
    }
    return file.delete();
  }

}
