package csdk.v2.runner.filesystem;

import java.awt.Window;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.text.MessageFormat;

import csdk.v2.api.filesystem.IFile;
import csdk.v2.api.filesystem.IFileLock;
import csdk.v2.runner.CSDKLogger;

/**
 * Representa um lock de um {@link IFile arquivo}.
 */
public class CSDKLocalFileLock implements IFileLock {

  /**
   * Lock real.
   */
  private final FileLock lock;

  /**
   * Canal para manipulao do arquivo.
   */
  private FileChannel channel;

  /**
   * Construtor.
   *
   * @param file o arquivo a ser bloqueado.
   * @param shared determina se o lock deve ser do tipo compartilhado.
   * @param window janela me da ao de lock.
   * @throws Exception em caso de erro ao criar o lock.
   */
  public CSDKLocalFileLock(final File file, final boolean shared,
    final Window window) throws Exception {
    if (file.isDirectory()) {
      String pattern = "A directory cannot be locked ({0})";
      String message = MessageFormat.format(pattern, file.getAbsolutePath());
      throw new Exception(message);
    }
    this.lock = lock(shared, window, file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public LockStatus getLockStatus() {
    if (lock != null) {
      if (this.lock.isValid()) {
        if (this.lock.isShared()) {
          return LockStatus.LOCK_SHARED;
        }
        else {
          return LockStatus.LOCK_EXCLUSIVE;
        }
      }
      else {
        return LockStatus.LOCK_RELEASED;
      }
    }
    else {
      return LockStatus.LOCK_DENIED;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void releaseLock(final Window window) {
    try {
      if (lock != null && lock.isValid()) {
        lock.release();
      }
    }
    catch (IOException e) {
      CSDKLogger logger = CSDKLogger.getInstance();
      logger.logSevere("Lock release failed");
    }
    finally {
      FileUtils.close(channel);
    }
  }

  /**
   * Obtm o lock do arquivo.
   *
   * @param shared determina se o lock deve ser do tipo compartilhado.
   * @param window janela me da ao de lock.
   * @param file arquivo a ser bloqueado.
   * @return o lock do arquivo.
   */
  @SuppressWarnings("resource")
  private FileLock lock(final boolean shared, final Window window,
    final File file) {
    RandomAccessFile randomAccessFile = null;
    try {
      final String mode = shared ? "r" : "rws";
      randomAccessFile = new RandomAccessFile(file, mode);
      this.channel = randomAccessFile.getChannel();
      return this.channel.tryLock(0, Long.MAX_VALUE, shared);
    }
    catch (OverlappingFileLockException e) {
      CSDKLogger logger = CSDKLogger.getInstance();
      String message = "Lock for file {0} already exists in current VM.";
      logger.logSevere(message, file.getPath());
      FileUtils.close(randomAccessFile);
      return null;
    }
    catch (Exception e) {
      FileUtils.close(randomAccessFile);
      return null;
    }
  }

}
