package csdk.v1_0.runner;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.PropertyResourceBundle;

import javax.swing.ImageIcon;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import csdk.v1_0.runner.application.ApplicationClassLoader;

/**
 * Registro da aplicao no {@link Runner} do CSDK.
 *
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class ApplicationRegistry implements Serializable {

  /**
   * Id de serialiazao.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Bundle interno da aplicao.
   */
  private transient PropertyResourceBundle bundle = null;

  /**
   * Tabela da propriedades da aplicao preenchida de acordo com a
   * parametrizao do {@link Runner}.
   */
  final private HashMap<String, String> applicationProperties =
    new HashMap<String, String>();

  /**
   * O classloader da aplicao.
   */
  private transient ClassLoader classloader;

  /**
   * Identificador da aplicao
   */
  final private String id;

  /**
   * Nome da classe que representa a aplicao.
   */
  private String className;

  /**
   * Indicativo de carga de bundle interno da aplicao.
   */
  private boolean isBundleRequired = false;

  /**
   * Nome da aplicao.
   */
  private String applicationName;

  /**
   * cone da aplicao.
   */
  private ImageIcon applicationIcon;

  /**
   * cone reduzido da aplicao.
   */
  private ImageIcon smallApplicationIcon;

  /**
   * Verso da aplicao.
   */
  private String version;

  /**
   * Tipos de arquivo associados  aplicao.
   */
  private String[] fileTypes;

  /**
   * O autor da aplicao.
   */
  private String author;

  /**
   * O e-mail do autor para contato.
   */
  private String authorEmail;

  /**
   * A descrio textual da aplicao.
   */
  private String description;

  /**
   * Indica se a aplicao s pode ter uma nica instncia.
   */
  private boolean singleton;

  /**
   * Indica se a aplicao precisa de um projeto para executar.
   */
  private boolean requiresProject;

  /**
   * O classpath da aplicao.
   */
  private List<URL> classpath;

  /**
   * Caminho para o cone da aplicao.
   */
  private String iconPath;

  /**
   * Caminho para o cone reduzido da aplicao.
   */
  private String smallIconPath;

  /**
   * Construtor.
   *
   * @param id identificador da aplicao.
   */
  public ApplicationRegistry(String id) {
    this.id = id;
  }

  /**
   * Carga do pacote de internacionalizao da aplicao.
   *
   * @return o pacote.
   */
  @SuppressWarnings("resource")
  public PropertyResourceBundle loadInternalBundle() {
    if (classloader == null) {
      return null;
    }
    if (isBundleRequired()) {
      Locale locale = LNG.getLocale();
      String language = locale.getLanguage();
      String country = locale.getCountry();

      String sep = "/";
      StringBuilder pathBuilder = new StringBuilder();
      pathBuilder.append(getResourceBaseDir());
      pathBuilder.append(sep);
      pathBuilder.append("resources");
      pathBuilder.append(sep);
      pathBuilder.append(getSimpleClassName());
      pathBuilder.append("_");
      pathBuilder.append(language);
      pathBuilder.append("_");
      pathBuilder.append(country);
      pathBuilder.append(".properties");
      String resourcePath = pathBuilder.toString();
      InputStream in = classloader.getResourceAsStream(resourcePath);
      if (in == null) {
        String err = "internal bundle not found: " + resourcePath;
        throw new IllegalStateException(err);
      }
      try {
        return new PropertyResourceBundle(in);
      }
      catch (IOException e) {
        String err = "internal bundle failure error";
        throw new IllegalStateException(err, e);
      }
      finally {
        FileUtils.close(in);
      }
    }
    return null;
  }

  /**
   * Obtm o caminho-base para busca dos recursos da aplicao.
   *
   * @return o caminho.
   */
  private String getResourceBaseDir() {
    String packageName = getPackageName();
    return packageName.replace(".", "/");
  }

  /**
   * Obtm o classloader da aplicao.
   *
   * @return o classloader.
   */
  public ClassLoader getClassloader() {
    if (classloader == null) {
      this.classloader = createClassLoader();
    }
    return classloader;
  }

  /**
   * Cria o {@link ClassLoader} da aplicao.
   *
   * @return o classloader.
   */
  private ClassLoader createClassLoader() {
    List<URL> paths = getClasspath();
    URL[] urls;
    if (paths == null) {
      urls = new URL[0];
    }
    else {
      urls = paths.toArray(new URL[paths.size()]);
    }
    ClassLoader loader =
      new ApplicationClassLoader(urls, getClass().getClassLoader());
    return loader;
  }

  /**
   * Busca uma imagem para a aplicao.
   *
   * @param path caminho dentro do diretrio de imagens da aplicao.
   * @return a imagem ou null caso no seja encontrada.
   */
  public ImageIcon getImageIcon(String[] path) {
    if (classloader == null) {
      return null;
    }
    try {
      String sep = "/";
      StringBuilder resPath = new StringBuilder();
      resPath.append(getResourceBaseDir());
      resPath.append(sep);
      resPath.append("resources");
      resPath.append(sep);
      resPath.append("images");
      resPath.append(sep);
      for (String pathElement : path) {
        resPath.append(pathElement);
        resPath.append(sep);
      }
      resPath.deleteCharAt(resPath.length() - 1);
      URL res = classloader.getResource(resPath.toString());
      return new ImageIcon(res);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Obtm o pacote da classe principal da aplicao.
   *
   * @return o pacote.
   */
  private String getPackageName() {
    String packageName;
    int index = className.lastIndexOf(".");
    if (index != -1) {
      packageName = className.substring(0, index);
    }
    else {
      packageName = "";
    }
    return packageName;
  }

  /**
   * Obtm o nome simples (sem pacote) da classe principal da aplicao.
   *
   * @return o nome.
   */
  private String getSimpleClassName() {
    String simpleClassName;
    int index = className.lastIndexOf(".");
    if (index != -1) {
      simpleClassName = className.substring(index + 1, className.length());
    }
    else {
      simpleClassName = className;
    }
    return simpleClassName;
  }

  /**
   * Consulta da existncia de uma string de idioma.
   *
   * @param key a chave de busca.
   * @return o texto no idioma correto.
   */
  public boolean hasString(String key) {
    if (!isBundleRequired()) {
      return false;
    }
    if (bundle == null) {
      bundle = loadInternalBundle();
    }
    boolean containsKey = bundle.containsKey(key);
    return containsKey;
  }

  /**
   * Obtm uma string de idioma.
   *
   * @param key a chave de busca.
   * @return o texto no idioma correto.
   */
  public String getString(String key) {
    if (!isBundleRequired()) {
      String err = "Not configured for bundles (--app-bundle)!";
      throw new IllegalStateException(err);
    }
    if (key == null) {
      String err = "Bundle key cannot be null.";
      throw new IllegalArgumentException(err);
    }
    if (bundle == null) {
      bundle = loadInternalBundle();
    }
    try {
      return bundle.getString(key);
    }
    catch (MissingResourceException e) {
      CSDKLogger logger = CSDKLogger.getInstance();
      String keyHolder = "<<<" + key + ">>>";
      logger.log("Key " + keyHolder + " not defined on language bundle.");
      return keyHolder;
    }
  }

  /**
   * Obtm uma string de idioma formatada com os argumentos definidos.
   *
   * @param key a chave de busca.
   * @param args os argumentos de formatao
   * @return o texto no idioma correto.
   */
  public String getString(String key, Object[] args) {
    String pattern = getString(key);
    return MessageFormat.format(pattern, args);
  }

  /**
   * Indicativo de propriedade sem valor.
   *
   * @param propName nome da propriedade.
   * @return indicativo
   */
  public boolean isPropertyNull(String propName) {
    String tag = id + "." + propName;
    if (!applicationProperties.containsKey(tag)) {
      return true;
    }
    String value = applicationProperties.get(tag);
    if (value.trim().isEmpty()) {
      return true;
    }
    return false;
  }

  /**
   * Obtm o valor de uma propriedade especfica.
   *
   * @param propName nome da propriedade.
   * @return o valor da propriedade.
   *
   * @throws IllegalStateException se a propriedade no foi definida
   */
  public String getProperty(String propName) {
    String tag = id + "." + propName;
    if (isPropertyNull(propName)) {
      String err = "Prop. indefinida [" + tag + "] para apl. " + id;
      throw new IllegalStateException(err);
    }

    String value = applicationProperties.get(tag);
    if (value == null) {
      return null;
    }
    if (value.isEmpty()) {
      return null;
    }
    return value.trim();
  }

  /**
   * Define o nome qualificado da classe principal da aplicao.
   *
   * @param className o nome da classe.
   */
  public void setClassName(String className) {
    this.className = className;
  }

  /**
   * Determina se a aplicao usa o pacote de internacionalizao.
   *
   * @param requiresBundle <code>true</code> se a aplicao usa o pacote de
   *        internacionalizao ou <code>false</code> caso contrrio.
   */
  public void setBundleRequired(boolean requiresBundle) {
    this.isBundleRequired = requiresBundle;
  }

  /**
   * Define o nome da aplicao.
   *
   * @param name o nome da aplicao.
   */
  public void setApplicationName(String name) {
    this.applicationName = name;
  }

  /**
   * Determina se aplicao tem alguma propriedade ajustada.
   *
   * @return <code>true</code> se aplicao no tem propriedades definidas ou
   *         <code>false</code> caso contrrio.
   */
  public boolean hasNoProperty() {
    return applicationProperties.size() == 0;
  }

  /**
   * Obtm o id da aplicao.
   *
   * @return o id com uma string.
   */
  public String getApplicationId() {
    return id;
  }

  /**
   * Obtm o nome da classe principal da aplicao.
   *
   * @return a classe.
   */
  public String getClassName() {
    return className;
  }

  /**
   * Determina se o pacote de idiomas   utilizado pela aplicao.
   *
   * @return indicativo.
   */
  public boolean isBundleRequired() {
    return isBundleRequired;
  }

  /**
   * Obtm o nome da aplicao.
   *
   * @return o nome.
   */
  public String getApplicationName() {
    return applicationName;
  }

  /**
   * Obtm o cone da aplicao.
   *
   * @return o cone da aplicao.
   */
  public ImageIcon getApplicationIcon() {
    return applicationIcon;
  }

  /**
   * Obtm o cone reduzido da aplicao.
   *
   * @return o cone reduzido da aplicao.
   */
  public ImageIcon getSmallApplicationIcon() {
    return smallApplicationIcon;
  }

  /**
   * Define o cone da aplicao.
   *
   * @param iconPath o caminho para o cone.
   */
  public void setApplicationIconPath(String iconPath) {
    this.iconPath = iconPath;
    this.applicationIcon = new ImageIcon(iconPath);
  }

  /**
   * Obtm o caminho para o cone da aplicao.
   *
   * @return o caminho.
   */
  public String getApplicationIconPath() {
    return this.iconPath;
  }

  /**
   * Obtm o caminho para o cone reduzido da aplicao.
   *
   * @return o caminho.
   */
  public String getSmallApplicationIconPath() {
    return this.smallIconPath;
  }

  /**
   * Define o cone reduzido da aplicao.
   *
   * @param iconPath o caminho para o cone.
   */
  public void setSmallApplicationIconPath(String iconPath) {
    this.smallIconPath = iconPath;
    this.smallApplicationIcon = new ImageIcon(iconPath);
  }

  /**
   * Obtm o identificador da aplicao.
   *
   * @return id da aplicao.
   */
  protected String getId() {
    return id;
  }

  /**
   * Atribui o valor da propriedade da aplicao.
   *
   * @param key a chave da propriedade.
   * @param value o valor da propriedade.
   */
  public void setProperty(String key, String value) {
    applicationProperties.put(key, value);
  }

  /**
   * Atribui a verso da aplicao.
   *
   * @param version a verso.
   */
  public void setVersion(String version) {
    this.version = version;
  }

  /**
   * Obtm a verso da aplicao.
   *
   * @return a verso.
   */
  public String getVersion() {
    if (version == null) {
      return "0.0.0";
    }
    return version;
  }

  /**
   * Atribui os tipos de arquivo associdados  aplicao.
   *
   * @param fileTypes os tipos de arquivo.
   */
  public void setFileTypes(String[] fileTypes) {
    this.fileTypes = Arrays.copyOf(fileTypes, fileTypes.length);
  }

  /**
   * Atribui os tipos de arquivo associdados  aplicao.
   *
   * @return file os tipos de arquivo.
   */
  public String[] getFileTypes() {
    if (fileTypes != null) {
      return Arrays.copyOf(fileTypes, fileTypes.length);
    }
    else {
      return new String[0];
    }
  }

  /**
   * Obtm o autor da aplicao.
   *
   * @return o autor da aplicao.
   */
  public String getAuthor() {
    return this.author;
  }

  /**
   * Obtm o endereo de contato do autor da aplicao.
   *
   * @return o endereo.
   */
  public String getAuthorEmail() {
    return this.authorEmail;
  }

  /**
   * Obtm a descrio da aplicao.
   *
   * @return a descrio.
   */
  public String getApplicationDescription() {
    return this.description;
  }

  /**
   * Indica se a aplicao s pode ter uma nica instncia ativa.
   *
   * @return <code>true</code> se a aplicao s pode ter uma nica instncia ou
   *         <code>false</code>, caso contrrio.
   */
  public boolean isSingleton() {
    return this.singleton;
  }

  /**
   * Atribui a descrio da aplicao.
   *
   * @param description a descrio.
   */
  protected void setApplicationDescription(String description) {
    this.description = description;
  }

  /**
   * Atribui o autor da aplicao.
   *
   * @param author o autor.
   */
  protected void setAuthor(String author) {
    this.author = author;
  }

  /**
   * Atribui o e-mail de contato do autor da aplicao.
   *
   * @param email o e-mail.
   */
  protected void setAuthorEmail(String email) {
    this.authorEmail = email;
  }

  /**
   * Determina se a aplicao s pode ter uma nica instncia.
   *
   * @param singleton <code>true</code> se a aplicao s pode ter uma nica
   *        instncia ou <code>false</code>, caso contrrio.
   */
  protected void setSingleton(boolean singleton) {
    this.singleton = singleton;
  }

  /**
   * Indica se a aplicao s pode ser executada com um projeto aberto.
   *
   * @return <code>true</code> se a aplicao precisa de um projeto para
   *         executar ou <code>false</code>, caso contrrio.
   */
  public boolean requiresProject() {
    return this.requiresProject;
  }

  /**
   * Determina se a aplicao s pode ser executada com um projeto aberto.
   *
   * @param requiresProject <code>true</code> se a aplicao precisa de um
   *        projeto para executar ou <code>false</code>, caso contrrio.
   */
  public void setRequiresProject(boolean requiresProject) {
    this.requiresProject = requiresProject;
  }

  /**
   * Obtm o classpath da aplicao.
   *
   * @return classpath
   */
  public List<URL> getClasspath() {
    return classpath;
  }

  /**
   * Define o classpath da aplicao.
   *
   * @param paths o classpath.
   */
  public void setClasspath(List<URL> paths) {
    this.classpath = paths;
  }
}
