package csdk.v1_0.runner.core;

import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.UIManager;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;

/**
 * Dilogo de erro padro do Runner.
 *
 * @author Tecgraf
 */
public class RunnerErrorDialog extends JDialog {

  /**
   * A cor do fundo dos painis.
   */
  private static final Color BACKGROUND_COLOR = UIManager.getDefaults()
    .getColor("TextField.inactiveBackground");

  /** Nmero de colunas */
  private static final int COLUMNS = 80;

  /** Nmero de linhas do campo de mensagem */
  private static final int MESSAGE_ROWS = 4;

  /** Nmero de linhas do campo de pilha */
  private static final int STACK_ROWS = 14;

  /** Erro ou exceo a ser visualizada na janela */
  protected final Throwable error;

  /** Conjunto de informaes adicionais que sero mostradas na janela de erro. */
  protected final String[] additionalInfo;

  /**
   * Contrutor.
   *
   * @param owner a janela me.
   * @param title ttulo do dilogo.
   * @param error exceo a ser visualizada na janela.
   * @param additionalInfo informaes adicionais que sero mostradas na janela
   *        de erro.
   */
  public RunnerErrorDialog(Window owner, String title, Throwable error,
    String[] additionalInfo) {
    super(owner, title);
    this.error = error;
    this.additionalInfo = additionalInfo;
    this.setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    createComponents();
  }

  /**
   * Mtodo interno de construo da inteface grfica.
   */
  private void createComponents() {

    String lngPrefix = RunnerErrorDialog.class.getSimpleName();

    final JLabel messageLabel = new JLabel(LNG.get(lngPrefix + ".message"));
    JScrollPane msgScrollPane = createMessagePane();

    final JLabel stackTraceLabel = new JLabel(LNG.get(lngPrefix + ".stack"));
    JScrollPane stackTraceScroll = createStackPane();
    JLabel label = new JLabel(LNG.get(lngPrefix + ".info"));
    JScrollPane scrollPane = createInfoPane();

    setLayout(new GridBagLayout());

    Insets li = new Insets(5, 6, 0, 10);
    Insets fi = new Insets(3, 10, 5, 10);

    int y = 0;
    add(label, new GBC(0, y++).west().insets(li));
    add(scrollPane, new GBC(0, y++).both().insets(fi));

    add(messageLabel, new GBC(0, y++).west().insets(li));
    add(msgScrollPane, new GBC(0, y++).both().insets(fi));

    add(stackTraceLabel, new GBC(0, y++).west().insets(li));
    add(stackTraceScroll, new GBC(0, y++).both().insets(fi));

    pack();
    GUIUtils.centerWindow(getOwner(), this);
  }

  /**
   * Cria o painel com as informaes adicionais.
   *
   * @return o painel.
   */
  private JScrollPane createInfoPane() {
    int rows = 3;
    if (additionalInfo != null && additionalInfo.length != 0) {
      rows = additionalInfo.length + 1;
    }

    JTextArea additionalInfoTextArea = new JTextArea(rows, 80);
    additionalInfoTextArea.setEditable(false);
    additionalInfoTextArea.setBackground(BACKGROUND_COLOR);
    additionalInfoTextArea.setText(createAdditionalInfoText());
    return new JScrollPane(additionalInfoTextArea);
  }

  /**
   * Cria o painel com a pilha de execuo.
   *
   * @return o painel.
   */
  private JScrollPane createStackPane() {
    JTextArea stackTraceTextArea = new JTextArea(getStackTraceText());
    stackTraceTextArea.setColumns(COLUMNS);
    stackTraceTextArea.setRows(STACK_ROWS);
    stackTraceTextArea.setEditable(false);
    stackTraceTextArea.setBackground(BACKGROUND_COLOR);
    return new JScrollPane(stackTraceTextArea);
  }

  /**
   * Cria o painel com a mensagem de erro.
   *
   * @return o painel.
   */
  private JScrollPane createMessagePane() {
    /** rea de texto de mensagens */
    JTextArea messageTextArea = new JTextArea(this.error.getLocalizedMessage());
    messageTextArea.setColumns(COLUMNS);
    messageTextArea.setRows(MESSAGE_ROWS);
    messageTextArea.setEditable(false);
    messageTextArea.setBackground(BACKGROUND_COLOR);
    return new JScrollPane(messageTextArea);
  }

  /**
   * Monta a string que representa as informaes adicionais sobre o contexto do
   * erro.
   *
   * @return texto formatado com as informaes adicionais.
   */
  private String createAdditionalInfoText() {
    StringBuilder info = new StringBuilder();
    if (additionalInfo != null && additionalInfo.length > 0) {
      for (int i = 0; i < additionalInfo.length; i++) {
        info.append(additionalInfo[i]);
        info.append("\n");
      }
    }
    return info.toString();
  }

  /**
   * Monta a string que representa a pilha de execuo.
   *
   * @return a string com a pilha.
   */
  private String getStackTraceText() {
    Writer writer = new StringWriter();
    error.printStackTrace(new PrintWriter(writer));
    return writer.toString();
  }

}
