package csdk.v1_0.runner;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.MessageFormat;
import java.util.Date;
import java.util.logging.ConsoleHandler;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

/**
 * Logger usado pelo {@link Runner} do CSDK.
 */
public class CSDKLogger {

  /**
   * Nvel padro do {@link Logger}.
   */
  private static final Level DEFAULT_LOGGER_LEVEL = Level.SEVERE;

  /**
   * Logger do CSDK.
   */
  private final Logger logger;

  /**
   * Instncia nico do {@link CSDKLogger}.
   */
  private static CSDKLogger instance;

  /**
   * Construtor.
   *
   */
  private CSDKLogger() {
    this.logger = Logger.getLogger(Logger.GLOBAL_LOGGER_NAME);
    setVerbosed(false);
    Formatter formatter = new CSDKLoggerFormatter();
    ConsoleHandler handler = new ConsoleHandler();
    handler.setFormatter(formatter);
    this.logger.addHandler(handler);
    this.logger.setUseParentHandlers(false);
  }

  /**
   * Obtm a instncia nica de {@link CSDKLogger}.
   *
   * @return a instncia de CSDKLogger.
   */
  public static CSDKLogger getInstance() {
    if (instance == null) {
      instance = new CSDKLogger();
    }
    return instance;
  }

  /**
   * Ajuste de indicao de verbose
   *
   * @param verbosed o indicativo a ser ajustado
   */
  public void setVerbosed(boolean verbosed) {
    Level level;
    if (verbosed) {
      level = Level.ALL;
    }
    else {
      level = DEFAULT_LOGGER_LEVEL;
    }
    this.logger.setLevel(level);
  }

  /**
   * Escreve uma mensagem de log de nvel normal.
   *
   * @param text mensagem.
   * @param args argumentos para formatao do texto.
   */
  public void log(String text, Object... args) {
    String message = MessageFormat.format(text, args);
    logger.log(Level.INFO, message);
  }

  /**
   * Escreve uma mensagem de log de nvel grave.
   *
   * @param text mensagem.
   * @param args argumentos para formatao do texto.
   */
  public void logSevere(String text, Object... args) {
    String message = MessageFormat.format(text, args);
    logger.log(Level.SEVERE, message);
  }

  /**
   * Escreve uma mensagem de log com o erro ocorrido.
   *
   * @param e o erro.
   */
  public void logException(Exception e) {
    String exceptionDetails = extractErrorInfo(e);
    logSevere(exceptionDetails);
    Throwable cause = e.getCause();
    if (cause != null) {
      logSevere("Caused by:");
      String causeInfo = extractErrorInfo(cause);
      logSevere(causeInfo);
    }
  }

  /**
   * Extrai as informaes da exceo para o log.
   *
   * @param e a exceo.
   * @return as informaes extradas.
   */
  private String extractErrorInfo(Throwable e) {
    StringWriter sw = new StringWriter();
    String message = e.getMessage();
    if (message != null) {
      sw.append("Exception message: ");
      sw.append(message);
      sw.append("\n");
    }
    e.printStackTrace(new PrintWriter(sw));
    String exceptionDetails = sw.toString();
    return exceptionDetails;
  }

  /**
   * Formatador de mensagens do log.
   */
  private final static class CSDKLoggerFormatter extends Formatter {
    /**
     * Data da mensagem no log.
     */
    private final Date date = new Date();

    /**
     * {@inheritDoc}
     */
    @Override
    public String format(LogRecord record) {
      StringBuilder b = new StringBuilder();
      date.setTime(record.getMillis());
      b.append(date);
      b.append(" [CSDK] ");
      b.append(record.getMessage());
      b.append(System.getProperty("line.separator"));
      return b.toString();
    }
  }

}
