package csdk.v1_0.runner;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import javax.swing.SwingUtilities;

import csdk.v1_0.runner.application.RunnerApplication;
import csdk.v1_0.runner.filesystem.FileType;
import csdk.v1_0.runner.filesystem.FileTypes;
import csdk.v1_0.runner.filesystem.FileUtils;

/**
 * Classe que representa o executor (<i>runner</i>) de simulaco do CSDK.
 *
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class Runner {

  /**
   * Identificador da aplicao
   */
  private String startingId;

  /**
   * Locale usado pelo Runner.
   */
  private static Locale locale = Locale.getDefault();

  /**
   * Propriedades do Runner.
   */
  private final Properties runnerProperties;

  /**
   * O charset do ambiente.
   */
  private Charset charset = Charset.defaultCharset();

  /**
   * Construtor
   */
  private Runner() {
    this.runnerProperties = new Properties();
  }

  /**
   * Entry point.
   *
   * @param args argumentos da linha de comando.
   */
  static public void main(String[] args) {
    final Runner runner = new Runner();

    runner.showHeader();

    // Teste de exibio de help (apenas).
    if (runner.isOnlyHelp(args)) {
      runner.showHelp();
      runner.logFinish(true);
      return;
    }

    loadDefaultFileTypes();

    // Verificao de parmetros
    try {
      runner.checkArguments(args);
    }
    catch (RuntimeException e) {
      runner.logSevere("Command line parametrization error. Check arguments!");
      runner.logException(e);
      runner.logFinish(false);
      return;
    }

    final ApplicationManager appManager = ApplicationManager.getInstance();
    Properties props = runner.getRunnerProperties();
    appManager.addContextProperties(props);

    String startId = runner.getStartingApplicationId();
    if (startId == null) {
      Set<String> appIds = appManager.getAllAplicationsIds();
      switch (appIds.size()) {
        case 0:
          runner.logSevere("No applications defined! Aborting...");
          runner.logFinish(false);
          return;
        case 1:
          startId = appIds.iterator().next();
          break;
        default:
          runner.logSevere(
            "Multiple applications found and no starting application defined! Aborting...");
          runner.logFinish(false);
          return;
      }
    }
    final String appId = startId;
    ApplicationRegistry reg = runner.getApplicationRegistry(startId);
    if (reg == null) {
      runner.logSevere("Starting application not found! Aborting...");
      runner.logFinish(false);
      return;
    }

    runner.log("Starting main application...");
    runner.log("  * Application class name: {0}", reg.getClassName());
    runner.log("  * Application id: {0}", reg.getApplicationId());
    // Aviso de que nenhuma propriedade foi setada.
    if (reg.hasNoProperty()) {
      runner.log("  * No property set for main application.");
    }

    Runtime.getRuntime().addShutdownHook(new Thread() {
      @Override
      public void run() {
        appManager.finishAllApplications();
      }
    });

    // Incio da execuo
    runner.log("Starting main application...");
    try {
      SwingUtilities.invokeAndWait(new Runnable() {
        @Override
        public void run() {
          RunnerApplication startApp =
            appManager.runApplication(appId, runner.getCharset());
          if (startApp == null) {
            throw new RunnerException("Main application execution failed!");
          }
          runner.logSevere("Main application started!");
        }
      });
    }
    catch (Exception e) {
      runner.logException(e);
      runner.logFinish(false);
      return;
    }
    while (appManager.hasActiveApplications()) {
      try {
        Thread.sleep(1000);
      }
      catch (InterruptedException e) {
        runner.logException(e);
      }
    }
    runner.log("No more active applications");
    runner.logFinish(true);
  }

  /**
   * Carrega um arquivo de propriedades que define os tipos de arquivo do
   * ambiente.
   *
   * @param path caminho para o arquivo.
   */
  private void loadFileTypes(String path) {
    InputStream fileTypeDefinition;
    try {
      fileTypeDefinition = new FileInputStream(path);
    }
    catch (FileNotFoundException e) {
      throw new RunnerException(e, "File type list not found: {0}", path);
    }
    FileTypes.loadFileTypes(fileTypeDefinition);
  }

  /**
   * Carrega o arquivo padro de propriedades que definem os tipos de arquivo do
   * ambiente.
   */
  private static void loadDefaultFileTypes() {
    InputStream defaultFileTypeDefinition = FileTypes.class
      .getResourceAsStream(FileTypes.class.getSimpleName() + ".properties");
    FileTypes.loadFileTypes(defaultFileTypeDefinition);
  }

  /**
   * Obtm o charset do ambiente.
   *
   * @return o charset.
   */
  private Charset getCharset() {
    return charset;
  }

  /**
   * Construo de um registry.
   *
   * @param id identificador da aplicao.
   */
  private void buildNewApplicationRegistry(String id) {
    if (id != null) {
      String appId = id.trim();
      ApplicationManager applicationManager = ApplicationManager.getInstance();
      if (!applicationManager.hasApplicationRegistry(appId)) {
        ApplicationRegistry reg = new ApplicationRegistry(appId);
        applicationManager.addApplicationRegistry(reg);
      }
    }
  }

  /**
   * Chacagem de argumentos da linha de comando.
   *
   * @param args os argumentos
   */
  private void checkArguments(String[] args) {
    int i = 0;
    int N = args.length;
    while (i < N) {
      String arg = args[i].trim();
      if (arg.equals("--locale")) {
        checkLocaleArgument(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--verbose")) {
        setVerbosed(true);
      }
      else if (arg.equals("--charset")) {
        checkCharsetArgument(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--file-types-property-file")) {
        loadFileTypes(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--file-type")) {
        checkFileTypeArgument(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-property")) {
        setApplicationProperty(args[i + 1], args[i + 2], args[i + 3]);
        i = i + 3;
      }
      else if (arg.equals("--runner-property")) {
        setRunnerProperty(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-bundle")) {
        setBundleLoaded(args[i + 1], true);
        i = i + 1;
      }
      else if (arg.equals("--app-singleton")) {
        setSingleton(args[i + 1], true);
        i = i + 1;
      }
      else if (arg.equals("--app-require-project")) {
        setRequireProject(args[i + 1], true);
        i = i + 1;
      }
      else if (arg.equals("--start-id")) {
        setStartingApplicationId(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--context-factory")) {
        setContextFactory(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--app-id")) {
        buildNewApplicationRegistry(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--app-name")) {
        setApplicationName(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-description")) {
        setDescription(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-author")) {
        setAuthorName(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-author-mail")) {
        setAuthorMail(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-icon")) {
        setApplicationIcon(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-small-icon")) {
        setSmallApplicationIcon(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-package")) {
        loadApplicationFromPackage(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-version")) {
        setApplicationVersion(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-class")) {
        setClassName(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-file-types")) {
        setApplicationFileTypes(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-classpath")) {
        File baseDir = new File(".");
        setApplicationClasspath(args[i + 1], baseDir, args[i + 2]);
        i = i + 2;
      }
      else {
        logSevere("Unrecognized argument found: {0}. Ignoring...", arg);
      }
      i = i + 1;
    }
  }

  /**
   * Ajuste de indicao de verbose
   *
   * @param verbosed o indicativo a ser ajustado
   */
  private void setVerbosed(boolean verbosed) {
    CSDKLogger instance = CSDKLogger.getInstance();
    instance.setVerbosed(verbosed);
  }

  /**
   * Define o caminho para a aplicao empacotada.
   *
   * @param id o identificador da aplicao.
   * @param appPath o caminho para o caminho da aplicao.
   */
  private void loadApplicationFromPackage(String id, String appPath) {
    if (id != null && appPath != null) {
      String appId = id.trim();
      String packagePath = appPath.trim();
      buildNewApplicationRegistry(appId);
      loadPropertiesFromPackage(appId, packagePath);
      loadIconsFromPackage(appId, packagePath);
    }
  }

  /**
   * Carrega as propriedades bsicas da aplicao.
   *
   * @param id o identificador da aplicao.
   * @param appPath o caminho para a aplicao empacotada.
   */
  private void loadPropertiesFromPackage(String id, String appPath) {
    InputStreamReader reader = null;
    try {
      File path = new File(appPath);
      String propertyFileName = id + ".properties";
      File propertyFile = new File(path, propertyFileName);
      InputStream input = new FileInputStream(propertyFile);
      reader = new InputStreamReader(input, charset);
      Properties props = new Properties();
      props.load(reader);
      for (ApplicationParameters param : ApplicationParameters.values()) {
        String key = param.getApplicationPropertyName(id);
        switch (param) {
          case APP_NAME_PROPERTY:
            String localizedName = getLocalizedProperty(key, props);
            setApplicationName(id, localizedName);
            break;
          case CLASS_NAME_PROPERTY:
            setClassName(id, (String) props.get(key));
            break;
          case FILE_TYPES_PROPERTY:
            setApplicationFileTypes(id, (String) props.get(key));
            break;
          case VERSION_PROPERTY:
            setApplicationVersion(id, (String) props.get(key));
            break;
          case AUTHOR_NAME_PROPERTY:
            setAuthorName(id, (String) props.get(key));
            break;
          case AUTHOR_MAIL_PROPERTY:
            setAuthorMail(id, (String) props.get(key));
            break;
          case DESCRIPTION_PROPERTY:
            String localizedDescription = getLocalizedProperty(key, props);
            setDescription(id, localizedDescription);
            break;
          case NEED_BUNDLE_PROPERTY:
            String bundleValue = (String) props.get(key);
            if (!isNull(bundleValue)) {
              setBundleLoaded(id, Boolean.valueOf(bundleValue));
            }
            break;
          case IS_SINGLETON_PROPERTY:
            String singletonValue = (String) props.get(key);
            if (!isNull(singletonValue)) {
              setSingleton(id, Boolean.valueOf(singletonValue));
            }
            break;
          case REQUIRE_PROJECT_PROPERTY:
            String requireValue = (String) props.get(key);
            if (!isNull(requireValue)) {
              setRequireProject(id, Boolean.valueOf(requireValue));
            }
            break;
          case CLASSPATH_PROPERTY:
            String[] classpath = getNumberedProperty(key, props);
            if (classpath.length != 0) {
              setApplicationClasspath(id, path, classpath);
            }
            else {
              String classpathAsString = (String) props.get(key);
              setApplicationClasspath(id, path, classpathAsString);
            }
            break;
        }
        props.remove(key);
      }
      loadSpecificApplicationProperties(id, props);
    }
    catch (FileNotFoundException e) {
      String msg = "Application definition file in package ({0}) not found!";
      throw new RunnerException(e, msg, appPath);
    }
    catch (Exception e) {
      String msg =
        "Error reading application definition file in package ({0}).";
      throw new RunnerException(e, msg, appPath);
    }
    finally {
      if (reader != null) {
        try {
          reader.close();
        }
        catch (IOException e) {
          logException(e);
        }
      }
    }
  }

  /**
   * Obtm uma lista de valores de propriedades que seguem o padro para
   * propriedades multi-valoradas usado pelo JDK da Sun.
   * </p>
   * <ul>
   * <li>propriedade.1=ABCD</li>
   * <li>propriedade.2=EFGH</li>
   * </ul>
   * <ul>
   * <li>propriedade.nome.1=ABCD</li>
   * <li>propriedade.nome.2=EFGH</li>
   * </ul>
   *
   * @param name O nome da propriedade (nos exemplos acima, seriam "propriedade"
   *        e "propriedade.nome" respectivamente).
   * @param props propriedades da aplicao.
   *
   * @return Uma lista com todas as propriedades (caso no existam propriedades,
   *         a lista estar vazia).
   */
  private String[] getNumberedProperty(final String name, Properties props) {
    List<String> list = new LinkedList<String>();
    for (int i = 1; true; i++) {
      String propName = name + "." + i;
      String value = props.getProperty(propName);
      if (value == null) {
        break;
      }
      list.add(value);
      props.remove(propName);
    }
    return list.toArray(new String[list.size()]);
  }

  /**
   *
   * @param propName O nome da propriedade.
   * @param props propriedades da aplicao.
   *
   * @return Uma lista com todas as propriedades (caso no existam propriedades,
   *         a lista estar vazia).
   */
  private String getLocalizedProperty(final String propName, Properties props) {
    String language = locale.getLanguage();
    String country = locale.getCountry();
    String localizedPropName = propName + "." + language + "." + country;
    String value = props.getProperty(localizedPropName);
    props.remove(localizedPropName);
    return value;
  }

  /**
   * Carrega as propriedades especficas da aplicao.
   *
   * @param id o identificador da aplicao.
   * @param props propriedades especficas.
   */
  private void loadSpecificApplicationProperties(String id, Properties props) {
    Enumeration<Object> keys = props.keys();
    while (keys.hasMoreElements()) {
      Object keyElement = keys.nextElement();
      if (keyElement != null) {
        String key = (String) keyElement;
        Object propertyValue = props.get(key);
        if (propertyValue != null) {
          String value = (String) propertyValue;
          setApplicationProperty(id, key, value);
        }
      }
    }
  }

  /**
   * Carrega os cones da aplicao.
   *
   * @param id o identificador da aplicao.
   * @param appPath o caminho para a aplicao empacotada.
   */
  private void loadIconsFromPackage(String id, String appPath) {
    File path = new File(appPath);
    String iconFileName = id + ".32.gif";
    File iconFile = new File(path, iconFileName);
    if (iconFile.exists()) {
      setApplicationIcon(id, iconFile.getAbsolutePath());
    }
    String smallIconFileName = id + ".16.gif";
    File smallIconFile = new File(path, smallIconFileName);
    if (smallIconFile.exists()) {
      setSmallApplicationIcon(id, smallIconFile.getAbsolutePath());
    }
  }

  /**
   * Define a classe que servir de fbrica de contextos.
   *
   * @param factoryClassName o nome da classe.
   */
  private void setContextFactory(String factoryClassName) {
    if (factoryClassName != null) {
      String className = factoryClassName.trim();
      try {
        Class<?> factoryClass = Class.forName(className);
        if (IContextFactory.class.isAssignableFrom(factoryClass)) {
          IContextFactory contextFactory =
            (IContextFactory) factoryClass.newInstance();
          ApplicationManager appManager = ApplicationManager.getInstance();
          appManager.setContextFactory(contextFactory);
        }
        else {
          String message =
            "Context factory class {0} must implement the inteface {1}";
          String interfaceName = IContextFactory.class.getName();
          throw new RunnerException(message, className, interfaceName);
        }
      }
      catch (ClassNotFoundException e) {
        String message =
          "Context factory class {0} not found in Runner classpath.";
        throw new RunnerException(e, message, className);
      }
      catch (InstantiationException e) {
        String message = "Context factory class {0} could not be instantiated."
          + " The class must have an public empty constructor and not be abstract.";
        throw new RunnerException(e, message, className);
      }
      catch (IllegalAccessException e) {
        String message = "Context factory class {0} could not be instantiated."
          + " The class must have an public empty constructor and not be abstract.";
        throw new RunnerException(e, message, className);
      }
    }
  }

  /**
   * Atribui os tipos de arquivos associados  aplicao.
   *
   * @param id o identificador da aplicao.
   * @param fileTypesList a lista de tipos separados por vrgula.
   */
  private void setApplicationFileTypes(String id, String fileTypesList) {
    if (checkApplicationRegistry(id) && !isNull(fileTypesList)) {
      String appId = id.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      String fileTypes = fileTypesList.trim();
      if (fileTypes != null) {
        String[] types = fileTypes.split(",");
        for (String type : types) {
          FileType fileType = FileTypes.getFileType(type);
          if (fileType == null) {
            String message = "Unknown file type {0} for application {1}.";
            throw new RunnerException(message, type, appId);
          }
        }
        reg.setFileTypes(types);
      }
    }
  }

  /**
   * Faz a checagem de um tipo de arquivo.
   *
   * @param typeName o nome do tipo
   * @param extension a extenso associada.
   */
  private void checkFileTypeArgument(String typeName, String extension) {
    if (isNull(typeName) || isNull(extension)) {
      return;
    }

    String type = typeName.trim();
    String ext = extension.trim();

    FileType fileType;
    if (FileTypes.hasFileType(type)) {
      fileType = FileTypes.getFileType(type);
    }
    else {
      log("Creating new file type: {0}...", type);
      fileType = FileTypes.createFileType(type);
    }
    log("Setting extension \"{0}\" to type: {1}...", ext, type);
    fileType.addExtension(ext);

  }

  /**
   * Ajusta o charset do ambiente.
   *
   * @param charset o nome do charset a ser usado.
   */
  private void checkCharsetArgument(String charset) {
    if (isNull(charset)) {
      return;
    }

    String charsetName = charset.trim();
    Charset runnerCharset;
    try {
      runnerCharset = Charset.forName(charsetName);
    }
    catch (Exception e) {
      throw new RunnerException(e, "Invalid charset: {0}", charsetName);
    }
    log("Setting charset to {0}...", charsetName);
    this.charset = runnerCharset;
  }

  /**
   * Ajusta o nome do pas para efeito de locale.
   *
   * @param languageName o nome do idioma a ser usado.
   * @param countryName nome do pas
   */
  private void checkLocaleArgument(String languageName, String countryName) {
    if (isNull(languageName) || isNull(countryName)) {
      return;
    }

    String lng = languageName.trim();
    String cty = countryName.trim();
    log("Setting locale to {0} ({1})...", lng, cty);
    Runner.locale = new Locale(lng, cty);
  }

  /**
   * Finaliza o executor.
   *
   * @param correctly indicativo que o executor terminou corretamente.
   */
  private void logFinish(boolean correctly) {
    String state = correctly ? "correctly" : "with errors";
    logSevere("Runner terminated {0}.", state);
    if (!correctly) {
      throw new RuntimeException();
    }
  }

  /**
   * Consulta o id da aplicao
   *
   * @return o id.
   */
  private String getStartingApplicationId() {
    return startingId;
  }

  /**
   * Indica se os argumentos do runner indicam que o usurio pediu o <i>help</i>
   * (ajuda).
   *
   * @param args argumentos
   * @return indicativo
   */
  private boolean isOnlyHelp(String[] args) {
    for (String arg : args) {
      if (arg.equals("--help")) {
        return true;
      }
    }
    return false;
  }

  /**
   * Escreve uma mensagem de log de nvel normal.
   *
   * @param text mensagem.
   * @param args argumentos para formatao do texto.
   */
  private void log(String text, Object... args) {
    CSDKLogger logger = CSDKLogger.getInstance();
    logger.log(text, args);
  }

  /**
   * Escreve uma mensagem de log com o erro ocorrido.
   *
   * @param e o erro.
   */
  private void logException(Exception e) {
    CSDKLogger logger = CSDKLogger.getInstance();
    logger.logException(e);
  }

  /**
   * Escreve uma mensagem de log de nvel grave.
   *
   * @param text mensagem.
   * @param args argumentos para formatao do texto.
   */
  private void logSevere(String text, Object... args) {
    CSDKLogger logger = CSDKLogger.getInstance();
    logger.logSevere(text, args);
  }

  /**
   * Ajusta o classpath da aplicao. O classpath  informado como uma string de
   * caminhos para os jars e diretrios da aplicao separados por vrugula.
   *
   * @param id identificador da aplicao.
   * @param baseDir diretrio base da aplicao.
   * @param classpath o classpath da aplicao.
   */
  private void setApplicationClasspath(String id, File baseDir,
    String[] classpath) {
    if (checkApplicationRegistry(id) && classpath != null) {
      String appId = id.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      List<URL> urls = new ArrayList<URL>();
      for (String path : classpath) {
        try {
          String filePath = path.trim();
          log("Classpath: {0} for application id: {1}", filePath, appId);
          File element;
          if (FileUtils.isAbsolutePath(path)) {
            element = new File(filePath);
          }
          else {
            element = new File(baseDir, filePath);
          }
          if (!element.exists()) {
            throw new RunnerException("Library: {0} not found!", path);
          }
          URL url = element.toURI().toURL();
          urls.add(url);
        }
        catch (MalformedURLException e) {
          String msg =
            "Invalid element in classpath: {0} for application id: {1}";
          throw new RunnerException(e, msg, path, appId);
        }
      }
      reg.setClasspath(urls);
    }
  }

  /**
   * Ajusta o classpath da aplicao. O classpath  informado como uma string de
   * caminhos para os jars e diretrios da aplicao separados por vrugula.
   *
   * @param id identificador da aplicao.
   * @param baseDir diretrio base da aplicao.
   * @param classpath o classpath da aplicao.
   */
  private void setApplicationClasspath(String id, File baseDir,
    String classpath) {
    if (checkApplicationRegistry(id) && !isNull(classpath)) {
      String appId = id.trim();
      String cpath = classpath.trim();
      String[] paths = cpath.split(",");
      setApplicationClasspath(appId, baseDir, paths);
    }
  }

  /**
   * Verifica se existe uma aplicao com o identificador especificado.
   *
   * @param id identificador da aplicao.
   * @return <code>true</code> se existir aplicao com o identificador ou
   *         <code>false</code> caso contrrio.
   */
  private boolean checkApplicationRegistry(String id) {
    ApplicationManager appManager = ApplicationManager.getInstance();
    if (!appManager.hasApplicationRegistry(id)) {
      logSevere("Application id not registered: [{0}]", id);
      return false;
    }
    return true;
  }

  /**
   * Obtm o registro de uma aplicao com base em um identificador.
   *
   * @param id identificador da aplicao.
   * @return o registro.
   */
  private ApplicationRegistry getApplicationRegistry(String id) {
    ApplicationManager appManager = ApplicationManager.getInstance();
    return appManager.getApplicationRegistry(id);
  }

  /**
   * Ajusta o cone da aplicao.
   *
   * @param id identificador da aplicao.
   * @param applicationIconPath o caminho para o cone
   */
  private void setApplicationIcon(String id, String applicationIconPath) {
    if (checkApplicationRegistry(id) && !isNull(applicationIconPath)) {
      String appId = id.trim();
      String appIcon = applicationIconPath.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setApplicationIconPath(appIcon);
    }
  }

  /**
   * Ajusta a verso da aplicao.
   *
   * @param id identificador da aplicao.
   * @param version verso da aplicao.
   */
  private void setApplicationVersion(String id, String version) {
    if (checkApplicationRegistry(id) && !isNull(version)) {
      String appId = id.trim();
      String appVersion = version.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setVersion(appVersion);
    }
  }

  /**
   * Ajusta o nome da aplicao.
   *
   * @param id identificador da aplicao.
   * @param applicationName o nome.
   */
  private void setApplicationName(String id, String applicationName) {
    if (checkApplicationRegistry(id) && !isNull(applicationName)) {
      String appId = id.trim();
      String appName = applicationName.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setApplicationName(appName);
    }
  }

  /**
   * Verifica se uma string  nula ou vazia.
   *
   * @param value a string.
   * @return <code>true</code> se o valor  nulo ou equivalente a nulo e
   *         <code>false</code> caso contrrio.
   */
  private boolean isNull(String value) {
    return value == null || value.trim().equals("");
  }

  /**
   * Atribui a descrio da aplicao.
   *
   * @param id identificador da aplicao.
   * @param description a descrio.
   */
  private void setDescription(String id, String description) {
    if (checkApplicationRegistry(id) && !isNull(description)) {
      String appId = id.trim();
      String appDescription = description.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setApplicationDescription(appDescription);
    }
  }

  /**
   * Ajusta o nome do autor da aplicao.
   *
   * @param id identificador da aplicao.
   * @param name o nome.
   */
  private void setAuthorName(String id, String name) {
    if (checkApplicationRegistry(id) && !isNull(name)) {
      String appId = id.trim();
      String authorName = name.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setAuthor(authorName);
    }
  }

  /**
   * Ajusta o e-mail do autor da aplicao.
   *
   * @param id identificador da aplicao.
   * @param mail o e-mail.
   */
  private void setAuthorMail(String id, String mail) {
    if (checkApplicationRegistry(id) && !isNull(mail)) {
      String appId = id.trim();
      String authorMail = mail.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setAuthorEmail(authorMail);
    }
  }

  /**
   * Ajusta o indicativo de que o bundle da aplicao  para ser carregado.
   *
   * @param id id da aplicao.
   * @param bundleLoaded o indicativo de ajuste
   */
  private void setBundleLoaded(String id, boolean bundleLoaded) {
    if (checkApplicationRegistry(id)) {
      String appId = id.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setBundleRequired(bundleLoaded);
    }
  }

  /**
   * Ajusta o indicativo de que a aplicao s pode ter uma nica instncia.
   *
   * @param id id da aplicao.
   * @param singleton o indicativo de ajuste.
   */
  private void setSingleton(String id, boolean singleton) {
    if (checkApplicationRegistry(id)) {
      String appId = id.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setSingleton(singleton);
    }
  }

  /**
   * Ajusta o indicativo de que a aplicao precisa de projeto para ser
   * executada.
   *
   * @param id id da aplicao.
   * @param requireProject o indicativo de ajuste.
   */
  private void setRequireProject(String id, boolean requireProject) {
    if (checkApplicationRegistry(id)) {
      String appId = id.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setRequiresProject(requireProject);
    }
  }

  /**
   * Ajusta o nome da classe a ser executada.
   *
   * @param id identificador da aplicao.
   * @param className o nome a ser ajustado
   */
  private void setClassName(String id, String className) {
    if (checkApplicationRegistry(id) && !isNull(className)) {
      String appId = id.trim();
      String appClass = className.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setClassName(appClass);
    }
  }

  /**
   * Ajuste de propriedade da aplicao.
   *
   * @param id identificador da aplicao.
   * @param key chave da propriedade.
   * @param value valor da propriedade.
   */
  private void setApplicationProperty(String id, String key, String value) {
    if (checkApplicationRegistry(id)) {
      if (isNull(key)) {
        final String err = "Empty key not allowed for property adjustment.";
        throw new IllegalArgumentException(err);
      }
      if (isNull(value)) {
        final String err = "Empty value not allowed for property adjustment.";
        throw new IllegalArgumentException(err);
      }
      String appId = id.trim();
      String propKey = key.trim();
      if (!propKey.startsWith(appId + ".")) {
        String message =
          "Application property key must start with applicationId: {0}. Ignoring property {1}.";
        log(message, appId, propKey);
      }
      String propValue = value.trim();
      log("Setting application property: {0} = {1}", propKey, propValue);
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setProperty(propKey, propValue);
    }
  }

  /**
   * Ajuste de propriedade do runner.
   *
   * @param key chave da propriedade.
   * @param value valor da propriedade.
   */
  private void setRunnerProperty(String key, String value) {
    if (isNull(key)) {
      final String err = "Empty key not allowed for property adjustment.";
      throw new IllegalArgumentException(err);
    }
    if (isNull(value)) {
      final String err = "Empty value not allowed for property adjustment.";
      throw new IllegalArgumentException(err);
    }
    String propKey = key.trim();
    String propValue = value.trim();
    log("Setting runner property: {0} = {1}", propKey, propValue);
    runnerProperties.setProperty(propKey, propValue);
  }

  /**
   * Obtm as propriedades especficas do {@link Runner}.
   *
   * @return as propriedades.
   */
  private Properties getRunnerProperties() {
    return runnerProperties;
  }

  /**
   * Ajusta o cone reduzido da aplicao.
   *
   * @param id identificador da aplicao.
   * @param applicationIconPath o caminho para o cone
   */
  private void setSmallApplicationIcon(String id, String applicationIconPath) {
    if (checkApplicationRegistry(id) && !isNull(applicationIconPath)) {
      String appId = id.trim();
      String appIcon = applicationIconPath.trim();
      ApplicationRegistry reg = getApplicationRegistry(appId);
      reg.setSmallApplicationIconPath(appIcon);
    }
  }

  /**
   * Ajusta o id da aplicao
   *
   * @param id o applicationId a ser ajustado
   */
  private void setStartingApplicationId(String id) {
    if (checkApplicationRegistry(id)) {
      this.startingId = id.trim();
    }
  }

  /**
   * Obtm o locale do Runner.
   *
   * @return o locale.
   */
  public static Locale getLocale() {
    return locale;
  }

  /**
   * Faz dump de apresentao.
   */
  private void showHeader() {
    String lineSeparator = System.getProperty("line.separator");
    String bar = "=========================================================";
    StringBuilder builder = new StringBuilder();
    builder.append(lineSeparator);
    builder.append(bar);
    builder.append(lineSeparator);
    builder.append("CSDK/Runner - Executor for CSBASE Applications");
    builder.append(lineSeparator);
    builder.append("(C) 2013, Tecgraf/PUC-Rio");
    builder.append(lineSeparator);
    builder.append(bar);
    builder.append(lineSeparator);
    logSevere(builder.toString());
  }

  /**
   * Faz o dump do help.
   */
  private void showHelp() {
    System.out.println(" Runner Parameters:");
    System.out.println("   --verbose");
    System.out.println("   --locale <language> <country>");
    System.out.println("   --file-type <type> <extension>");
    System.out.println("   --file-types-property-file <path>");
    System.out.println("   --context-factory <class>");
    System.out.println("   --runner-property <key> <value>");
    System.out.println("   --start-id <id>");
    System.out.println("   --charset <charset>");
    System.out.println("   --help");
    System.out.println("");
    System.out.println(" Application Parameters:");
    System.out
      .println(" >> If application is already packaged in a directory:");
    System.out.println("   --app-package <id> <path>");
    System.out.println(" >> Or define specific properties:");
    System.out.println("   --app-id <id>");
    System.out.println("   --app-bundle <id>");
    System.out.println("   --app-classpath <id> <classpath>");
    System.out.println("   --app-name <id> <application name>");
    System.out.println("   --app-description <id> <application description>");
    System.out.println("   --app-author <id> <author name>");
    System.out.println("   --app-author-mail <id> <author mail>");
    System.out.println("   --app-class <id> <class name>");
    System.out.println("   --app-property <id> <key> <value>");
    System.out.println("   --app-version <id> <version name>");
    System.out.println("   --app-singleton <id>");
    System.out.println("   --app-require-project <id>");
    System.out.println("   --app-icon <id> <image path>");
    System.out.println("   --app-file-types <id> <comma-separated type list>");
    System.out.println("   --app-small-icon <id> <image path>");
  }

}
