package csdk.v1_0.runner;

import java.util.Date;
import java.util.logging.ConsoleHandler;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

/**
 * Logger usado pelo {@link Runner} do CSDK.
 */
class CSDKLogger {

  /**
   * Nvel padro do {@link Logger}.
   */
  private static final Level DEFAULT_LOGGER_LEVEL = Level.SEVERE;

  /**
   * Logger do CSDK.
   */
  private final Logger logger;

  /**
   * Instncia nico do {@link CSDKLogger}.
   */
  private static CSDKLogger instance;

  /**
   * Construtor.
   * 
   */
  private CSDKLogger() {
    this.logger = Logger.getLogger(Logger.GLOBAL_LOGGER_NAME);
    setVerbosed(false);
    Formatter formatter = new CSDKLoggerFormatter();
    ConsoleHandler handler = new ConsoleHandler();
    handler.setFormatter(formatter);
    this.logger.addHandler(handler);
    this.logger.setUseParentHandlers(false);
  }

  /**
   * Obtm a instncia nica de {@link CSDKLogger}.
   * 
   * @return a instncia de CSDKLogger.
   */
  public static CSDKLogger getInstance() {
    if (instance == null) {
      instance = new CSDKLogger();
    }
    return instance;
  }

  /**
   * Ajuste de indicao de verbose
   * 
   * @param verbosed o indicativo a ser ajustado
   */
  public void setVerbosed(boolean verbosed) {
    Level level;
    if (verbosed) {
      level = Level.ALL;
    }
    else {
      level = DEFAULT_LOGGER_LEVEL;
    }
    this.logger.setLevel(level);
  }

  /**
   * Escreve uma mensagem de log de nvel normal.
   * 
   * @param text mensagem.
   */
  public void log(String text) {
    logger.log(Level.INFO, text);
  }

  /**
   * Escreve uma mensagem de log de nvel grave.
   * 
   * @param text mensagem.
   */
  public void logSevere(String text) {
    logger.log(Level.SEVERE, text);
  }

  /**
   * Formatador de mensagens do log.
   */
  private final class CSDKLoggerFormatter extends Formatter {
    /**
     * Data da mensagem no log.
     */
    private final Date date = new Date();

    /**
     * {@inheritDoc}
     */
    @Override
    public String format(LogRecord record) {
      StringBuilder b = new StringBuilder();
      date.setTime(record.getMillis());
      b.append(date);
      b.append(" [CSDK] ");
      b.append(record.getMessage());
      b.append(System.getProperty("line.separator"));
      return b.toString();
    }
  }

}
