package csdk.v1_0.runner.filesystem.local;

import java.awt.Window;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;

import tecgraf.javautils.LNG;
import tecgraf.javautils.core.io.FileUtils;
import csdk.v1_0.api.filesystem.FileLocationType;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.local.ILocalFileSystemContext;
import csdk.v1_0.api.filesystem.local.LocalFileSystemException;
import csdk.v1_0.runner.Runner;
import csdk.v1_0.runner.core.AbstractContext;
import csdk.v1_0.runner.filesystem.FileType;
import csdk.v1_0.runner.filesystem.FileTypes;
import csdk.v1_0.runner.filesystem.RunnerFile;

/**
 * Contexto da aplicao com acesso ao sistema local de arquivos na sandbox do
 * CSDK.
 * 
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class LocalFileSystemContext extends AbstractContext implements
  ILocalFileSystemContext {

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getLocalFile(String[] path) throws LocalFileSystemException {
    return getFile(path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile createLocalFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException {
    File file = createFile(path, name, isDir);
    return new RunnerFile(file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile browseLocalFileInOpenMode(String[] fileTypes, boolean allowAll,
    IFile currentDir, FileSelectionType fileSelectionType, Window frame)
    throws LocalFileSystemException {
    int mode = JFileChooser.OPEN_DIALOG;
    List<IFile> selectedFiles =
      browse(mode, fileTypes, allowAll, null, currentDir, fileSelectionType,
        frame, false);
    if (selectedFiles == null || selectedFiles.isEmpty()) {
      return null;
    }
    return selectedFiles.iterator().next();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleLocalFilesInOpenMode(String[] fileTypes,
    boolean allowAll, IFile currentDir, FileSelectionType fileSelectionType,
    Window frame) throws LocalFileSystemException {
    int mode = JFileChooser.OPEN_DIALOG;
    List<IFile> selectedFiles =
      browse(mode, fileTypes, allowAll, null, currentDir, fileSelectionType,
        frame, true);
    if (selectedFiles == null || selectedFiles.isEmpty()) {
      return null;
    }
    return selectedFiles.toArray(new IFile[selectedFiles.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile browseLocalFileInSaveMode(String[] fileTypes, boolean allowAll,
    String defaultName, IFile currentDir, Window frame)
    throws LocalFileSystemException {
    int mode = JFileChooser.SAVE_DIALOG;
    List<IFile> selectedFiles =
      browse(mode, fileTypes, allowAll, defaultName, currentDir,
        FileSelectionType.FILES_ONLY, frame, false);
    if (selectedFiles == null || selectedFiles.isEmpty()) {
      return null;
    }
    return selectedFiles.iterator().next();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeLocalFile(String[] path) {
    String filePath = FileUtils.joinPath(path);
    File file = new File(filePath);
    return file.delete();
  }

  /**
   * Mtodo interno de busca de arquivos.
   * 
   * @param mode o modo do dilogo.
   * @param fileTypes os tipos de arquivos selecionveis
   * @param allowAll indicativo de incluso da opo "todos os arquivos".
   * @param defaultName nome padro do arquivo selecionado
   * @param currentDir diretrio inicial do seletor.
   * @param selectionType tipo de seleo do navegador (somente arquivos ou
   *        somente diretrios).
   * @param frame a janela-me do seletor de arquivos.
   * @param isMultipleSelection indicativo de seleo mltipla.
   * @return o arquivo selecionado ou <code>null</code>
   * @throws LocalFileSystemException em caso de erro.
   */
  private List<IFile> browse(int mode, String[] fileTypes, boolean allowAll,
    String defaultName, IFile currentDir, FileSelectionType selectionType,
    Window frame, boolean isMultipleSelection) throws LocalFileSystemException {

    JFileChooser fc = new JFileChooser() {
      @Override
      public void approveSelection() {
        File f = getSelectedFile();
        if (f.exists() && !f.isDirectory() && getDialogType() == SAVE_DIALOG) {
          String confirmMessage =
            LNG.get(LocalFileSystemContext.class.getSimpleName()
              + ".overwrite.message");
          int result =
            JOptionPane.showConfirmDialog(this, confirmMessage, null,
              JOptionPane.YES_NO_CANCEL_OPTION);
          switch (result) {
            case JOptionPane.YES_OPTION:
              super.approveSelection();
              return;
            case JOptionPane.NO_OPTION:
              return;
            case JOptionPane.CLOSED_OPTION:
              return;
            case JOptionPane.CANCEL_OPTION:
              cancelSelection();
              return;
          }
        }
        super.approveSelection();
      }
    };
    fc.setLocale(LNG.getLocale());
    fc.setDialogType(mode);
    int fileChooserType;
    if (selectionType == FileSelectionType.DIRECTORIES_ONLY) {
      fileChooserType = JFileChooser.DIRECTORIES_ONLY;
    }
    else {
      fileChooserType = JFileChooser.FILES_ONLY;
    }
    fc.setFileSelectionMode(fileChooserType);
    fc.setMultiSelectionEnabled(isMultipleSelection);
    if (currentDir != null) {
      if (currentDir.getFileLocationType() == FileLocationType.LOCAL) {
        File dir = new File(currentDir.getStringPath());
        fc.setCurrentDirectory(dir);
      }
      else {
        throw new LocalFileSystemException("Current directory "
          + currentDir.getStringPath() + "is not a local file");
      }
    }
    if (defaultName != null) {
      fc.setSelectedFile(new File(defaultName));
    }

    if (fileTypes != null && fileChooserType != JFileChooser.DIRECTORIES_ONLY) {
      for (String ft : fileTypes) {
        FileType tp = FileTypes.getFileType(ft);
        if (tp != null) {
          ArrayList<String> extensions = tp.getExtensions();
          FileNameExtensionFilter filter =
            new FileNameExtensionFilter(tp.getName(), extensions
              .toArray(new String[extensions.size()]));
          fc.addChoosableFileFilter(filter);
        }
      }
      fc.setAcceptAllFileFilterUsed(allowAll);
    }

    int returnVal = fc.showDialog(frame, null);
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      List<IFile> runnerFiles = new ArrayList<IFile>();
      if (isMultipleSelection) {
        File[] selectedFiles = fc.getSelectedFiles();
        for (File selectedFile : selectedFiles) {
          runnerFiles.add(new RunnerFile(selectedFile));
        }
      }
      else {
        File selectedFile = fc.getSelectedFile();
        runnerFiles.add(new RunnerFile(selectedFile));
      }
      return runnerFiles;
    }
    return null;
  }

  /**
   * Busca um arquivo pelo caminho.
   * 
   * @param path o caminho.
   * @return o arquivo.
   */
  private IFile getFile(String[] path) {
    String filePath = FileUtils.joinPath(path);
    File file = new File(filePath);
    if (file.exists()) {
      return new RunnerFile(file);
    }
    else {
      return null;
    }
  }

  /**
   * Cria um arquivo/diretrio.
   * 
   * @param path o caminho do diretrio pai do arquivo/diretrio.
   * @param name o nome do arquivo/diretrio
   * @param isDir determina se  arquivo ou diretrio
   * @return o arquivo.
   * @throws LocalFileSystemException em caso de erro ao criar o
   *         arquivo/diretrio.
   */
  private File createFile(String[] path, String name, boolean isDir)
    throws LocalFileSystemException {
    FileUtils.joinPath(path);
    String filePath = FileUtils.joinPath(path);
    File file = new File(filePath, name);
    if (isDir) {
      if (!file.mkdir()) {
        String error =
          LNG.get(LocalFileSystemContext.class.getSimpleName()
            + ".mkdir.error.message", new Object[] { file.getAbsolutePath() });
        throw new LocalFileSystemException(error);
      }
    }
    else {
      try {
        file.createNewFile();
      }
      catch (IOException e) {
        throw new LocalFileSystemException(e);
      }
    }
    return file;
  }

}
