package csdk.v1_0.runner;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.Set;

import javax.swing.SwingUtilities;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import csdk.v1_0.runner.application.RunnerApplication;
import csdk.v1_0.runner.filesystem.FileType;
import csdk.v1_0.runner.filesystem.FileTypes;

/**
 * Classe que representa o executor (<i>runner</i>) de simulaco do CSDK.
 * 
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class Runner {

  /**
   * Locale padro.
   */
  private static final Locale DEFAULT_LOCALE = new Locale("pt", "BR");

  /**
   * Identificador da aplicao
   */
  private String startingId;

  /**
   * Locale usado pelo Runner.
   */
  private Locale locale = DEFAULT_LOCALE;

  /**
   * Propriedades do Runner.
   */
  private final Properties runnerProperties;

  /**
   * Construtor
   */
  private Runner() {
    this.runnerProperties = new Properties();
  }

  /**
   * Entry point.
   * 
   * @param args argumentos da linha de comando.
   */
  static public void main(String[] args) {
    final Runner runner = new Runner();

    runner.showHeader();

    runner.log("Starting runner...");
    // Teste de exibio de help (apenas).
    if (runner.isOnlyHelp(args)) {
      runner.showHelp();
      runner.logFinish(true);
      return;
    }

    // Verificao de parmetros
    try {
      runner.checkArguments(args);
    }
    catch (RuntimeException e) {
      runner.logSevere("Command line parametrization error: " + e + "\n");
      runner.logSevere("Check arguments! Execution aborted!");
      StringWriter sw = new StringWriter();
      e.printStackTrace(new PrintWriter(sw));
      runner.logSevere(sw.toString());
      runner.logFinish(false);
      return;
    }

    runner.log("Loading runner locale bundle...");
    boolean localeOk = runner.loadBundle();
    if (!localeOk) {
      runner.logSevere("Locale bundle load failed!");
      runner.logFinish(false);
      return;
    }

    final ApplicationManager appManager = ApplicationManager.getInstance();
    Properties props = runner.getRunnerProperties();
    appManager.addContextProperties(props);

    String startId = runner.getStartingApplicationId();
    if (startId == null) {
      Set<String> appIds = appManager.getAllAplicationsIds();
      switch (appIds.size()) {
        case 0:
          runner.logSevere("No applications defined! Aborting...");
          runner.logFinish(false);
          return;
        case 1:
          startId = appIds.iterator().next();
          break;
        default:
          runner
            .logSevere("Multiple applications found and no starting application defined! Aborting...");
          runner.logFinish(false);
          return;
      }
    }
    final String appId = startId;
    ApplicationRegistry reg = runner.getApplicationRegistry(startId);
    if (reg == null) {
      runner.logSevere("Starting application not found! Aborting...");
      runner.logFinish(false);
      return;
    }

    runner.log("Starting main application...");
    runner.log("  * Application class name: " + reg.getClassName());
    runner.log("  * Application id: " + reg.getApplicationId());
    // Aviso de que nenhuma propriedade foi setada.
    if (reg.hasNoProperty()) {
      runner.log("  * No property set for main application.");
    }

    Runtime.getRuntime().addShutdownHook(new Thread() {
      @Override
      public void run() {
        appManager.finishAllApplications();
      }
    });

    // Incio da execuo
    runner.log("Starting main application...");
    try {
      SwingUtilities.invokeAndWait(new Runnable() {
        @Override
        public void run() {
          RunnerApplication startApp = appManager.runApplication(appId);
          if (startApp == null) {
            runner.logSevere("Main application execution failed!");
            runner.logFinish(false);
            return;
          }
          runner.logSevere("Main application started!");
        }
      });
    }
    catch (Exception e) {
      String msg = e.getMessage();
      if (msg != null) {
        runner.logSevere("Exception detected: " + msg);
      }
      Throwable cause = e.getCause();
      if (cause != null) {
        String err = "Cause of exception inside runner: ";
        runner.logSevere(err + cause.getMessage());
        cause.printStackTrace();
      }
    }
  }

  /**
   * Construo de um registry.
   * 
   * @param id identificador da aplicao.
   */
  private void buildNewApplicationRegistry(String id) {
    String appId = id.trim();
    ApplicationRegistry reg = new ApplicationRegistry(appId);
    ApplicationManager applicationManager = ApplicationManager.getInstance();
    applicationManager.addApplicationRegistry(reg);
  }

  /**
   * Chacagem de argumentos da linha de comando.
   * 
   * @param args os argumentos
   */
  private void checkArguments(String[] args) {
    int i = 0;
    int N = args.length;
    while (i < N) {
      String arg = args[i].trim();
      if (arg.equals("--locale")) {
        checkLocaleArgument(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--file-type")) {
        checkFileTypeArgument(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-property")) {
        setApplicationProperty(args[i + 1], args[i + 2], args[i + 3]);
        i = i + 3;
      }
      else if (arg.equals("--runner-property")) {
        setRunnerProperty(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-bundle")) {
        setBundleLoaded(args[i + 1], true);
        i = i + 1;
      }
      else if (arg.equals("--app-singleton")) {
        setSingleton(args[i + 1], true);
        i = i + 1;
      }
      else if (arg.equals("--app-require-project")) {
        setRequireProject(args[i + 1], true);
        i = i + 1;
      }
      else if (arg.equals("--verbose")) {
        setVerbosed(true);
      }
      else if (arg.equals("--start-id")) {
        setStartingApplicationId(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--context-factory")) {
        setContextFactory(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--app-id")) {
        buildNewApplicationRegistry(args[i + 1]);
        i = i + 1;
      }
      else if (arg.equals("--app-name")) {
        setApplicationName(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-description")) {
        setDescription(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-author")) {
        setAuthorName(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-author-mail")) {
        setAuthorMail(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-icon")) {
        setApplicationIcon(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-small-icon")) {
        setSmallApplicationIcon(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-package")) {
        loadApplicationFromPackage(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-version")) {
        setApplicationVersion(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-class")) {
        setClassName(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-file-types")) {
        setApplicationFileTypes(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else if (arg.equals("--app-classpath")) {
        setApplicationClasspath(args[i + 1], args[i + 2]);
        i = i + 2;
      }
      else {
        logSevere("Unrecognized argument found: " + arg + ". Ignoring...");
      }
      i = i + 1;
    }
  }

  /**
   * Ajuste de indicao de verbose
   * 
   * @param verbosed o indicativo a ser ajustado
   */
  private void setVerbosed(boolean verbosed) {
    CSDKLogger instance = CSDKLogger.getInstance();
    instance.setVerbosed(verbosed);
  }

  /**
   * Define o caminho para a aplicao empacotada.
   * 
   * @param id o identificador da aplicao.
   * @param appPath o caminho para o caminho da aplicao.
   */
  private void loadApplicationFromPackage(String id, String appPath) {
    String appId = id.trim();
    String packagePath = appPath.trim();
    loadPropertiesFromPackage(appId, packagePath);
    loadIconsFromPackage(appId, packagePath);
  }

  /**
   * Carrega as propriedades bsicas da aplicao.
   * 
   * @param id o identificador da aplicao.
   * @param appPath o caminho para a aplicao empacotada.
   */
  private void loadPropertiesFromPackage(String id, String appPath) {
    InputStreamReader reader = null;
    try {
      File path = new File(appPath);
      String propertyFileName = id + ".properties";
      File propertyFile = new File(path, propertyFileName);
      InputStream input = new FileInputStream(propertyFile);
      reader = new InputStreamReader(input, Charset.defaultCharset());
      Properties props = new Properties();
      props.load(reader);
      for (ApplicationParameters param : ApplicationParameters.values()) {
        String key = param.getApplicationPropertyName(id);
        switch (param) {
          case APP_NAME_PROPERTY:
            String localizedName = getLocalizedProperty(key, props);
            setApplicationName(id, localizedName);
            break;
          case CLASS_NAME_PROPERTY:
            setClassName(id, (String) props.get(key));
            break;
          case FILE_TYPES_PROPERTY:
            setApplicationFileTypes(id, (String) props.get(key));
            break;
          case VERSION_PROPERTY:
            setApplicationVersion(id, (String) props.get(key));
            break;
          case AUTHOR_NAME_PROPERTY:
            setAuthorName(id, (String) props.get(key));
            break;
          case AUTHOR_MAIL_PROPERTY:
            setAuthorMail(id, (String) props.get(key));
            break;
          case DESCRIPTION_PROPERTY:
            String localizedDescription = getLocalizedProperty(key, props);
            setDescription(id, localizedDescription);
            break;
          case NEED_BUNDLE_PROPERTY:
            String bundleValue = (String) props.get(key);
            if (bundleValue != null) {
              setBundleLoaded(id, Boolean.valueOf(bundleValue));
            }
            break;
          case IS_SINGLETON_PROPERTY:
            String singletonValue = (String) props.get(key);
            if (singletonValue != null) {
              setSingleton(id, Boolean.valueOf(singletonValue));
            }
            break;
          case REQUIRE_PROJECT_PROPERTY:
            String requireValue = (String) props.get(key);
            if (requireValue != null) {
              setRequireProject(id, Boolean.valueOf(requireValue));
            }
            break;
          case CLASSPATH_PROPERTY:
            String[] classpath = getNumberedProperty(key, props);
            setApplicationClasspath(id, path, classpath);
            break;
        }
        props.remove(key);
      }
      loadSpecificApplicationProperties(id, props);
    }
    catch (FileNotFoundException e) {
      logSevere("Application definition file (" + appPath
        + ") not found! Aborting...");
      logFinish(false);
    }
    catch (Exception e) {
      e.printStackTrace();
      logSevere("Error reading application definition file (" + appPath
        + "). Aborting...");
      logFinish(false);
    }
    finally {
      FileUtils.close(reader);
    }
  }

  /**
   * Obtm uma lista de valores de propriedades que seguem o padro para
   * propriedades multi-valoradas usado pelo JDK da Sun. </p>
   * <ul>
   * <li>propriedade.1=ABCD</li>
   * <li>propriedade.2=EFGH</li>
   * </ul>
   * <ul>
   * <li>propriedade.nome.1=ABCD</li>
   * <li>propriedade.nome.2=EFGH</li>
   * </ul>
   * 
   * @param name O nome da propriedade (nos exemplos acima, seriam "propriedade"
   *        e "propriedade.nome" respectivamente).
   * @param props propriedades da aplicao.
   * 
   * @return Uma lista com todas as propriedades (caso no existam propriedades,
   *         a lista estar vazia).
   */
  private String[] getNumberedProperty(final String name, Properties props) {
    List<String> list = new LinkedList<String>();
    for (int i = 1; true; i++) {
      String propName = name + "." + i;
      String value = props.getProperty(propName);
      if (value == null) {
        break;
      }
      list.add(value);
      props.remove(propName);
    }
    return list.toArray(new String[list.size()]);
  }

  /**
   * 
   * @param propName O nome da propriedade.
   * @param props propriedades da aplicao.
   * 
   * @return Uma lista com todas as propriedades (caso no existam propriedades,
   *         a lista estar vazia).
   */
  private String getLocalizedProperty(final String propName, Properties props) {
    String language = locale.getLanguage();
    String country = locale.getCountry();
    String localizedPropName = propName + "." + language + "." + country;
    String value = props.getProperty(localizedPropName);
    props.remove(localizedPropName);
    return value;
  }

  /**
   * Carrega as propriedades especficas da aplicao.
   * 
   * @param id o identificador da aplicao.
   * @param props propriedades especficas.
   */
  private void loadSpecificApplicationProperties(String id, Properties props) {
    Enumeration<Object> keys = props.keys();
    while (keys.hasMoreElements()) {
      Object keyElement = keys.nextElement();
      if (keyElement != null) {
        String key = (String) keyElement;
        Object propertyValue = props.get(key);
        if (propertyValue != null) {
          String value = (String) propertyValue;
          setApplicationProperty(id, key, value);
        }
      }
    }
  }

  /**
   * Carrega os cones da aplicao.
   * 
   * @param id o identificador da aplicao.
   * @param appPath o caminho para a aplicao empacotada.
   */
  private void loadIconsFromPackage(String id, String appPath) {
    File path = new File(appPath);
    String iconFileName = id + ".32.gif";
    File iconFile = new File(path, iconFileName);
    if (iconFile.exists()) {
      setApplicationIcon(id, iconFile.getAbsolutePath());
    }
    String smallIconFileName = id + ".16.gif";
    File smallIconFile = new File(path, smallIconFileName);
    if (smallIconFile.exists()) {
      setSmallApplicationIcon(id, smallIconFile.getAbsolutePath());
    }
  }

  /**
   * Define a classe que servir de fbrica de contextos.
   * 
   * @param factoryClassName o nome da classe.
   */
  private void setContextFactory(String factoryClassName) {
    String className = factoryClassName.trim();
    try {
      Class<?> factoryClass = Class.forName(className);
      if (IContextFactory.class.isAssignableFrom(factoryClass)) {
        IContextFactory contextFactory =
          (IContextFactory) factoryClass.newInstance();
        ApplicationManager appManager = ApplicationManager.getInstance();
        appManager.setContextFactory(contextFactory);
      }
      else {
        logSevere("Context factory class " + className
          + " must implement the inteface " + IContextFactory.class.getName());
        logFinish(false);
        return;
      }
    }
    catch (ClassNotFoundException e) {
      logSevere("Context factory class " + className
        + " not found in Runner classpath.");
      logFinish(false);
      return;
    }
    catch (InstantiationException e) {
      logSevere("Context factory class "
        + className
        + " could not be instantiated."
        + " The class must have an public empty constructor and not be abstract.");
      logFinish(false);
      return;
    }
    catch (IllegalAccessException e) {
      logSevere("Context factory class "
        + className
        + " could not be instantiated."
        + " The class must have an public empty constructor and not be abstract.");
      logFinish(false);
      return;
    }
  }

  /**
   * Atribui os tipos de arquivos associados  aplicao.
   * 
   * @param id o identificador da aplicao.
   * @param fileTypesList a lista de tipos separados por vrgula.
   */
  private void setApplicationFileTypes(String id, String fileTypesList) {
    String appId = id.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    String fileTypes = fileTypesList.trim();
    if (fileTypes != null) {
      String[] types = fileTypes.split(",");
      for (String type : types) {
        FileType fileType = FileTypes.getFileType(type.trim());
        if (fileType == null) {
          logSevere("Unknown file type " + type + " for application " + appId
            + ".");
          logFinish(false);
          return;
        }
      }
      reg.setFileTypes(types);
    }
  }

  /**
   * Faz a checagem de um tipo de arquivo.
   * 
   * @param typeName o nome do tipo
   * @param extension a extenso associada.
   */
  private void checkFileTypeArgument(String typeName, String extension) {
    String type = typeName.trim();
    String ext = extension.trim();
    if (isEmptyString(type) || isEmptyString(ext)) {
      return;
    }

    FileType fileType;
    if (FileTypes.hasFileType(type)) {
      fileType = FileTypes.getFileType(type);
    }
    else {
      String msg = "Creating new file type: " + type + "...";
      log(msg);
      fileType = FileTypes.createFileType(type);
    }
    log("Setting extension \"" + ext + "\" to type: " + type + "...");
    fileType.addExtension(ext);

  }

  /**
   * Ajusta o nome do pas para efeito de locale.
   * 
   * @param languageName o nome do idioma a ser usado.
   * @param countryName nome do pas
   */
  private void checkLocaleArgument(String languageName, String countryName) {
    String lng = languageName.trim();
    String cty = countryName.trim();
    if (isEmptyString(lng) || isEmptyString(cty)) {
      return;
    }
    log("Setting locale to " + lng + " (" + cty + ")...");
    this.locale = new Locale(lng, cty);
  }

  /**
   * Finaliza o executor.
   * 
   * @param correctly indicativo que o executor terminou corretamente.
   */
  private void logFinish(boolean correctly) {
    String state = correctly ? "correctly" : "with errors";
    String text = String.format("Runner terminated %s!", state);
    logSevere(text);
    if (!correctly) {
      throw new RuntimeException();
    }
  }

  /**
   * Consulta o id da aplicao
   * 
   * @return o id.
   */
  private String getStartingApplicationId() {
    return startingId;
  }

  /**
   * Teste se uma string  nula.
   * 
   * @param str a string a ser testada
   * @return um indicativo booleano
   */
  private boolean isEmptyString(String str) {
    if (str == null) {
      return true;
    }
    String s = str.trim();
    if (s.equals("")) {
      return true;
    }
    return false;
  }

  /**
   * Indica se os argumentos do runner indicam que o usurio pediu o <i>help</i>
   * (ajuda).
   * 
   * @param args argumentos
   * @return indicativo
   */
  private boolean isOnlyHelp(String[] args) {
    for (String arg : args) {
      if (arg.equals("--help")) {
        return true;
      }
    }
    return false;
  }

  /**
   * Lana a aplicao.
   * 
   * @return indicativo de sucesso.
   */
  private boolean loadBundle() {
    try {
      String className = Runner.class.getSimpleName();
      String classFullName = Runner.class.getName();
      String packageName =
        classFullName.replaceAll(className, "resources.idiom");
      LNG.load(packageName, locale);
      return true;
    }
    catch (Exception e) {
      String msg = e.getMessage();
      if (msg != null) {
        logSevere("Exception detected: " + msg);
      }
      Throwable cause = e.getCause();
      if (cause != null) {
        String err = "Cause of exception inside application class: ";
        logSevere(err + cause.getMessage());
        cause.printStackTrace();
      }
      return false;
    }
  }

  /**
   * Escreve uma mensagem de log de nvel normal.
   * 
   * @param text mensagem.
   */
  private void log(String text) {
    CSDKLogger logger = CSDKLogger.getInstance();
    logger.log(text);
  }

  /**
   * Escreve uma mensagem de log de nvel grave.
   * 
   * @param text mensagem.
   */
  private void logSevere(String text) {
    CSDKLogger logger = CSDKLogger.getInstance();
    logger.logSevere(text);
  }

  /**
   * Ajusta o classpath da aplicao. O classpath  informado como uma string de
   * caminhos para os jars e diretrios da aplicao separados por vrugula.
   * 
   * @param id identificador da aplicao.
   * @param baseDir diretrio base da aplicao.
   * @param classpath o classpath da aplicao.
   */
  private void setApplicationClasspath(String id, File baseDir,
    String[] classpath) {
    ApplicationRegistry reg = getApplicationRegistry(id);
    List<URL> urls = new ArrayList<URL>();
    for (String path : classpath) {
      try {
        logSevere("Classpath: " + path + " for application id: " + id);
        File element;
        String filePath = path.trim();
        if (FileUtils.isAbsolutePath(path)) {
          element = new File(filePath);
        }
        else {
          element = new File(baseDir, filePath);
        }
        if (!element.exists()) {
          logSevere("Library: " + path + " not found!");
          logFinish(false);
        }
        URL url = element.toURI().toURL();
        urls.add(url);
      }
      catch (MalformedURLException e) {
        logSevere("Invalid element in classpath: " + path
          + " for application id: " + id);
        logSevere("Runner aborted!");
        logFinish(false);
        return;
      }
    }
    reg.setClasspath(urls);
  }

  /**
   * Ajusta o classpath da aplicao. O classpath  informado como uma string de
   * caminhos para os jars e diretrios da aplicao separados por vrugula.
   * 
   * @param id identificador da aplicao.
   * @param classpath o classpath da aplicao.
   */
  private void setApplicationClasspath(String id, String classpath) {
    String appId = id.trim();
    String cpath = classpath.trim();
    String[] paths = cpath.split(",");
    File currentDir = new File(".");
    setApplicationClasspath(appId, currentDir, paths);
  }

  /**
   * Obtm o registro de uma aplicao com base em um identificador.
   * 
   * @param id identificador da aplicao.
   * @return o registro.
   */
  private ApplicationRegistry getApplicationRegistry(String id) {
    ApplicationManager appManager = ApplicationManager.getInstance();
    return appManager.getApplicationRegistry(id);
  }

  /**
   * Ajusta o cone da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param applicationIconPath o caminho para o cone
   */
  private void setApplicationIcon(String id, String applicationIconPath) {
    String appId = id.trim();
    String appIcon = applicationIconPath.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setApplicationIconPath(appIcon);
  }

  /**
   * Ajusta a verso da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param version verso da aplicao.
   */
  private void setApplicationVersion(String id, String version) {
    String appId = id.trim();
    String appVersion = version.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setVersion(appVersion);
  }

  /**
   * Ajusta o nome da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param applicationName o nome.
   */
  private void setApplicationName(String id, String applicationName) {
    String appId = id.trim();
    String appName = applicationName.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setApplicationName(appName);
  }

  /**
   * Atribui a descrio da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param description a descrio.
   */
  private void setDescription(String id, String description) {
    String appId = id.trim();
    String appDescription = description.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setApplicationDescription(appDescription);
  }

  /**
   * Ajusta o nome do autor da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param name o nome.
   */
  private void setAuthorName(String id, String name) {
    String appId = id.trim();
    String authorName = name.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setAuthor(authorName);
  }

  /**
   * Ajusta o e-mail do autor da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param mail o e-mail.
   */
  private void setAuthorMail(String id, String mail) {
    String appId = id.trim();
    String authorMail = mail.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setAuthorEmail(authorMail);
  }

  /**
   * Ajusta o indicativo de que o bundle da aplicao  para ser carregado.
   * 
   * @param id id da aplicao.
   * @param bundleLoaded o indicativo de ajuste
   */
  private void setBundleLoaded(String id, boolean bundleLoaded) {
    String appId = id.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setBundleRequired(bundleLoaded);
  }

  /**
   * Ajusta o indicativo de que a aplicao s pode ter uma nica instncia.
   * 
   * @param id id da aplicao.
   * @param singleton o indicativo de ajuste.
   */
  private void setSingleton(String id, boolean singleton) {
    String appId = id.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setSingleton(singleton);
  }

  /**
   * Ajusta o indicativo de que a aplicao precisa de projeto para ser
   * executada.
   * 
   * @param id id da aplicao.
   * @param requireProject o indicativo de ajuste.
   */
  private void setRequireProject(String id, boolean requireProject) {
    String appId = id.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setRequiresProject(requireProject);
  }

  /**
   * Ajusta o nome da classe a ser executada.
   * 
   * @param id identificador da aplicao.
   * @param className o nome a ser ajustado
   */
  private void setClassName(String id, String className) {
    String appId = id.trim();
    String appClass = className.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    if (reg == null) {
      logSevere("Class name set to not registered application id: [" + appId
        + "]");
      logSevere("Runner aborted!");
      logFinish(false);
      return;
    }
    reg.setClassName(appClass);
  }

  /**
   * Ajuste de propriedade da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param key chave da propriedade.
   * @param value valor da propriedade.
   */
  private void setApplicationProperty(String id, String key, String value) {
    String appId = id.trim();
    String propKey = key.trim();
    String propValue = value.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    if (reg == null) {
      logSevere("Property set to not registered application id: [" + appId
        + "]. Ignoring...");
      return;
    }
    if (isEmptyString(propKey)) {
      final String err = "Empty key not allowed for property adjustment.";
      throw new IllegalArgumentException(err);
    }
    if (isEmptyString(propValue)) {
      final String err = "Empty value not allowed for property adjustment.";
      throw new IllegalArgumentException(err);
    }
    log("Setting application property: " + propKey + " = " + propValue);
    reg.setProperty(propKey, propValue);
  }

  /**
   * Ajuste de propriedade do runner.
   * 
   * @param key chave da propriedade.
   * @param value valor da propriedade.
   */
  private void setRunnerProperty(String key, String value) {
    String propKey = key.trim();
    String propValue = value.trim();
    if (isEmptyString(propKey)) {
      final String err = "Empty key not allowed for property adjustment.";
      throw new IllegalArgumentException(err);
    }
    if (isEmptyString(propValue)) {
      final String err = "Empty value not allowed for property adjustment.";
      throw new IllegalArgumentException(err);
    }
    log("Setting runner property: " + propKey + " = " + propValue);
    runnerProperties.setProperty(propKey, propValue);
  }

  /**
   * Obtm as propriedades especficas do {@link Runner}.
   * 
   * @return as propriedades.
   */
  private Properties getRunnerProperties() {
    return runnerProperties;
  }

  /**
   * Ajusta o cone reduzido da aplicao.
   * 
   * @param id identificador da aplicao.
   * @param applicationIconPath o caminho para o cone
   */
  private void setSmallApplicationIcon(String id, String applicationIconPath) {
    String appId = id.trim();
    String appIcon = applicationIconPath.trim();
    ApplicationRegistry reg = getApplicationRegistry(appId);
    reg.setSmallApplicationIconPath(appIcon);
  }

  /**
   * Ajusta o id da aplicao
   * 
   * @param applicationId o applicationId a ser ajustado
   */
  private void setStartingApplicationId(String applicationId) {
    this.startingId = applicationId.trim();
  }

  /**
   * Faz dump de apresentao.
   */
  private void showHeader() {
    String lineSeparator = System.getProperty("line.separator");
    String bar = "=========================================================";
    StringBuilder builder = new StringBuilder();
    builder.append(lineSeparator);
    builder.append(bar);
    builder.append(lineSeparator);
    builder.append("CSDK/Runner - Executor for CSBASE Applications");
    builder.append(lineSeparator);
    builder.append("(C) 2013, Tecgraf/PUC-Rio");
    builder.append(lineSeparator);
    builder.append(bar);
    builder.append(lineSeparator);
    logSevere(builder.toString());
  }

  /**
   * Faz o dump do help.
   */
  private void showHelp() {
    System.out.println(" Runner Parameters:");
    System.out.println("   --verbose");
    System.out.println("   --locale <language> <country>");
    System.out.println("   --file-type <type> <extension>");
    System.out.println("   --context-factory <class>");
    System.out.println("   --runner-property <key> <value>");
    System.out.println("   --start-id <id>");
    System.out.println("   --help");
    System.out.println("");
    System.out.println(" Application Parameters:");
    System.out
      .println(" >> If application is already packaged in a directory:");
    System.out.println("   --app-package <id> <path>");
    System.out.println(" >> Or define specific properties:");
    System.out.println("   --app-id <id>");
    System.out.println("   --app-bundle <id>");
    System.out.println("   --app-classpath <id> <classpath>");
    System.out.println("   --app-name <id> <application name>");
    System.out.println("   --app-description <id> <application description>");
    System.out.println("   --app-author <id> <author name>");
    System.out.println("   --app-author-mail <id> <author mail>");
    System.out.println("   --app-class <id> <class name>");
    System.out.println("   --app-property <id> <key> <value>");
    System.out.println("   --app-version <id> <version name>");
    System.out.println("   --app-singleton <id>");
    System.out.println("   --app-require-project <id>");
    System.out.println("   --app-icon <id> <image path>");
    System.out.println("   --app-file-types <id> <comma-separated type list>");
    System.out.println("   --app-small-icon <id> <image path>");
  }

}
